// map.js
let map, marker;

// Config
const latInput = document.getElementById('latitude');
const lngInput = document.getElementById('longitude');
const apiKey = document.querySelector('meta[name="google-maps-key"]')?.content || '';
const mapId = document.querySelector('meta[name="google-maps-id"]')?.content || null;
const customPinInput = document.getElementById('customPinUrl');
const customPinUrl = customPinInput?.value || window.baseUrl + '/assets/img/order.png';
const baseMarkerSize = 40;
const addressInput = document.getElementById('address');

// Load Google Maps API
function loadGoogleMapsApi(key) {
  return new Promise((resolve, reject) => {
    if (window.google && window.google.maps && window.google.maps.marker) return resolve();

    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${key}&libraries=marker,places,geocoding,core&v=weekly`;
    script.async = true;
    script.defer = true;
    script.onload = () => resolve();
    script.onerror = () => reject(new Error('Failed to load Google Maps API'));
    document.head.appendChild(script);
  });
}

// Initialize map
async function initMap() {
  await loadGoogleMapsApi(apiKey);

  let defaultLat = parseFloat(latInput?.value) || null;
  let defaultLng = parseFloat(lngInput?.value) || null;

  // If no lat/lng, get browser geolocation
  if (defaultLat === null || defaultLng === null) {
    try {
      const position = await new Promise((resolve, reject) => {
        navigator.geolocation.getCurrentPosition(resolve, reject, { timeout: 10000 });
      });
      defaultLat = position.coords.latitude;
      defaultLng = position.coords.longitude;
    } catch (err) {
      console.warn('Geolocation failed, using fallback coordinates');
      defaultLat = 31.3501; // fallback
      defaultLng = 75.58795;
    }
  }

  const defaultPosition = { lat: defaultLat, lng: defaultLng };

  map = new google.maps.Map(document.getElementById('map'), {
    center: defaultPosition,
    zoom: 14,
    mapId: mapId || undefined
  });

  //geocoder = new google.maps.Geocoder();
  // Marker content
  const markerDiv = document.createElement('div');
  markerDiv.style.cursor = 'grab';

  const img = document.createElement('img');
  img.src = customPinUrl;
  img.style.width = `${baseMarkerSize}px`;
  img.style.height = `${baseMarkerSize}px`;
  img.style.userSelect = 'none';
  img.style.pointerEvents = 'none';
  markerDiv.appendChild(img);

  // AdvancedMarkerElement
  marker = new google.maps.marker.AdvancedMarkerElement({
    map,
    position: defaultPosition,
    gmpDraggable: true,
    content: markerDiv
  });

  marker.addListener('dragend', event => {
    const position = marker.position;
    infoWindow.close();
    // infoWindow.setContent(`Pin dropped at: ${position.lat}, ${position.lng}`);
    //infoWindow.open(marker.map, marker);
    infoWindow.setContent(`
        <b>Marker Position</b><br>
        Latitude: ${position.lat.toFixed(6)}<br>
        Longitude: ${position.lng.toFixed(6)}
      `);

    infoWindow.open({ anchor: marker, map });
    if (position) {
      latInput.value = position.lat.toFixed(6);
      lngInput.value = position.lng.toFixed(6);
    }
  });

  // Click → show coordinates
  const infoWindow = new google.maps.InfoWindow();
  marker.addListener('click', () => {
    const pos = marker.position;
    if (pos) {
      infoWindow.setContent(`
        <div style="min-width:180px; padding:5px;">
          <b>Marker Position</b><br>
          Latitude: ${pos.lat.toFixed(6)}<br>
          Longitude: ${pos.lng.toFixed(6)}
        </div>
      `);
      infoWindow.open({ anchor: marker, map });
    }
  });

  // Manual input → move marker
  function updateMarkerFromInputs() {
    const lat = parseFloat(latInput.value);
    const lng = parseFloat(lngInput.value);
    if (!isNaN(lat) && !isNaN(lng)) {
      const newPos = new google.maps.LatLng(lat, lng);
      marker.position = newPos;
      map.panTo(newPos);
    }
  }

  latInput.addEventListener('change', updateMarkerFromInputs);
  lngInput.addEventListener('change', updateMarkerFromInputs);

  // --- AUTOCOMPLETE (ADDED) ---
  // Uses google.maps.places.Autocomplete (stable). On place select:
  //  - move marker, pan map, set lat/lng
  //  - change marker image to place icon if available, else use customPinUrl
  if (addressInput && google.maps.places) {
    try {
      const autocomplete = new google.maps.places.Autocomplete(addressInput, {
        fields: ['geometry', 'formatted_address', 'icon', 'name'],
        types: ['geocode'] // or ['address'] if you prefer
      });

      autocomplete.addListener('place_changed', () => {
        const place = autocomplete.getPlace();
        if (!place || !place.geometry || !place.geometry.location) return;

        const loc = place.geometry.location;
        const lat = loc.lat();
        const lng = loc.lng();

        // Update marker icon if place provides an icon (some Places do)
        // place.icon can be either a URL string or an object — handle both.
        try {
          let newIcon = customPinUrl;
          if (place.icon) {
            // If icon is an object with url field (older responses), prefer url
            if (typeof place.icon === 'object' && place.icon.url) newIcon = place.icon.url;
            else if (typeof place.icon === 'string' && place.icon.length) newIcon = place.icon;
          }
          // set new image source (keeps markerDiv/img structure)
          img.src = newIcon;
          // optionally adjust size for remote icons (keeps baseMarkerSize)
          img.style.width = `${baseMarkerSize}px`;
          img.style.height = `${baseMarkerSize}px`;
        } catch (e) {
          // ignore icon errors and keep default
          console.warn('Failed to update marker icon from place.icon', e);
        }

        // Update map + marker
        const latLng = new google.maps.LatLng(lat, lng);
        marker.position = latLng;
        map.panTo(latLng);
        map.setZoom(15);

        // Update hidden inputs
        latInput.value = lat.toFixed(6);
        lngInput.value = lng.toFixed(6);

        // Put formatted address back into input, if available
        if (place.formatted_address) {
          addressInput.value = place.formatted_address;
        }
      });
    } catch (err) {
      console.warn('Autocomplete setup failed, falling back to no-autocomplete:', err);
    }
  }
  // --- END AUTOCOMPLETE ---

  // Set initial lat/lng
  latInput.value = defaultLat;
  lngInput.value = defaultLng;
}

// Auto-init
document.addEventListener('DOMContentLoaded', () => {
  if (document.getElementById('map') && apiKey) {
    initMap();
  }
});
