'use strict';

(function () {
  const cardColor = config.colors.cardColor;
  const labelColor = config.colors.textMuted;
  const headingColor = config.colors.headingColor;
  const fontFamily = 'inherit';
  const daysOfWeek = ['Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa', 'Su'];

  // ✅ Helper: Normalize/ensure non-empty array
  const safeArray = (arr, length = 7) => {
    if (!Array.isArray(arr) || arr.length === 0) return Array(length).fill(0);
    return arr.map(v => (v == null ? 0 : Number(v)));
  };

  // ✅ Helper: Check if array actually has data (non-zero)
  const hasRealData = arr => Array.isArray(arr) && arr.some(v => v !== 0);

  // ✅ Helper: Show "No data available" message in chart container
  function renderNoData(el, message = 'No data available') {
    el.innerHTML = `<div class="text-center text-muted py-4">${message}</div>`;
  }

  // ---------------------------------------------------------------------
  // 📈 1. Average Daily Sales (Area Chart)
  // ---------------------------------------------------------------------
  fetch(`${baseURL}/dashboard/average-daily-sales`)
    .then(res => res.json())
    .then(response => {
      const salesData = safeArray(response.data, 7);
      const averageDailySalesEl = document.querySelector('#averageDailySales');
      if (!averageDailySalesEl) return;

      if (!hasRealData(salesData)) {
        renderNoData(averageDailySalesEl.parentNode);
        return;
      }

      // Total display (above chart)
      document.querySelector('#averageDailySalesTotal').textContent =
        `${window.totalEarningChartData.currency}${response.total?.toLocaleString() || 0}`;

      const chartConfig = {
        chart: { height: 105, type: 'area', toolbar: { show: false }, sparkline: { enabled: true } },
        markers: { colors: 'transparent', strokeColors: 'transparent' },
        grid: { show: false },
        colors: [config.colors.success],
        fill: {
          type: 'gradient',
          gradient: {
            shadeIntensity: 1,
            opacityFrom: 0.4,
            gradientToColors: [cardColor],
            opacityTo: 0.1,
            stops: [0, 100]
          }
        },
        dataLabels: { enabled: false },
        stroke: { width: 2, curve: 'smooth' },
        series: [{ data: salesData }],
        xaxis: {
          categories: daysOfWeek, // ✅ force show Mo…Su
          axisBorder: { show: false },
          axisTicks: { show: false },
          labels: { style: { colors: labelColor, fontSize: '13px', fontFamily } }
        },
        yaxis: { show: false },
        tooltip: { enabled: false }
      };

      new ApexCharts(averageDailySalesEl, chartConfig).render();
    })
    .catch(() => {
      const el = document.querySelector('#averageDailySales');
      if (el) renderNoData(el.parentNode);
    });

  // ---------------------------------------------------------------------
  // 📊 2. Weekly Earnings - Bar Chart
  // ---------------------------------------------------------------------
  const weeklyRaw = document.getElementById('weeklyChart')?.dataset.weekly;
  const weeklyData = safeArray(weeklyRaw ? JSON.parse(weeklyRaw) : [], 7);

  const weeklyEarningReportsEl = document.querySelector('#weeklyEarningReports');
  if (weeklyEarningReportsEl) {
    if (!hasRealData(weeklyData)) {
      renderNoData(weeklyEarningReportsEl);
    } else {
      const weeklyConfig = {
        chart: { height: 161, type: 'bar', toolbar: { show: false }, parentHeightOffset: 0 },
        plotOptions: {
          bar: {
            barHeight: '60%',
            columnWidth: '38%',
            startingShape: 'rounded',
            endingShape: 'rounded',
            borderRadius: 4,
            distributed: true
          }
        },
        grid: { show: false, padding: { top: -30, bottom: 0, left: -10, right: -10 } },
        colors: [
          config.colors_label.primary,
          config.colors_label.primary,
          config.colors_label.primary,
          config.colors_label.primary,
          config.colors.primary,
          config.colors_label.primary,
          config.colors_label.primary
        ],
        dataLabels: { enabled: false },
        series: [{ data: weeklyData }],
        legend: { show: false },
        xaxis: {
          categories: daysOfWeek, // ✅ force show Mo…Su
          axisBorder: { show: false },
          axisTicks: { show: false },
          labels: { style: { colors: labelColor, fontSize: '13px', fontFamily } }
        },
        yaxis: { labels: { show: false } },
        tooltip: { enabled: false }
      };
      new ApexCharts(weeklyEarningReportsEl, weeklyConfig).render();
    }
  }

  // ---------------------------------------------------------------------
  // 🧾 3. Total Earnings vs Expenses - Stacked Bar Chart
  // ---------------------------------------------------------------------
  const totalEarningChartEl = document.querySelector('#totalEarningChart');
  const { earnings = [], expenses = [] } = window.totalEarningChartData || {};

  const earningsData = safeArray(earnings, 7);
  const expensesData = safeArray(expenses, 7);

  if (totalEarningChartEl) {
    if (!hasRealData(earningsData) && !hasRealData(expensesData)) {
      renderNoData(totalEarningChartEl);
    } else {
      const earningsWithLabels = daysOfWeek.map((day, i) => ({
        x: day,
        y: earningsData[i] || 0
      }));
      const expensesWithLabels = daysOfWeek.map((day, i) => ({
        x: day,
        y: expensesData[i] || 0
      }));

      const totalEarningChartOptions = {
        chart: { height: 175, stacked: true, type: 'bar', toolbar: { show: false }, parentHeightOffset: 0 },
        series: [
          { name: 'Earning', data: earningsWithLabels },
          { name: 'Expense', data: expensesWithLabels }
        ],
        plotOptions: {
          bar: {
            horizontal: false,
            columnWidth: '40%',
            borderRadius: 7,
            startingShape: 'rounded',
            endingShape: 'rounded',
            borderRadiusApplication: 'around',
            borderRadiusWhenStacked: 'last'
          }
        },
        colors: [config.colors.primary, config.colors.secondary],
        dataLabels: { enabled: false },
        stroke: { curve: 'smooth', width: 5, lineCap: 'round', colors: [cardColor] },
        legend: { show: false },
        fill: { opacity: 1 },
        grid: { show: false, padding: { top: -40, bottom: -40, left: -10, right: -2 } },
        xaxis: {
          categories: daysOfWeek, // ✅ force show Mo…Su
          axisBorder: { show: false },
          axisTicks: { show: false },
          labels: { style: { colors: labelColor, fontSize: '13px', fontFamily } }
        },
        yaxis: { labels: { show: false } }
      };

      new ApexCharts(totalEarningChartEl, totalEarningChartOptions).render();
    }
  }
})();
