/**
 * App Chat
 */
'use strict';

document.addEventListener('DOMContentLoaded', () => {
  // DOM Elements
  const elements = {
    chatContactsBody: document.querySelector('.app-chat-contacts .sidebar-body'),
    chatHistoryBody: document.querySelector('.chat-history-body'),
    chatSidebarLeftBody: document.querySelector('.app-chat-sidebar-left .sidebar-body'),
    chatSidebarRightBody: document.querySelector('.app-chat-sidebar-right .sidebar-body'),
    chatUserStatus: [...document.querySelectorAll(".form-check-input[name='chat-user-status']")],
    chatSidebarLeftUserAbout: document.getElementById('chat-sidebar-left-user-about'),
    formSendMessage: document.querySelector('.form-send-message'),
    messageInput: document.querySelector('.message-input'),
    searchInput: document.querySelector('.chat-search-input'),
    chatContactListItems: [...document.querySelectorAll('.chat-contact-list-item:not(.chat-contact-list-item-title)')],
    textareaInfo: document.getElementById('textarea-maxlength-info'),
    conversationButton: document.getElementById('app-chat-conversation-btn'),
    chatHistoryHeader: document.querySelector(".chat-history-header [data-target='#app-chat-contacts']"),
    speechToText: $('.speech-to-text'),
    appChatConversation: document.getElementById('app-chat-conversation'),
    appChatHistory: document.getElementById('app-chat-history'),
    replyPreviewContainer: document.querySelector('.reply-preview-container'),
    replyPreviewText: document.querySelector('.reply-preview-container .reply-text'),
    fileInput: document.querySelector('#attach-doc'),
    sendBtn: document.querySelector('.send-msg-btn')
  };
  const userStatusClasses = {
    active: 'avatar-online',
    offline: 'avatar-offline',
    away: 'avatar-away',
    busy: 'avatar-busy'
  };

  /**
   * Initialize PerfectScrollbar on provided elements.
   * @param {HTMLElement[]} elements - List of elements to initialize.
   */
  const initPerfectScrollbar = elements => {
    elements.forEach(el => {
      if (el) {
        new PerfectScrollbar(el, {
          wheelPropagation: false,
          suppressScrollX: true
        });
      }
    });
  };

  /**
   * Scroll chat history to the bottom.
   */
  const scrollToBottom = () => elements.chatHistoryBody?.scrollTo(0, elements.chatHistoryBody.scrollHeight);

  /**
   * Update user status avatar classes.
   * @param {string} status - Status key from userStatusClasses.
   */
  const updateUserStatus = status => {
    const leftSidebarAvatar = document.querySelector('.chat-sidebar-left-user .avatar');
    const contactsAvatar = document.querySelector('.app-chat-contacts .avatar');

    [leftSidebarAvatar, contactsAvatar].forEach(avatar => {
      if (avatar) avatar.className = avatar.className.replace(/avatar-\w+/, userStatusClasses[status]);
    });
  };

  // Handle textarea max length count.
  function handleMaxLengthCount(inputElement, infoElement, maxLength) {
    const currentLength = inputElement.value.length;
    const remaining = maxLength - currentLength;

    infoElement.className = 'maxLength label-success';

    if (remaining >= 0) {
      infoElement.textContent = `${currentLength}/${maxLength}`;
    }
    if (remaining <= 0) {
      infoElement.textContent = `${currentLength}/${maxLength}`;
      infoElement.classList.remove('label-success');
      infoElement.classList.add('label-danger');
    }
  }

  /**
   * Switch to chat conversation view.
   */
  const switchToChatConversation = () => {
    elements.appChatConversation.classList.replace('d-flex', 'd-none');
    elements.appChatHistory.classList.replace('d-none', 'd-block');
  };

  /**
   * Filter chat contacts by search input.
   * @param {string} selector - CSS selector for chat/contact list items.
   * @param {string} searchValue - Search input value.
   * @param {string} placeholderSelector - Selector for placeholder element.
   */
  const filterChatContacts = (selector, searchValue, placeholderSelector) => {
    const items = document.querySelectorAll(`${selector}:not(.chat-contact-list-item-title)`);
    let visibleCount = 0;

    items.forEach(item => {
      const isVisible = item.textContent.toLowerCase().includes(searchValue);
      item.classList.toggle('d-flex', isVisible);
      item.classList.toggle('d-none', !isVisible);
      if (isVisible) visibleCount++;
    });

    document.querySelector(placeholderSelector)?.classList.toggle('d-none', visibleCount > 0);
  };

  /**
   * Initialize speech-to-text functionality.
   */
  const initSpeechToText = () => {
    const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
    if (!SpeechRecognition || elements.speechToText.length === 0) return;

    const recognition = new SpeechRecognition();
    let listening = false;

    elements.speechToText.on('click', function () {
      if (!listening) recognition.start();
      recognition.onspeechstart = () => (listening = true);
      recognition.onresult = event => {
        $(this).closest('.form-send-message').find('.message-input').val(event.results[0][0].transcript);
      };
      recognition.onspeechend = () => (listening = false);
      recognition.onerror = () => (listening = false);
    });
  };

  // Initialize PerfectScrollbar
  initPerfectScrollbar([
    elements.chatContactsBody,
    elements.chatHistoryBody,
    elements.chatSidebarLeftBody,
    elements.chatSidebarRightBody
  ]);

  // Scroll to the bottom of the chat history
  scrollToBottom();

  // Attach user status change event
  elements.chatUserStatus.forEach(statusInput => {
    statusInput.addEventListener('click', () => updateUserStatus(statusInput.value));
  });

  // Handle max length for textarea
  const maxLength = parseInt(elements.chatSidebarLeftUserAbout.getAttribute('maxlength'), 10);
  handleMaxLengthCount(elements.chatSidebarLeftUserAbout, elements.textareaInfo, maxLength);

  elements.chatSidebarLeftUserAbout.addEventListener('input', () => {
    handleMaxLengthCount(elements.chatSidebarLeftUserAbout, elements.textareaInfo, maxLength);
  });

  // Attach chat conversation switch event
  elements.conversationButton?.addEventListener('click', switchToChatConversation);

  // Attach chat contact selection event
  elements.chatContactListItems.forEach(item => {
    item.addEventListener('click', () => {
      elements.chatContactListItems.forEach(contact => contact.classList.remove('active'));
      item.classList.add('active');
      switchToChatConversation();
    });
  });

  // Attach chat search filter event
  elements.searchInput?.addEventListener(
    'keyup',
    debounce(e => {
      const searchValue = e.target.value.toLowerCase();
      filterChatContacts('#chat-list li', searchValue, '.chat-list-item-0');
      filterChatContacts('#contact-list li', searchValue, '.contact-list-item-0');
    }, 300)
  );

  function scrollToMessage(messageId) {
    const target = document.querySelector(`#message-${messageId}`);
    if (target) {
      target.scrollIntoView({
        behavior: 'smooth',
        block: 'center'
      });
      // Optional: highlight the message briefly
      target.classList.add('highlight-reply');
      setTimeout(() => target.classList.remove('highlight-reply'), 3000);
    }
  }

  let replyToMessageId = null;
  let replyToMessageType = null;
  let replyToMessageContent = null;

  // Click message to reply
  document.addEventListener('click', function (e) {
    // 1️⃣ Ignore copy and download buttons
    if (e.target.closest('.copy-btn') || e.target.closest('.copyIcons a')) return;

    // 2️⃣ Handle quoted reply clicks (scroll to original)
    if (e.target.closest('.replyClick')) {
      const replyEl = e.target.closest('.replyClick');
      const targetId = replyEl.dataset.id;
      if (targetId) {
        scrollToMessage(targetId);
      }
      return; // ⬅ Stop here so it doesn't trigger reply mode again
    }

    // 3️⃣ Handle normal message reply (when clicking on message bubble)
    if (e.target.closest('.chat-message-text')) {
      const msgDiv = e.target.closest('.chat-message-text');
      replyToMessageId = msgDiv.dataset.messageId || null;
      replyToMessageType = msgDiv.dataset.messageType || 'text';
      replyToMessageContent = msgDiv.innerHTML;

      elements.replyPreviewText.innerHTML = replyToMessageContent;
      elements.replyPreviewContainer.classList.remove('d-none');
    }
  });

  // Remove reply preview
  elements.replyPreviewContainer.querySelector('.btn-close').addEventListener('click', () => {
    replyToMessageId = null;
    replyToMessageType = null;
    replyToMessageContent = null;
    elements.replyPreviewContainer.classList.add('d-none');
  });
  const attachInput = document.getElementById('attach-doc');
  const attachLabel = attachInput.closest('label');
  const attachLabelIcon = attachLabel.querySelector('i');
  const chatForm = document.getElementById('chatForm');

  let selectedFiles = [];

  // Container for file badges
  let fileBadgeContainer = document.createElement('div');
  fileBadgeContainer.className = 'file-badge-container position-absolute';
  fileBadgeContainer.style.cssText = 'top:-5px; right:3px; display:flex; gap:2px;';
  attachLabel.style.position = 'relative';
  attachLabel.appendChild(fileBadgeContainer);

  function updateIconAndBadges() {
    fileBadgeContainer.innerHTML = '';

    if (selectedFiles.length === 0) {
      attachLabelIcon.className = 'icon-base ti tabler-paperclip icon-22px text-heading';
    } else if (selectedFiles.length === 1) {
      const file = selectedFiles[0];
      if (file.type.startsWith('image/')) {
        attachLabelIcon.className = 'icon-base ti tabler-photo icon-22px text-success';
      } else if (file.type.startsWith('video/')) {
        attachLabelIcon.className = 'icon-base ti tabler-video icon-22px text-success';
      } else {
        attachLabelIcon.className = 'icon-base ti tabler-file icon-22px text-success';
      }
    } else {
      attachLabelIcon.className = 'icon-base ti tabler-paperclip icon-22px text-success';
    }

    // Add badges for each file with remove button
    selectedFiles.forEach((file, index) => {
      const badge = document.createElement('span');
      badge.className = 'file-badge bg-danger text-white rounded-circle';
      badge.style.cssText = 'font-size:9px; padding:2px 4px; cursor:pointer;';
      badge.title = 'Remove file';
      badge.textContent = index + 1;
      badge.addEventListener('click', () => {
        selectedFiles.splice(index, 1);
        attachInput.value = ''; // Reset input
        updateIconAndBadges();
      });
      fileBadgeContainer.appendChild(badge);
    });
  }

  attachInput.addEventListener('change', e => {
    selectedFiles = Array.from(e.target.files);
    updateIconAndBadges();
  });

  // Attach message send event
  elements.formSendMessage?.addEventListener('submit', e => {
    e.preventDefault();
    const form = elements.formSendMessage;
    const formData = new FormData(form);
    // Append reply info if replying
    if (replyToMessageId) {
      formData.append('reply_to_message_id', replyToMessageId);
    }

    const $submitBtn = $(form).find('.send-msg-btn');

    // Toggle button loading (show spinner)
    toggleButtonLoading($submitBtn, true, 'Sending...');

    $.ajax({
      url: baseURL + '/chat/sendmessage', // single endpoint
      type: 'POST',
      data: formData,
      contentType: false,
      processData: false,
      success: function (res) {
        // Create message bubble
        const messageDiv = document.createElement('div');
        messageDiv.className = 'chat-message-text mt-2';
        // Set data attributes
        messageDiv.dataset.messageId = res.id;
        messageDiv.dataset.messageType = res.type;
        // Read indicator class
        const readIndicatorClass = 'text-success';
        const readIndicator = `<i class="icon-base ti tabler-checks icon-16px ${readIndicatorClass} me-1"></i>`;
        // Build inner HTML: message + read indicator + time
        messageDiv.innerHTML = `
  ${res.message}
  <div class="text-end text-body-secondary mt-0">
    ${readIndicator}<small>${res.time}</small>
  </div>
`;
        const lastMessage = document.querySelector('.chat-history li:last-child');
        if (lastMessage && lastMessage.classList.contains('chat-message-right')) {
          // 👉 Last message is mine, append to its wrapper
          lastMessage.querySelector('.chat-message-wrapper')?.appendChild(messageDiv);
        } else {
          // 👉 Create a new message bubble on the right
          const newLi = document.createElement('li');
          newLi.className = 'chat-message chat-message-right';
          newLi.innerHTML = `
        <div class="d-flex overflow-hidden" style="max-width: 75%; width: fit-content;min-width:35%">
          <div class="chat-message-wrapper flex-grow-1"></div>
        </div>
      `;
          newLi.querySelector('.chat-message-wrapper')?.appendChild(messageDiv);
          document.querySelector('.chat-history')?.appendChild(newLi);
        }
        elements.messageInput.value = '';
        form.reset();
        updateIconAndBadges();
        scrollToBottom();
      },
      error: function (xhr) {
        toggleButtonLoading($submitBtn, false);
        console.error(xhr.responseText);
      },
      complete: function () {
        // Re-enable button & hide spinner
        toggleButtonLoading($submitBtn, false);
      }
    });
  });

  // Fix overlay issue for chat sidebar
  elements.chatHistoryHeader?.addEventListener('click', () => {
    document.querySelector('.app-chat-sidebar-left .close-sidebar')?.removeAttribute('data-overlay');
  });

  // Initialize speech-to-text
  initSpeechToText();
});

/**
 * Debounce utility function.
 * @param {Function} func - Function to debounce.
 * @param {number} wait - Delay in milliseconds.
 */
function debounce(func, wait) {
  let timeout;
  return (...args) => {
    clearTimeout(timeout);
    timeout = setTimeout(() => func.apply(this, args), wait);
  };
}
