<?php
namespace App\Services;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\DB;
use App\Helpers\Helpers;
class WalletService
{
  public function getOrCreate(string $ownerType, int $ownerId): Wallet
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    if ($ownerType === 'user') {
      $currencyFromCountry = DB::table('users as u')
        ->join('countries as c', 'c.id', '=', 'u.current_country_id')
        ->where('u.id', $ownerId)
        ->value('c.currency_symbol');
    } elseif ($ownerType === 'store') {
      $currencyFromCountry = DB::table('stores as s')
        ->join('countries as c', 'c.id', '=', 's.country_id')
        ->where('s.id', $ownerId)
        ->value('c.currency_symbol');
    } elseif ($ownerType === 'driver') {
      $currencyFromCountry = DB::table('drivers as d')
        ->join('countries as c', 'c.id', '=', 'd.country_id')
        ->where('d.id', $ownerId)
        ->value('c.currency_symbol');
    } else {
      $currencyFromCountry = $currency;
    }

    if ($currencyFromCountry) {
      $currency = $currencyFromCountry;
    }
    return DB::transaction(function () use ($ownerType, $ownerId, $currency) {
      // Lock row if exists
      $wallet = Wallet::where('owner_type', $ownerType)
        ->where('owner_id', $ownerId)
        ->lockForUpdate()
        ->first();
      if (!$wallet) {
        $wallet = Wallet::create([
          'owner_type' => $ownerType,
          'owner_id' => $ownerId,
          'balance' => 0,
          'pending_dues' => 0,
          'currency' => $currency,
        ]);
      }
      return $wallet;
    });
  }

  /** Credit earning to wallet */
  public function creditEarning(
    Wallet $wallet,
    float $amount,
    string $orderType,
    int $orderId,
    string $paymentMode,
    ?string $idempotencyKey = null,
    string $desc = ''
  ): Wallet {
    return DB::transaction(function () use ($wallet, $amount, $orderType, $orderId, $paymentMode, $idempotencyKey, $desc) {
      // Idempotency
      if ($idempotencyKey && WalletTransaction::where('wallet_id', $wallet->id)->where('idempotency_key', $idempotencyKey)->exists()) {
        return $wallet->refresh();
      }
      // Lock
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      $wallet->balance = bcadd($wallet->balance, $amount, 2);
      $wallet->save();
      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => $orderType,
        'order_id' => $orderId,
        'amount' => $amount,
        'type' => 'credit',
        'payment_mode' => $paymentMode,
        'status' => 'completed',
        'description' => $desc ?: 'Earning credited',
        'idempotency_key' => $idempotencyKey,
      ]);

      $this->syncOwnerCachedWallet($wallet);
      return $wallet;
    });
  }

  /** Add commission dues (driver/store owes platform) */
  public function addDue(
    Wallet $wallet,
    float $due,
    string $orderType,
    int $orderId,
    ?string $idempotencyKey = null,
    string
    $desc = ''
  ): Wallet {
    return DB::transaction(function () use ($wallet, $due, $orderType, $orderId, $idempotencyKey, $desc) {
      if ($idempotencyKey && WalletTransaction::where('wallet_id', $wallet->id)->where('idempotency_key', $idempotencyKey)->exists()) {
        return $wallet->refresh();
      }

      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      $wallet->pending_dues = bcadd($wallet->pending_dues, $due, 2);
      $wallet->save();

      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => $orderType,
        'order_id' => $orderId,
        'amount' => $due,
        'type' => 'debit',
        'payment_mode' => 'system',
        'status' => 'completed',
        'description' => $desc ?: 'Commission due added',
        'idempotency_key' => $idempotencyKey,
      ]);

      $this->syncOwnerCachedWallet($wallet);
      return $wallet;
    });
  }

  /** Try to settle dues automatically from wallet balance */
  public function tryAutoSettleDues(Wallet $wallet, ?string $reason = null): Wallet
  {
    return DB::transaction(function () use ($wallet, $reason) {
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      if ($wallet->pending_dues <= 0)
        return $wallet;
      $settle = min($wallet->balance, $wallet->pending_dues);
      if ($settle <= 0)
        return $wallet;
      $wallet->balance = bcsub($wallet->balance, $settle, 2);
      $wallet->pending_dues = bcsub($wallet->pending_dues, $settle, 2);
      $wallet->save();

      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => 'settlement',
        'order_id' => 0,
        'amount' => $settle,
        'type' => 'debit',
        'payment_mode' => 'system',
        'status' => 'completed',
        'description' => $reason ?: 'Auto dues settlement',
      ]);

      $this->syncOwnerCachedWallet($wallet);
      return $wallet;
    });
  }

  /** Manual dues payment (e.g., driver pays online) */
  public function manualPayDues(Wallet $wallet, float $amount, string $desc = 'Manual dues payment'): Wallet
  {
    return DB::transaction(function () use ($wallet, $amount, $desc) {
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      //$wallet->pending_dues = max(0, bcsub($wallet->pending_dues, $amount, 2));
      if ($wallet->pending_dues <= 0)
        return $wallet;
      $settle = min($wallet->balance, $amount);
      if ($settle <= 0)
        return $wallet;
      $wallet->balance = bcsub($wallet->balance, $settle, 2);
      $wallet->pending_dues = bcsub($wallet->pending_dues, $settle, 2);
      $wallet->save();

      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => 'settlement',
        'order_id' => 0,
        'amount' => $amount,
        'type' => 'debit',
        'payment_mode' => 'system',
        'status' => 'completed',
        'description' => $desc,
      ]);

      $this->syncOwnerCachedWallet($wallet);
      return $wallet;
    });
  }

  public function manualReceiveDues(Wallet $wallet, float $amount, string $desc = 'Manual receive payment'): Wallet
  {
    return DB::transaction(function () use ($wallet, $amount, $desc) {
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      //$wallet->pending_dues = max(0, bcsub($wallet->pending_dues, $amount, 2));
      if ($wallet->pending_dues <= 0)
        return $wallet;
      $wallet->pending_dues = bcsub($wallet->pending_dues, $amount, 2);
      $wallet->save();

      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => 'settlement',
        'order_id' => 0,
        'amount' => $amount,
        'type' => 'debit',
        'payment_mode' => 'system',
        'status' => 'completed',
        'description' => $desc,
      ]);
      return $wallet;
    });
  }

  public function withdrawalRequest($user, $amount, $type = 'decrement'): ?Wallet
  {
    // Update individual user type table
    if ($user->user_type === 'Driver') {
      DB::table('drivers')->where('user_id', $user->id)->{$type}('wallet', $amount);
      $wallet = Wallet::where('owner_type', 'driver')
        ->where('owner_id', $user->driverId)->lockForUpdate()->first();
    } elseif ($user->user_type === 'Store') {
      DB::table('stores')->where('user_id', $user->id)->{$type}('wallet', $amount);
      $wallet = Wallet::where('owner_type', 'store')
        ->where('owner_id', $user->storeId)->lockForUpdate()->first();
    } else {
      DB::table('users')->where('id', $user->id)->{$type}('wallet', $amount);
      $wallet = Wallet::where('owner_type', 'user')
        ->where('owner_id', $user->id)->lockForUpdate()->first();
    }

    // Update central wallet balance
    if ($wallet) {
      if ($type === 'decrement') {
        if ($wallet->balance >= $amount) {
          $wallet->decrement('balance', $amount);
        } else {
          throw new \Exception(__('locale.Insufficient wallet balance'));
        }
      } else {
        $wallet->increment('balance', $amount);
      }
    }

    return $wallet;
  }

  public function walletTransaction($user, $amount, $withdrawalRequest, $type): ?Wallet
  {
    // Update individual user type table
    if ($user->user_type === 'Driver') {
      $wallet = Wallet::where('owner_type', 'driver')
        ->where('owner_id', $user->driverId)->lockForUpdate()->first();
    } elseif ($user->user_type === 'Store') {
      $wallet = Wallet::where('owner_type', 'store')
        ->where('owner_id', $user->storeId)->lockForUpdate()->first();
    } else {
      $wallet = Wallet::where('owner_type', 'user')
        ->where('owner_id', $user->id)->lockForUpdate()->first();
    }

    WalletTransaction::create([
      'wallet_id' => $wallet->id,
      'order_type' => 'wallet',
      'order_id' => $withdrawalRequest->id,
      'amount' => $amount,
      'type' => $type,
      'payment_mode' => 'system',
      'status' => 'completed',
      'description' => $type == 'debit' ? 'Earning credited (withdrawal request)' : 'Earning debited (withdrawal request)',
    ]);
    return $wallet;
  }

  public function walletPayment(Wallet $wallet, float $amount, string $desc = 'Wallet payment', $idempotencyKey, $id): Wallet
  {
    return DB::transaction(function () use ($wallet, $amount, $desc, $idempotencyKey, $id) {
      if ($idempotencyKey && WalletTransaction::where('wallet_id', $wallet->id)->where('idempotency_key', $idempotencyKey)->exists()) {
        return $wallet->refresh();
      }
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      $settle = min($wallet->balance, $amount);
      if ($settle <= 0)
        return $wallet;
      $wallet->balance = bcsub($wallet->balance, $settle, 2);
      $wallet->save();

      WalletTransaction::create([
        'wallet_id' => $wallet->id,
        'order_type' => 'wallet',
        'order_id' => $id,
        'amount' => $settle,
        'type' => 'debit',
        'payment_mode' => 'system',
        'status' => 'completed',
        'description' => $desc,
        'idempotency_key' => $idempotencyKey,
      ]);

      $this->syncOwnerCachedWallet($wallet);
      return $wallet;
    });
  }
  public function chargeOrDue(Wallet $wallet, float $amount, string $type, int $rideId, ?string $idempotencyKey = null, string $note = 'Ride cancellation fee'): array
  {
    return DB::transaction(function () use ($wallet, $amount, $type, $rideId, $idempotencyKey, $note) {
      if (
        $idempotencyKey && WalletTransaction::where('wallet_id', $wallet->id)
          ->where('idempotency_key', $idempotencyKey)
          ->exists()
      ) {
        return ['deducted' => 0, 'due' => 0]; // already processed
      }
      // Lock wallet for update
      $wallet = Wallet::whereKey($wallet->id)->lockForUpdate()->firstOrFail();
      // Determine amount to settle from wallet
      $settle = min($wallet->balance, $amount);
      $due = $amount - $settle;
      // Deduct wallet if any
      if ($settle > 0) {
        $wallet->balance = bcsub($wallet->balance, $settle, 2);
        $wallet->save();
        WalletTransaction::create([
          'wallet_id' => $wallet->id,
          'order_type' => $type,
          'order_id' => $rideId,
          'amount' => $settle,
          'type' => 'debit',
          'payment_mode' => 'system',
          'status' => 'completed',
          'description' => $note,
          'idempotency_key' => $idempotencyKey ? $idempotencyKey . '-wallet' : null,
        ]);
      }
      // Add due if any
      if ($due > 0) {
        $wallet->addDue($wallet, $due, $type, $rideId, $idempotencyKey ? $idempotencyKey . '-due' : null, $note . ' due');
      }
      // Sync wallet cache to owner table
      $this->syncOwnerCachedWallet($wallet);
      return ['deducted' => $settle, 'due' => $due];
    });
  }


  /** Keep drivers.wallet / stores.wallet fields in sync (cache) */
  protected function syncOwnerCachedWallet(Wallet $wallet): void
  {
    if ($wallet->owner_type === 'driver') {
      DB::table('drivers')->where('id', $wallet->owner_id)->update(['wallet' => $wallet->balance]);
    } elseif ($wallet->owner_type === 'store') {
      DB::table('stores')->where('id', $wallet->owner_id)->update(['wallet' => $wallet->balance]);
    } else {
      DB::table('users')->where('id', $wallet->owner_id)->update(['wallet' => $wallet->balance]);
    }
  }


}
