<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Ride;
use App\Models\User;
use App\Models\Setting;
use App\Models\Referral;
use App\Services\WalletService;
use Illuminate\Support\Str;
class ReferralService
{
  /**
   * Give referral reward (for customers or drivers).
   */
  public function giveReferralReward(User $user): void
  {
    // Skip if no referrer or already rewarded
    if (!$user->referred_by || $user->referral_bonus_credited) {
      return;
    }
    // Fetch settings once
    $keys = [
      'customer_min_order',
      'customer_refer_amount',
      'driver_min_order',
      'driver_refer_amount'
    ];
    $settings = Setting::whereIn('meta_key', $keys)
      ->pluck('meta_value', 'meta_key');
    $minOrders = (int) ($settings['customer_min_order'] ?? 2);
    $customerReward = (float) ($settings['customer_refer_amount'] ?? 100);

    $minRides = (int) ($settings['driver_min_order'] ?? 2);
    $driverReward = (float) ($settings['driver_refer_amount'] ?? 100);

    // 🔹 Check if user is a **Customer**
    if ($user->user_type === 'User') {
      $completedOrders = Order::where('user_id', $user->id)
        ->where('status', 'Delivered')
        ->count();

      $completedRides = Ride::where('user_id', $user->id)
        ->where('status', 'Completed')
        ->count();

      $totalCompleted = $completedOrders + $completedRides;

      if ($totalCompleted >= $minOrders) {
        $this->creditReferral($user, $customerReward, 'Order');
      }
    }

    // 🔹 Check if user is a **Driver**
    if ($user->user_type === 'Driver') {
      $completedRides = Ride::where('driver_id', $user->id)
        ->where('status', 'Completed')
        ->count();

      if ($completedRides >= $minRides) {
        $this->creditReferral($user, $driverReward, 'Ride');
      }
    }
  }

  /**
   * Credit referral reward to referrer.
   */
  protected function creditReferral(User $user, float $amount, string $type): void
  {
    $referrer = User::find($user->referred_by);

    if (!$referrer) {
      return;
    }
    // 💰 Add to referrer wallet
    $referrer->wallet += $amount;
    $referrer->save();

    // 📦 Log referral transaction
    Referral::create([
      'user_id' => $referrer->id,
      'refer_id' => $user->id,
      'points' => $amount,
      'type' => $type,
      'status' => 1,
    ]);
    // ✅ Mark referred user/driver as rewarded
    $user->referral_bonus_credited = true;
    $user->save();
    $wallets = new WalletService;
    $idempotencyKey = Str::uuid()->toString();
    $userType = strtolower($referrer->user_type);
    $ownerId = $userType == 'driver' ? $referrer->driverId : ($userType == 'store' ? $referrer->storeId : $referrer->id);
    $userWallet = $wallets->getOrCreate($userType, $ownerId);
    $wallets->creditEarning($userWallet, $amount, 'referral', $user->referred_by, 'online', $idempotencyKey, 'Referral earning');
    cacheForget("dropdown_referrals_user", ['dropdown']);
  }
}
