<?php

namespace App\Repositories\Eloquent;
use App\Models\User;
use App\Models\Driver;
use App\Models\Address;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Mail\AccountApproveEmail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\SetPasswordEmail;
use App\Models\PendingVerification;
use App\Mail\EmailOtpMail;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\DB;
use App\Traits\ApiResponseTrait;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Repositories\Contracts\UserRepositoryInterface;
class UserRepository implements UserRepositoryInterface
{
  use ApiResponseTrait;
  public function all()
  {
    return User::where('user_type', 'User')->get();
  }
  public function allList(Request $request)
  {
    $canDelete = Auth::user()->can('user-delete');
    $canEdit = Auth::user()->can('user-edit');
    $canCreate = Auth::user()->can('user-create');
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $builder = User::where('user_type', 'User');

    // Clone for filters
    $query = clone $builder;
    if ($search = $request->input('search.value')) {
      $query->where(function ($q) use ($search) {
        $q->where('name', 'LIKE', "%{$search}%")
          ->orWhere('email', 'LIKE', "%{$search}%")
          ->orWhere('phone', 'LIKE', "%{$search}%");
      });
    }
    /*   if (!is_null($request->input('verified'))) {
        $verified = filter_var($request->input('verified'), FILTER_VALIDATE_BOOLEAN);
        $query->where($verified ? fn($q) => $q->whereNotNull('email_verified_at') : fn($q) => $q->whereNull('email_verified_at'));
      } */
    if (!is_null($request->input('verified'))) {
      $query->where('is_verified', (int) $request->input('verified'));
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', (int) $request->input('status'));
    }
    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('default_currency', $currency)
      ->addColumn('verified', fn($user) => $user->is_verified)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($user) => url('admin/users/' . $user->id))
      ->addColumn('status-url', fn() => url('admin/users/changestatus'))
      ->with([
        'recordsTotal' => $builder->count(), // ✅ total regardless of filters
        'create' => $canCreate // ✅ make it root-level
      ])
      ->make(true);
  }
  public function find($id)
  {
    return User::with('addresses')->withTrashed()->find($id);
  }
  public function store(Request $request): array
  {
    $data = [];
    $userID = $request->id;
    if ($userID) {
      $user = User::find($userID);
      $user->name = $request->name;
      $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
      $user->dial_code = $request->dial_code;
      $user->dial_code_iso = $request->dial_code_iso;
      $user->email = $request->email;
      $user->save();
      $data['form'] = 'Updated';
    } else {
      $country = Country::where('sortname', $request->dial_code_iso)->first();
      $user = new User;
      $user->name = $request->name;
      $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
      $user->dial_code = $request->dial_code;
      $user->dial_code_iso = $request->dial_code_iso;
      $user->email = $request->email;
      $user->status = 1;
      $user->user_type = $request->user_type ?? 'User';
      $user->email_verified_at = now();
      $user->is_verified = 1;
      $user->password = Hash::make($request->password);
      $user->current_country_id = $country ? $country->id : 0;
      $user->currency = $country ? $country->currency_symbol : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $user->timezone = $country ? $country->timezone : config('app.timezone', 'UTC');
      $user->save();
      $user->referral_code = Helpers::random_strings(6) . $user->id;
      $user->save();
      $data['form'] = 'Created';

      $message = 'Your Account on ' . config('app.name') . ' has been created by Admin. Now you can login to your account on ' . config('app.name') . ' App with the below credentials. <br>Following are the credentials for login <br><br> <b>Login ID: </b>' . $user->phone . '<br><b>Password: </b>' . $request->password . '<br><b>url: </b>' . url('');
      $greetings = 'Congratulations';
      $title = "Account Registration " . $user->email;
      $array['view'] = 'emails.accountApprove';
      $array['subject'] = $title;
      $array['name'] = $user->name;
      $array['greetings'] = $greetings;
      $array['content'] = $message;
      try {
        if (!empty($user->email)) {
          Mail::to($user->email)->queue(new AccountApproveEmail($array));
        }
      } catch (\Exception $e) {
      }
    }
    $data['id'] = $user->id;
    return $data;
  }
  public function softDelete($id)
  {
    $user = User::find($id);
    /*  $user->status = 2;
     if (!empty($user->email)) {
       $user->email = $user->email . '_anonymous' . $user->id;
     }
     $user->phone = $user->phone . '_' . $user->id;
     $user->save(); */
    $user->tokens()->where('tokenable_id', $id)->delete();
    $user->delete();
    return true;
  }
  public function changeStatus($id, $status)
  {
    $user = User::find($id);
    if ($user) {
      $user->status = $status == 1 ? 0 : 1;
      $user->save();
    }
    return true;
  }
  public function setPassword(array $data)
  {
    $user = User::find($data['id']);
    $user->password = Hash::make($data['password']);
    $user->save();
    $array['view'] = 'emails.setPassword';
    $array['subject'] = 'Reset Password By ' . config('app.name') . ' Admin';
    $array['name'] = $user->name;
    $array['content'] = "Your Password on " . config('app.name') . "  has been reset by admin. Please login with new password. Email: " . $user->email . " Password: " . $data['password'] . ' .';
    try {
      if (!empty($user->email)) {
        Mail::to($user->email)->queue(new SetPasswordEmail($array));
      }
    } catch (\Exception $e) {
    }
    return true;
  }

  public function create(Request $request)
  {
    try {
      //code for email and mobile otp
      $pending = PendingVerification::where('email', $request->email)
        ->where('mobile', $request->mobile)->first();
      if (!$pending) {
        return $this->errorResponse([], __('locale.No pending verification found.'), 400);
      }
      // 1️⃣ Verify Email OTP
      if ($pending->email_otp !== $request->email_otp || $pending->email_otp_expires_at->isPast()) {
        return $this->errorResponse([], __('locale.Invalid or expired email OTP.'), 400);
      }
      // 2️⃣ Verify Mobile OTP with Firebase
      $firebaseApiKey = config('services.firebase.api_key');
      $firebaseResponse = Http::post("https://identitytoolkit.googleapis.com/v1/accounts:signInWithPhoneNumber?key={$firebaseApiKey}", [
        'sessionInfo' => $request->firebase_verification_id,
        'code' => $request->firebase_otp,
      ]);
      if (!$firebaseResponse->successful()) {
        return $this->errorResponse([], __('locale.Invalid mobile OTP.'), 400);
      }
      //code for email and mobile otp
      $country = Country::where('sortname', $request->dial_code_iso)->first();
      $user = new User;
      $user->name = $request->name;
      $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
      $user->dial_code = $request->dial_code;
      $user->dial_code_iso = $request->dial_code_iso;
      $user->email = $request->email;
      $user->password = Hash::make($request->password);
      $user->current_country_id = $country ? $country->id : 0;
      $user->currency = $country ? $country->currency_symbol : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $user->timezone = $country ? $country->timezone : config('app.timezone', 'UTC');
      $user->user_type = $request->user_type ?? 'User';
      $user->fcm_token = $request->fcm_token;
      // 🔹 If referral code is provided, find referrer
      if ($request->filled('referral_code')) {
        $referrer = User::where('referral_code', $request->referral_code)->first();
        if ($referrer) {
          $user->referred_by = $referrer->id;
        }
      }
      $user->save();
      $user->referral_code = Helpers::random_strings(6) . $user->id;
      $user->save();
      if ($request->user_type == 'Driver') {
        $driver = new Driver;
        $driver->user_id = $user->id;
        $driver->type = $request->ride_type;
        $user->country_id = $country ? $country->id : 0;
        $user->country = $country ? $country->name : null;
        $driver->save();
      }

      // Send verification email
      $user->sendEmailVerificationNotification();

    } catch (\Exception $e) {
    }
    return $user;
  }
  public function addressList()
  {
    $userId = auth()->id();
    return Address::where('user_id', $userId)->where('status', 1)->orderByDesc('is_default')->get();
  }
  public function address($request)
  {

    $isDefault = 0;
    $userId = auth()->id();
    if (!$request->id) {
      $existingAddressesCount = Address::where('user_id', $userId)->count();
      $isDefault = $existingAddressesCount === 0 ? 1 : 0;
    } else {
      $existingAddresses = Address::find($request->id);
      $isDefault = $request->is_default ?? $existingAddresses->is_default;
    }
    $country = DB::table('countries')->find($request->country_id);
    $state = DB::table('states')->find($request->state_id);
    $city = DB::table('cities')->find($request->city_id);
    return Address::updateOrCreate(
      ['id' => $request->id ?? null], // update if ID exists
      [
        'user_id' => $userId,
        'name' => $request->name,
        'address' => $request->address,
        'country' => $country->name,
        'state' => $state->name,
        'city' => $city->name,
        'country_id' => $request->country_id,
        'state_id' => $request->state_id,
        'city_id' => $request->city_id,
        'latitude' => $request->latitude,
        'longitude' => $request->longitude,
        'postcode' => $request->postcode,
        'type' => $request->type,
        'is_default' => $isDefault,
        'status' => $request->status ?? 1,
      ]
    );
  }
  public function deleteAddress($id)
  {
    $address = Address::where('user_id', auth()->id())->findOrFail($id);
    $address->delete();
    // If the deleted one was default, set another one as default
    if ($address->is_default) {
      $next = Address::where('user_id', auth()->id())->first();
      if ($next) {
        $next->update(['is_default' => 1]);
      }
    }
    return true;
  }
  public function setDefaultAddress($id)
  {
    $userId = auth()->id();
    // Reset all to 0
    Address::where('user_id', $userId)->update(['is_default' => 0]);
    // Set selected as default
    $address = Address::where('user_id', $userId)->findOrFail($id);
    $address->update(['is_default' => 1]);
    return $address;
  }
  public function sendOtp(Request $request)
  {
    $emailOtp = rand(100000, 999999);
    PendingVerification::updateOrCreate(
      ['email' => $request->email],
      [
        'mobile' => $request->mobile,
        'email_otp' => $emailOtp,
        'email_otp_expires_at' => now()->addMinutes(10),
      ]
    );
    // Send Email OTP
    Mail::to($request->email)->send(new EmailOtpMail($emailOtp));
    // Trigger Firebase mobile OTP from client side (not server)
    return true;
  }

}
