<?php

namespace App\Repositories\Eloquent;
use App\Models\User;
use App\Models\Store;
use App\Models\BusinessHour;
use App\Models\Category;
use App\Models\Favorite;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Mail\AccountApproveEmail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Storage;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\StoreRepositoryInterface;
class StoreRepository extends BaseRepository implements StoreRepositoryInterface
{
  public function __construct(User $storeModel)
  {
    parent::__construct($storeModel);
  }

  public function allList($request)
  {
    $canDelete = Auth::user()->can('store-delete');
    $canEdit = Auth::user()->can('store-edit');
    $canCreate = Auth::user()->can('store-create');
    $walletList = Auth::user()->can('wallet-list');
    $builder = User::with('store')->where('user_type', 'Store');
    // Clone for filters
    $query = clone $builder;
    if ($search = $request->input('search.value')) {
      $query->where(function ($q) use ($search) {
        $q->where('name', 'LIKE', "%{$search}%")
          ->orWhere('email', 'LIKE', "%{$search}%")
          ->orWhere('phone', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('verified'))) {
      $query->where('is_verified', (int) $request->input('verified'));
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', (int) $request->input('status'));
    }
    return \DataTables::eloquent($query->with(['store']))
      ->addIndexColumn()
      ->addColumn('verified', fn($user) => $user->is_verified)
      ->addColumn('popular', fn($user) => $user['store']->is_popular)
      ->addColumn('rating', fn($user) => $user['store']->rating)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('walletList', $walletList)
      ->addColumn('delete-url', fn($user) => url('admin/users/' . $user->id))
      ->addColumn('status-url', fn() => url('admin/users/changestatus'))
      ->addColumn('popular-url', fn() => url('admin/stores/changePopular'))
      ->addColumn('impersonate-url', fn($user) => url('admin/stores/impersonate/' . $user->id))
      ->addColumn('store_id', fn($user) => optional($user->store)->id)
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }


  public function allMaps(): array
  {
    $users = User::with('store')->whereHas('store', function ($query) {
      $query->whereNotNull('latitude');
    })->get();
    foreach ($users as $user) {
      $data[] = [
        'id' => $user->id,
        'name' => $user->name,
        'email' => $user->email,
        'phone' => $user->phone,
        'profile_photo_path' => $user->profile_photo_path,
        'status' => $user->status,
        'email_verified_at' => $user->email_verified_at,
        'store_id' => $user['store']->id,
        'latitude' => $user['store']->latitude,
        'longitude' => $user['store']->longitude,
      ];
    }
    return $data;
  }

  public function getStores(Request $request)
  {
    $category_id = $request->main_category;
    // Get all child category IDs
    $categoryIds = Category::where('parent_id', $category_id)
      ->pluck('id'); // ✅ get only IDs
    return Store::with(['user:id,name'])
      ->whereHas('user', function ($query) {
        $query->where('status', 1);
      })
      ->whereHas('categories', function ($query) use ($categoryIds) {
        $query->whereIn('categories.id', $categoryIds);
      })->select('id', 'user_id')->get();
  }

  public function store(Request $request, $Id = null): array
  {
    $data = [];

    $store = $Id ? Store::find($Id) : new Store;
    $country = DB::table('countries')->find($request->country);
    $state = DB::table('states')->find($request->state);
    $city = DB::table('cities')->find($request->city);
    $store->address = $request->address;
    $store->city = $city->name ?? $request->city;
    $store->state = $state->name ?? $request->state;
    $store->country = $country->name ?? $request->country;
    $store->city_id = $request->city;
    $store->state_id = $request->state;
    $store->country_id = $request->country;
    $store->postcode = $request->postcode;
    $store->landmark = $request->landmark;
    $store->latitude = $request->latitude;
    $store->longitude = $request->longitude;
    $store->description = $request->description;

    $store->contact_person_name = $request->contact_person_name;
    $store->contact_person_phone = preg_replace('/[\s\-\(\)]+/', '', $request->contact_person_phone);
    $store->dial_code_contact = $request->dial_code_contact;
    $store->dial_code_contact_iso = $request->dial_code_contact_iso;
    $store->delivery_time = $request->delivery_time;
    $store->service_tax = $request->service_tax;
    $store->commission_charge = $request->commission_charge;
    $store->website = $request->website;
    $store->save();

    $user = $Id ? User::find($store->user_id) : new User;
    if ($request->hasFile('store_image')) {
      if ($Id && $user->profile_photo_path) {
        Storage::disk('public')->delete($user->profile_photo_path);
      }
      $user->profile_photo_path = $request->file('store_image')->store("stores", 'public');
    }
    $user->name = $request->name;
    $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
    $user->dial_code_iso = $request->dial_code_iso;
    $user->dial_code = $request->dial_code;
    $user->email = $request->email;
    $user->bank_details = $request->bank_details;
    if (empty($Id)) {
      $user->status = 1;
      $user->user_type = $request->user_type ?? 'Store';
      $user->email_verified_at = now();
      $user->is_verified = 1;
      $user->password = Hash::make($request->password);
    }
    $user->currency = $country ? $country->currency_symbol : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $user->save();
    $store->user_id = $user->id;
    $store->save();
    $store->categories()->sync($request->categories);
    $user->referral_code = Helpers::random_strings(6) . $user->id;
    $user->save();

    if (empty($Id)) {
      $message = 'Your Store Account on ' . config('app.name') . ' has been created by Admin. Now you can login to your account on ' . config('app.name') . ' Dashboard with the below credentials. <br>Following are the credentials for login <br><br> <b>Login ID: </b>' . $user->email . '<br><b>Password: </b>' . $request->password . '<br><b>url: </b>' . url('');
      $greetings = 'Congratulations';
      $title = "Account Registration " . $user->email;
      $array['view'] = 'emails.accountApprove';
      $array['subject'] = $title;
      $array['name'] = $user->name;
      $array['greetings'] = $greetings;
      $array['content'] = $message;
      try {
        if (!empty($user->email)) {
          Mail::to($user->email)->queue(new AccountApproveEmail($array));
        }
      } catch (\Exception $e) {
      }
    }
    $data['id'] = $user->id;
    return $data;
  }

  public function changeStatus($id, $status)
  {
    $user = User::find($id);
    if ($user) {
      $user->is_verified = $status == '1' ? 1 : 0;
      $user->save();
    }
    return $user;
  }

  public function changePopular($id, $status)
  {
    $user = Store::where('user_id', $id)->first();
    if ($user) {
      $user->is_popular = $status == 1 ? 0 : 1;
      $user->save();
    }
    return $user;
  }
  public function impersonate($storeId)
  {

    $storeOwner = User::where('id', $storeId)
      ->where('user_type', 'Store')
      ->firstOrFail();
    session(['impersonate_admin_id' => Auth::id()]);
    session()->regenerate();
    Auth::guard('web')->login($storeOwner);

    return true;
  }
  public function stopImpersonate()
  {
    if (session()->has('impersonate_admin_id')) {
      $adminId = session('impersonate_admin_id');
      session()->forget('impersonate_admin_id');
      session()->regenerate();
      Auth::guard('web')->loginUsingId($adminId);
    }
    return true;
  }

  public function storeStatus($id, $status)
  {
    $store = Store::find($id);
    if ($store) {
      $store->is_active = $status == 1 ? 0 : 1;
      $store->save();
    }
    return true;
  }

  public function storePaymentStatus($id, $status, $type)
  {
    $store = Store::find($id);
    if ($store) {
      $store->$type = $status == 1 ? 0 : 1;
      $store->save();
    }
    return true;
  }

  public function availabilityUpdate(Request $request)
  {
    $store = Store::where('user_id', auth()->id())->firstOrFail();

    $store->always_open = $request->always_open ?? 1;
    $store->save();
    function businessDays()
    {
      return ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
    }
    // Loop through business days
    foreach (businessDays() as $key => $day) {
      $status = $request->input("d{$key}_status", 0);
      $start = $request->input("d{$key}_OH", null);
      $end = $request->input("d{$key}_CH", null);
      // Convert times to 24-hour format if provided
      $startFormatted = $start ? date("H:i:s", strtotime($start)) : null;
      $endFormatted = $end ? date("H:i:s", strtotime($end)) : null;
      BusinessHour::updateOrCreate(
        ['store_id' => $store->id, 'day' => $day],
        [
          'status' => $status,
          'start_time' => $startFormatted,
          'end_time' => $endFormatted
        ]
      );
    }
    return true;
  }

  public function storeDashboard(Request $request, $type, $perPage = 10)
  {
    $user = auth()->user();
    $countryId = $user->current_country_id ?? 94;
    $query = Store::query()
      ->select('stores.*')
      ->with('user')
      ->active($countryId);
    // Featured stores
    if ($type === 'featured') {
      $query->where('is_popular', 1);
    }
    $showDietType = false;
    // Category filter (parent + subcategories)
    if ($request->filled('category_id')) {
      $category = Category::with('children')->find($request->category_id);
      if ($category) {
        $showDietType = $category->diet == 1;
        $categoryIds = $this->getAllCategoryIds($category);
        $query->whereHas('categories', fn($q) => $q->whereIn('categories.id', $categoryIds));
      }
    }
    // Rating filter
    if ($request->filled('rating')) {
      $rating = (int) $request->rating;
      // only allow valid thresholds
      if (in_array($rating, [2, 3, 4, 5])) {
        $query->where('rating', '>=', $rating);
      }
    }
    // Veg Type filter
    if ($request->filled('veg_type')) {
      $vegType = $request->veg_type;
      if ($vegType === 'veg') {
        $query->where('is_non_veg', 0);
      } elseif ($vegType === 'non_veg') {
        $query->where('is_non_veg', 1);
      }
    }


    // Search filter (store fields + user name)
    if ($request->filled('search')) {
      $search = $request->search;
      $query->where(function ($q) use ($search) {
        $q->where('address', 'like', "%{$search}%")
          ->orWhere('city', 'like', "%{$search}%")
          ->orWhere('state', 'like', "%{$search}%")
          ->orWhere('country', 'like', "%{$search}%")
          ->orWhereHas('user', fn($q2) => $q2->where('name', 'like', "%{$search}%"));
      });
    }
    // Distance calculation
    if ($request->filled('latitude') && $request->filled('longitude')) {
      $lat = $request->latitude;
      $lng = $request->longitude;
      $maxTime = $request->input('max_time'); // e.g., 60 minutes
      $query->maxDeliveryTime($maxTime, $lat, $lng);
    } else {
      $query->select('stores.*');
      $query->orderByDesc('stores.created_at');
    }

    // Pagination
    $stores = $query->paginate($perPage);
    // Add is_favorite dynamically
    $favoriteStoreIds = $user
      ? Favorite::where('user_id', $user->id)
        ->where('favoritable_type', Store::class)
        ->pluck('favoritable_id')
        ->toArray()
      : [];
    $stores->getCollection()->transform(function ($store) use ($favoriteStoreIds, $showDietType) {
      $store->is_favorite = in_array($store->id, $favoriteStoreIds);
      $store->show_diet_type = $showDietType; // ✅ add diet type flag
      return $store;
    });
    return $stores;
  }

  private function getAllCategoryIds(Category $category)
  {
    $ids = [$category->id];
    if ($category->children->count()) {
      foreach ($category->children as $child) {
        $ids = array_merge($ids, $this->getAllCategoryIds($child));
      }
    }
    return $ids;
  }

  public function storeDetail(Request $request, $id)
  {
    $user = auth()->user();
    $countryId = $user->current_country_id ?? 94;
    $query = Store::query()
      ->select('stores.*')
      ->with([
        'user',
        'categories',
        'products' => function ($q) {
          $q->withCount('orderItems') // count how many times each product was sold
            ->orderByDesc('order_items_count')
            ->take(2)->with([
                'images' => function ($img) {
                  $img->orderBy('id')->limit(1); // only get first image
                }
                ,
                'category.parent'
              ]); // get top 4 best sellers
        }
      ])
      ->active($countryId)
      ->where('stores.id', $id);
    // Distance + estimated time filter
    if ($request->filled('latitude') && $request->filled('longitude')) {
      $lat = $request->latitude;
      $lng = $request->longitude;
      $maxTime = $request->input('max_time'); // e.g., 60 minutes
      $query->maxDeliveryTime($maxTime, $lat, $lng);
    } else {
      $query->orderByDesc('stores.created_at');
    }
    // Favorite stores
    $favoriteStoreIds = $user
      ? Favorite::where('user_id', $user->id)
        ->where('favoritable_type', Store::class)
        ->pluck('favoritable_id')
        ->toArray()
      : [];
    $store = $query->firstOrFail();
    // Add is_favorite flag
    $store->is_favorite = in_array($store->id, $favoriteStoreIds);
    // Store-level diet flag
    $store->show_diet_type = $store->categories->contains(fn($cat) => $cat->diet == 1);
    // Product-level diet flag for each product
    $store->products->transform(function ($product) {
      $product->show_diet_type =
        ($product->category && $product->category->diet == 1) ||
        ($product->category && $product->category->parent && $product->category->parent->diet == 1);
      return $product;
    });
    return $store;
  }


}
