<?php

namespace App\Repositories\Eloquent;
use App\Models\SosAlert;
use App\Models\SosContact;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use Carbon\Carbon;
use App\Jobs\SendSosSms;
use App\Repositories\BaseRepository;
use App\Traits\ApiResponseTrait;
use App\Repositories\Contracts\SosAlertRepositoryInterface;
class SosAlertRepository extends BaseRepository implements SosAlertRepositoryInterface
{
  use ApiResponseTrait;
  public function __construct(SosAlert $sosAlertModel)
  {
    parent::__construct($sosAlertModel);
  }

  public function allList($request)
  {
    $canDelete = Auth::user()->can('sos-delete');
    $canEdit = Auth::user()->can('sos-edit');
    $canCreate = Auth::user()->can('sos-create');
    $builder = SosAlert::query();
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('address', 'LIKE', "%{$search}%")
          ->orWhere('message', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    // 🎯 Filter by user_type
    if (!is_null($request->input('user_type'))) {
      $query->whereHas('user', function ($u) use ($request) {
        $u->where('user_type', $request->input('user_type'));
      });
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    return \DataTables::eloquent($query->with(['user']))
      ->addIndexColumn()
      ->addColumn('customer', fn($sos) => $sos['user']?->name ?? '-')
      ->addColumn('email', fn($sos) => $sos['user']?->email ?? '-')
      ->addColumn('user_type', fn($sos) => $sos['user']?->user_type ?? '')
      ->addColumn('avatar', function ($sos) {
        if ($sos['user']->profile_photo_path) {
          return asset('storage/' . $sos['user']->profile_photo_path);
        }
      })
      ->addColumn('currency', fn($sos) => $sos['user']?->currency ?? $currency)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($sos) => url($prefix . 'sos_alerts/' . $sos->id))
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }

  public function store(Request $request)
  {
    $user = auth()->user();
    // 1. Save SOS alert
    $sos = new SosAlert;
    $sos->user_id = auth()->id();
    $sos->lat = $request->latitude;
    $sos->lng = $request->longitude;
    $sos->address = $request->address;
    $sos->message = $request->message ?? '';
    $sos->save();
    // 2. Fetch user SOS contacts
    $contacts = SosContact::where('user_id', $user->id)->get();
    if ($contacts->isNotEmpty()) {
      // 3. Build SOS message
      $smsMessage = "🚨 SOS ALERT 🚨\n"
        . "From: {$user->name}\n"
        . "Message: {$sos->message}\n"
        . "Location: {$sos->address}\n"
        . "Map: https://maps.google.com/?q={$sos->lat},{$sos->lng}";

      // 4. Send SMS to all contacts
      foreach ($contacts as $contact) {
        SendSosSms::dispatch($contact->phone, $smsMessage);
      }
    }
    return true;
  }

  public function allContact()
  {
    $contacts = SosContact::where('user_id', auth()->id())->get();
    return $contacts;
  }

  public function addContacts(Request $request)
  {
    $userId = auth()->id();
    $contactsData = $request->input('contacts', []);
    // already saved contacts
    $existingCount = SosContact::where('user_id', $userId)->count();
    $newCount = count($contactsData);
    if (($existingCount + $newCount) > 3) {
      return $this->errorResponse([], __('locale.You can add maximum 3 SOS contacts'), 401);
    }
    $contacts = [];
    foreach ($contactsData as $contact) {
      +
        $contacts[] = SosContact::create([
          'user_id' => $userId,
          'name' => $contact['name'] ?? null,
          'phone' => $contact['phone'] ?? null,
        ]);
    }
    cacheForget("dropdown_sos_user", ['dropdown']);
    return $this->successResponse($contacts, __('locale.SOS Contacts added successfully.'), 200);
  }

  public function destroy($id)
  {
    SosContact::find($id)->delete();
    return true;
  }



}
