<?php
namespace App\Repositories\Eloquent;
use Illuminate\Http\Request;
use App\Models\Ride;
use App\Models\RideTemp;
use App\Models\Driver;
use App\Models\Order;
use App\Models\User;
use App\Models\VehicleCategory;
use App\Models\RideRequest;
use App\Models\CouponUsage;
use App\Models\DriverBlockLog;
use App\Models\RideWaitingTime;
use App\Events\RideAcceptedEvent;
use App\Traits\ApiResponseTrait;
use App\Events\RideTakenEvent;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Events\RideExpiredEvent;
use App\Events\RideEndEvent;
use App\Services\WalletService;
use App\Support\Finance;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Services\RidePriceService;
use App\Jobs\DriverUnblockJob;
use App\Events\DriverLocationUpdatedEvent;
use App\Jobs\UpdateDriverLocationJob;
use Illuminate\Support\Facades\Cache;
use App\Http\Resources\RideResource;
use App\Repositories\Contracts\RideRequestRepositoryInterface;
use App\Repositories\Contracts\RideRepositoryInterface;
use Illuminate\Support\Facades\Log;
class RideRepository implements RideRepositoryInterface
{
  use ApiResponseTrait;
  protected $rideRequestRepository, $rideService;
  public function __construct(
    RideRequestRepositoryInterface $rideRequestRepository,
    RidePriceService $rideService
  ) {
    $this->rideRequestRepository = $rideRequestRepository;
    $this->rideService = $rideService;
  }

  public function all()
  {
    $query = Ride::orderByDesc('created_at');
    return $query->get();
  }

  public function allList($request)
  {
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $canDelete = Auth::user()->user_type == 'Store' ? false : Auth::user()->can('ride-delete');
    $canEdit = Auth::user()->user_type == 'Store' ? false : Auth::user()->can('ride-edit');
    $canCreate = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('ride-create');
    $builder = Ride::orderByDesc('created_at');
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('ride_id', 'LIKE', "%{$search}%")
          ->orWhere('pickup_location', 'LIKE', "%{$search}%")
          ->orWhere('drop_location', 'LIKE', "%{$search}%");
      });
    }

    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }

    if (!is_null($request->input('mtype'))) {
      $query->whereIn('type', $request->input('mtype'));
    }
    if (!is_null($request->input('type'))) {
      $query->where('type', $request->input('type'));
    }

    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    if (!is_null($request->input('user_id'))) {
      $query->where('user_id', (int) $request->input('user_id'));
    }
    if (!is_null($request->input('driver_id'))) {
      $query->where('driver_id', (int) $request->input('driver_id'));
    }
    if (!is_null($request->input('driver'))) {
      $query->where('driver_id', (int) $request->input('driver'));
    }

    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    return \DataTables::eloquent($query->with([
      'user:id,name,email,profile_photo_path',
      'driver' => function ($q) {
        $q->select('id', 'user_id');
      },
      'driver.user:id,name,email,profile_photo_path',
      'transaction' => function ($q) {
        $q->select('id', 'txn_id');
      },
      'parcelCategory:id,name',
      'order:id,order_id',
    ]))
      ->addIndexColumn()
      ->addColumn('customer', fn($ride) => $ride['user']?->name ?? '-')
      ->addColumn('email', fn($ride) => $ride['user']?->email ?? '-')
      ->addColumn('avatar', function ($ride) {
        if ($ride['user'] && $ride['user']->profile_photo_path) {
          return asset('storage/' . $ride['user']->profile_photo_path);
        }
        return '';
      })
      ->addColumn('driver', fn($ride) => $ride['driver']['user']?->name ?? '-')
      ->addColumn('driver_email', fn($ride) => $ride['driver']['user']?->email ?? '-')
      ->addColumn('driverId', fn($ride) => $ride['driver']['user']?->id ?? '')
      ->addColumn('driver_avatar', function ($ride) {
        if ($ride['driver']['user']->profile_photo_path) {
          return asset('storage/' . $ride['driver']['user']->profile_photo_path);
        }
      })
      ->addColumn('currency', fn($ride) => $ride->currency ?? $currency)
      ->addColumn('txn', fn($ride) => optional($ride->transaction)->txn_id ?? $ride->payment_type)
      ->addColumn('orderID', fn($ride) => $ride->order?->order_id ?? '')
      ->addColumn('order_id', fn($ride) => $ride->order?->id ?? '')
      ->addColumn('parcel_category', fn($ride) => $ride->parcelCategory?->name ?? '')
      ->addColumn('user_type', $userType)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($ride) => url($prefix . 'rides/' . $ride->id))
      ->addColumn('status-url', fn() => url($prefix . 'rides/changestatus'))
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }

  public function find($id)
  {
    return Ride::with('user')->find($id);
  }

  //Api's
  public function myRides($perPage = 10)
  {
    $user = auth()->user();
    $query = Ride::with(['driver.user:id,name,profile_photo_path', 'user:id,name,dial_code,phone', 'order.store.user:id,name,dial_code,phone,profile_photo_path'])->where('status', 'Completed')->orderByDesc('created_at');
    if ($user->user_type === 'User') {
      $query->where('user_id', $user->id)->where('type', 'Ride');
    } elseif ($user->user_type === 'Driver') {
      $driverId = $user->driverId ?? null;
      if ($driverId) {
        $query->where('driver_id', $user->driverId);
      } else {
        return collect();
      }
    }
    return $query->paginate($perPage);
  }

  public function findNearestDrivers($lat, $lng, $limit = 2, $vehicleCategoryId, $countryId = 94, $exclude = [])
  {
    if (empty($vehicleCategoryId)) {
      // 1️⃣ Get delivery parent category
      $parentCategory = $parentCategory = VehicleCategory::where('type', 'Delivery')->first();
      // 2️⃣ Get all subcategory IDs (including parent if needed)
      $vehicleCategoryIds = $parentCategory
        ? VehicleCategory::where('parent_id', $parentCategory->id)->pluck('id')->toArray()
        : [];
      $type = 'Delivery';
    } else {
      $vehicleCategoryIds = [$vehicleCategoryId];
      $type = 'Ride';
    }
    return Driver::select('drivers.id')->whereHas('vehicle', function ($q) use ($vehicleCategoryIds) {
      $q->whereIn('vehicle_category_id', $vehicleCategoryIds);
    })
      ->where('type', $type)
      ->withDistance($lat, $lng, 'distance')
      ->whereNotIn('id', $exclude)
      ->active($countryId)
      ->orderBy('distance')
      ->limit($limit)
      ->pluck('id')
      ->toArray();
  }

  public function createFromTemp(RideTemp $temp, $driverId)
  {
    $otp = random_int(1000, 9999);
    $ride = Ride::create([
      'user_id' => $temp->user_id,
      'driver_id' => $driverId,
      'pickup_location' => $temp->pickup_location,
      'vehicle_category_id' => $temp->vehicle_category_id,
      'drop_location' => $temp->drop_location,
      'pickup_lat' => $temp->pickup_lat,
      'pickup_lng' => $temp->pickup_lng,
      'drop_lat' => $temp->drop_lat,
      'drop_lng' => $temp->drop_lng,
      'fare' => $temp->fare,
      'currency' => $temp->currency,
      'trip_distance' => $temp->trip_distance,
      'trip_time' => $temp->trip_time,
      'details' => $temp->details,
      'type' => $temp->type,
      'ride_otp' => $otp,
      'sender_name' => $temp->sender_name,
      'receiver_name' => $temp->receiver_name,
      'receiver_phone' => $temp->receiver_phone,
      'parcel_category_id' => $temp->parcel_category_id,
      'package_details' => $temp->package_details,
      'order_id' => $temp->order_id,
      'status' => 'Pending',
    ]);
    if (!empty($temp->order_id)) {
      $grand_total = $temp->fare;
      $driverCommission = Helpers::setting('driver_commission', 'site_setting');
      $platform_commission = $grand_total * $driverCommission / 100;
      $ride->platform_commission = $platform_commission;
    }
    if (isset($temp->driver_lat) && isset($temp->driver_lng)) {
      $ride->driver_ride_accept_lat = $temp->driver_lat;
      $ride->driver_ride_accept_lng = $temp->driver_lng;
    }
    $ride->ride_id = 'TPSRIDE' . $ride->id;
    $ride->save();
    // Mark driver as busy
    $this->driverStatus($ride->driver_id, 1);
    return $ride;
  }

  public function respondToRequest(Request $request)
  {

    $driver = Driver::where('user_id', auth()->id())->first();
    $rideRequest = RideRequest::where('ride_temp_id', $request->ride_temp_id)
      ->where('driver_id', $driver->id)
      ->lockForUpdate()
      ->first();
    // Prevent re-accept if already taken
    if ($rideRequest->status === 'Accepted') {
      return [
        'message' => __('locale.Ride already accepted.'),
        'ride_id' => $rideRequest->ride_id ?? '',
        'status' => 400
      ];
    }
    $rideRequest->update(['status' => $request->status]);

    // Lock the ride_temp row too (to prevent double confirmation)
    $temp = RideTemp::where('id', $request->ride_temp_id)
      ->lockForUpdate()
      ->first();
    if ($request->status === 'Accepted') {
      if ($temp->status === 'Confirmed') {
        return [
          'message' => __('locale.Ride already taken.'),
          'ride_id' => $temp->ride->id ?? '',
          'status' => 400
        ];
      }
      $temp->driver_lat = $request->latitude;
      $temp->driver_lng = $request->longitude;
      $temp->update(['status' => 'Confirmed']);

      $ride = $this->createFromTemp($temp, $driver->id);

      // reject all others
      RideRequest::where('ride_temp_id', $request->ride_temp_id)
        ->where('driver_id', '!=', $driver->id)
        ->update(['status' => 'Rejected']);

      broadcast(new RideAcceptedEvent($ride))->toOthers();

      // ✅ NEW: Notify other drivers to remove the UI
      $otherDrivers = RideRequest::where('ride_temp_id', $request->ride_temp_id)
        ->where('driver_id', '!=', $driver->id)
        ->pluck('driver_id')
        ->toArray();
      foreach ($otherDrivers as $otherDriverId) {
        broadcast(new RideTakenEvent($otherDriverId, $temp->id))->toOthers();
      }

      $result['message'] = __('locale.Ride accepted.');
      $result['ride_id'] = $ride->id;
      $result['status'] = 200;
      return $result;
    }

    // If rejected, check if all requests are handled immediately
    $pendingCount = RideRequest::where('ride_temp_id', $temp->id)
      ->where('status', 'Pending')
      ->count();
    // Dispatch timeout job if still pending
    if ($pendingCount === 0) {
      $driverLimit = Helpers::setting('driver_request_simultaneously', 'site_setting');
      $requestTime = Helpers::setting('request_time_minutes', 'site_setting');
      if ($temp->status === 'Confirmed' || $temp->status === 'Expired') {
        $result['message'] = __('locale.Action already taken');
        $result['ride_id'] = '';
        $result['status'] = 400;
        return $result;
      }
      // Check if specific minutes passed
      if ($temp->created_at->addMinutes((int) $requestTime)->isPast()) {
        $temp->update(['status' => 'Expired']);
        broadcast(new RideExpiredEvent($temp))->toOthers();
        return;
      }
      $alreadyTried = RideRequest::where('ride_temp_id', $temp->id)
        ->pluck('driver_id')->toArray();
      $details = json_decode($temp->details);
      $countryId = $details->country_id ?? 94;
      $nextDrivers = $this->findNearestDrivers($temp->pickup_lat, $temp->pickup_lng, $driverLimit, $temp->vehicle_category_id, $countryId, $alreadyTried);

      if (count($nextDrivers)) {
        $this->rideRequestRepository->createForDrivers($temp, $nextDrivers);
      } else {
        $temp->update(['status' => 'Expired']);
        broadcast(new RideExpiredEvent($temp))->toOthers();
      }
    }

    $result['message'] = __('locale.Response recorded.');
    $result['ride_id'] = '';
    return $result;
  }

  public function rideStart(Request $request, $rideId)
  {
    $otp = $request->otp;
    $ride = Ride::where('id', $rideId)->firstOrFail();
    if ($ride->ride_otp != $otp) {
      return $this->errorResponse([], __('locale.OTP mismatched'), 400);
    }
    $ride->departure_time = now();
    $ride->status = 'Ongoing';
    $ride->save();
    return $this->successResponse([], __('locale.Ride started'), 200);
  }

  public function rideEnd(Request $request, $rideId)
  {
    $idempotencyKey = $request->header('Idempotency-Key') ?? Str::uuid()->toString();
    return DB::transaction(function () use ($rideId, $idempotencyKey, $request) {
      $ride = Ride::where('id', $rideId)->lockForUpdate()->firstOrFail();
      if ($ride->status === 'Completed') {
        return $this->successResponse([], __('locale.Already completed'), 200);
      }
      if ($ride->status === 'Invoiced') {
        return $this->errorResponse([], __('locale.Already completed.'), 400);
      }
      $details = json_decode($ride->details);
      $couponId = $details->coupon_id ?? 0;
      if ($couponId) {
        $discount = $details->estimate_strike_fare ? max($details->estimate_strike_fare - $ride->fare, 0) : 0;
        $couponUsage = new CouponUsage;
        $couponUsage->user_id = $ride->user_id;
        $couponUsage->coupon_id = $couponId;
        $couponUsage->discount = $couponId;
        $couponUsage->save();
      }
      $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $taxRate = 0; //Helpers::setting('tax', 'site_setting'); //tax on ride 0
      $perMinuteRate = DB::table('country_vehicle_category')->where('vehicle_category_id', $ride->vehicle_category_id)->where('country_id', $details->country_id)->first()?->per_minute_waiting_price ?? 0;
      $waiting_time = $request->waiting_time;
      $departure = Carbon::parse($ride->departure_time);
      $now = Carbon::now();
      $total_time = (int) $departure->diffInMinutes($now);
      $trip_time = $total_time;
      $waiting_time_fare = $waiting_time > 0 ? $waiting_time * $perMinuteRate : 0;
      $fare = $ride->fare;
      $tax = ($fare * $taxRate) / (100 + $taxRate);
      $baseFare = $fare - $tax;
      $sub_total = $fare + $tax + $waiting_time_fare;
      $discount = $details->estimate_strike_fare ? max($details->estimate_strike_fare - $ride->fare, 0) : 0;
      $grand_total = $sub_total;  //$sub_total - $discount;
      $driverCommission = Helpers::setting('driver_commission', 'site_setting');
      $platform_commission = $grand_total * $driverCommission / 100;

      $invoice_details = [
        'invoice_no' => 'TPSY' . $ride->id,
        'invoice_date' => date('Y-m-d H:i:s'),
        'old_fare' => $ride->fare,
        'ride_cost' => round($baseFare, 2),
        'taxes' => round($tax, 2),
        'waiting_charges' => round($waiting_time_fare, 2),
        'sub_total' => round($sub_total, 2),
        'discount' => round($discount, 2),
        'grand_total' => round($grand_total, 2),
        'currency' => $ride->currency ?? $currency,
      ];
      // Mark ride completed
      $ride->status = 'Invoiced';
      $ride->invoice_details = json_encode($invoice_details);
      $ride->trip_time = $trip_time;
      $ride->arrived_time = now();
      $ride->fare = round($grand_total, 2);
      $ride->platform_commission = round($platform_commission, 2);
      $ride->save();
      try {
        broadcast(new RideEndEvent($ride))->toOthers();
      } catch (\Exception $e) {
      }
      return $this->successResponse($ride, __('locale.Ride ended successfully.'), 200);

    });
  }

  public function rideComplete(Request $request, $rideId)
  {
    $walletService = new WalletService;
    $idempotencyKey = $request->header('Idempotency-Key') ?? Str::uuid()->toString();
    return DB::transaction(function () use ($rideId, $walletService, $idempotencyKey) {
      $ride = Ride::where('id', $rideId)->lockForUpdate()->firstOrFail();
      if ($ride->status === 'Completed') {
        return $this->successResponse([], __('locale.Already completed'), 200);
      }
      if ($ride->status != 'Invoiced') {
        return $this->errorResponse([], __('locale.End ride before mark completed.'), 400);
      }
      // Compute amounts
      [$commission, $earning] = Finance::split($ride->fare, $ride->platform_commission);
      // Mark ride completed
      $ride->payment_status = 'Paid';
      $ride->status = 'Completed';
      $ride->save();

      //driver table
      $driver = Driver::find($ride->driver_id);
      $driver->increment('rides_count');

      // Driver wallet
      $driverWallet = $walletService->getOrCreate('driver', $ride->driver_id);

      if ($ride->payment_type === 'online') {

        // System already holds money → credit driver earning

        $walletService->creditEarning($driverWallet, $earning, 'ride', $ride->id, 'online', $idempotencyKey, 'Ride earning (online)');
        // Try settling old dues from this credit

        $walletService->tryAutoSettleDues($driverWallet, 'After online ride earning');
      } else { // cash
        // Driver collected full fare; platform is owed commission
        $walletService->creditEarning($driverWallet, $earning, 'ride', $ride->id, 'cash', $idempotencyKey, 'Ride earning (cash)');
        $walletService->addDue($driverWallet, $commission, 'ride', $ride->id, $idempotencyKey ? $idempotencyKey . '-due' : null, 'Ride commission due (cash)');
        // Optionally block if dues exceed threshold — handled by middleware/policy
      }
      // Mark driver as idle
      $this->driverStatus($ride->driver_id, 0);
      $result = [
        'id' => $ride->id,
        'status' => $ride->status,
        'driver_wallet' => $driverWallet->fresh(['transactions'])->only(['balance', 'pending_dues']),
      ];
      return $this->successResponse($result, __('locale.Ride completed successfully.'), 200);

    });
  }

  public function markDelivered(Request $request, $orderId)
  {
    $walletService = new WalletService;
    $idempotencyKey = $request->header('Idempotency-Key') ?? Str::uuid()->toString();

    return DB::transaction(function () use ($orderId, $walletService, $request, $idempotencyKey) {
      $order = Order::where('id', $orderId)->lockForUpdate()->firstOrFail();
      if ($order->status === 'Delivered') {
        return $this->successResponse([], __('locale.Already delivered'), 200);
      }
      // Find linked delivery ride (if any)
      $ride = Ride::where('order_id', $order->id)->lockForUpdate()->first();
      if ($ride->ride_otp != $request->otp) {
        return $this->errorResponse([], __('locale.OTP Mismatched'), 400);
      }
      // ---- Store settlement (merchandise) ----
      $storeTotal = $order->subtotal + $order->subtotal;
      [$storeCommission, $storeEarning] = Finance::split($storeTotal, (float) $order->platform_commission);
      $storeWallet = $walletService->getOrCreate('store', $order->store_id);

      if ($order->payment_type === 'Online' && $order->payment_status === 'Paid') {
        // Platform holds funds → credit store their earning
        $walletService->creditEarning($storeWallet, $storeEarning, 'order', $order->id, 'online', $idempotencyKey ? $idempotencyKey . '-store' : null, 'Store earning (online order)');
        $walletService->tryAutoSettleDues($storeWallet, 'After online order earning');
      } else {
        // Cash order → store (or driver on behalf) collected cash
        // Credit store with their earning (as balance) and add commission as dues
        $walletService->creditEarning($storeWallet, $storeEarning, 'order', $order->id, 'cash', $idempotencyKey ? $idempotencyKey . '-store' : null, 'Store earning (cash order)');
        /*   $walletService->addDue($storeWallet, $storeCommission, 'order', $order->id, $idempotencyKey ? $idempotencyKey . '-store-due' : null, 'Store commission due (cash order)'); */
      }

      // ---- Driver settlement (delivery) ----
      if ($ride) {
        [$rideCommission, $driverEarning] = Finance::split($ride->fare, (float) $ride->platform_commission ?? 0);
        $driverWallet = $walletService->getOrCreate('driver', $ride->driver_id);

        if ($order->payment_type === 'Online' && $order->payment_status === 'Paid') {
          // Delivery fee paid online (either explicitly or platform absorbs) → credit driver
          $walletService->creditEarning($driverWallet, $driverEarning, 'ride', $ride->id, 'online', $idempotencyKey ? $idempotencyKey . '-driver' : null, 'Delivery earning (online order)');
          $walletService->tryAutoSettleDues($driverWallet, 'After online delivery earning');
        } else {
          // Cash on delivery → driver received cash from customer; owes commission
          $walletService->creditEarning($driverWallet, $driverEarning, 'ride', $ride->id, 'cash', $idempotencyKey ? $idempotencyKey . '-driver' : null, 'Delivery earning (cash order)');
          $walletService->addDue($driverWallet, $rideCommission, 'ride', $ride->id, $idempotencyKey ? $idempotencyKey . '-driver-due' : null, 'Delivery commission due (cash order)');
          $amoutFromDriver = $order->grand_total - $ride->fare;
          $walletService->addDue($driverWallet, $amoutFromDriver, 'ride', $ride->id, $idempotencyKey ? $idempotencyKey . '-driver-due' : null, 'Delivery payment due (cash order)');
        }

        // Mark ride completed if not already
        if ($ride->status !== 'Completed') {
          $ride->status = 'Completed';
          $ride->payment_status = 'Paid';
          if ($order->payment_type === 'Online' && $order->payment_status === 'Paid') {
            $ride->payment_type = 'online';
          }
          $ride->save();
        }
        $this->driverStatus($ride->id, 0);
      }
      // Mark order delivered
      $order->status = 'Delivered';
      if ($order->payment_status == 'Pending') {
        $order->payment_status = 'Paid';
      }

      $order->delivered_at = now();
      $order->save();
      $result = [
        'order_id' => $order->id,
        'order_status' => $order->status,
        'store_wallet' => $storeWallet->fresh()->only(['balance', 'pending_dues']),
        'driver_wallet' => isset($driverWallet) ? $driverWallet->fresh()->only(['balance', 'pending_dues']) : null,
      ];
      return $this->successResponse($result, __('locale.Ride completed successfully.'), 200);
    });

  }

  public function rideCancel(Request $request)
  {
    $auth = User::find(auth()->id());
    $ride = $this->find($request->id);

    if (in_array($ride->status, ['Picked_up', 'Ongoing', 'Invoiced'])) {
      return $this->errorResponse([], __('locale.Ride already started'), 400);
    }
    if ($ride->status === 'Cancelled') {
      return $this->errorResponse([], __('locale.Ride already cancelled'), 400);
    }

    DB::transaction(function () use ($ride, $auth, $request) {

      // Update ride
      $ride->status = 'Cancelled';
      $ride->cancelled_by = $auth->user_type;
      $ride->cancelled_message = $request->cancelled_message;
      $ride->cancel_time = now();
      $ride->save();

      // === Driver actions ===
      if ($auth->user_type == 'Driver') {
        $driverId = $ride->driver_id;
        $dailyCancellations = Ride::where('driver_id', $driverId)
          ->where('cancelled_by', 'Driver')
          ->whereDate('cancel_time', now()->toDateString())
          ->count();

        $monthlyCancellations = Ride::where('driver_id', $driverId)
          ->where('cancelled_by', 'Driver')
          ->where('cancel_time', '>=', now()->subDays(30))
          ->count();

        $driver = Driver::find($driverId);

        if ($dailyCancellations >= 3 && $driver->is_blocked != 2 && $monthlyCancellations < 30) {
          $driver->is_blocked = 1;
          $driver->blocked_until = now()->addDay();
          $driver->save();

          DriverBlockLog::create([
            'driver_id' => $driver->id,
            'block_type' => 1,
            'reason' => 'Daily cancellations >= 3',
            'ride_id' => $ride->id,
            'blocked_at' => now(),
          ]);

          DriverUnblockJob::dispatch($driver->id)->delay(now()->addDay());
        }

        if ($monthlyCancellations >= 30) {
          $driver->is_blocked = 2;
          $driver->save();

          DriverBlockLog::create([
            'driver_id' => $driver->id,
            'block_type' => 2,
            'reason' => 'Monthly cancellations >= 30',
            'ride_id' => null,
            'blocked_at' => now(),
          ]);
        }
      }

      // === User actions ===
      if ($auth->user_type == 'User') {
        $gracePeriod = 2; // minutes
        $driverAssignedAt = $ride->created_at;
        $minutes = $driverAssignedAt->diffInMinutes(now());
        $cancellationFee = $driverFee = 0;

        if ($minutes > $gracePeriod) {
          $driverId = $ride->driver_id;
          $driver = Driver::find($driverId);

          $driverLat = $ride->driver->latitude;
          $driverLng = $ride->driver->longitude;
          $driverRideAcceptLat = $ride->driver_ride_accept_lat;
          $driverRideAcceptLng = $ride->driver_ride_accept_lng;
          $pickupLat = $ride->pickup_lat;
          $pickupLng = $ride->pickup_lng;

          $distance = Helpers::haversine($driverRideAcceptLat, $driverRideAcceptLng, $pickupLat, $pickupLng);
          $rideDetails = json_decode($ride->details);
          $fare = $rideDetails->estimate_strike_fare;

          $cancellationFee = round((($fare / $ride->trip_distance) * $distance), 2);
          $distanceTravelByDriver = Helpers::haversine($driverRideAcceptLat, $driverRideAcceptLng, $driverLat, $driverLng);
          $driverFee = round((($fare / $ride->trip_distance) * $distanceTravelByDriver), 2);

          $walletService = new WalletService;
          $idempotencyKey = Str::uuid()->toString();

          $userWallet = $walletService->getOrCreate('user', $ride->user_id);
          $walletService->addDue($userWallet, $cancellationFee, 'ride', $ride->id, $idempotencyKey, 'Ride cancellation');
          $idempotencyKey1 = Str::uuid()->toString();
          $driverWallet = $walletService->getOrCreate('driver', $driver->user_id);
          $walletService->creditEarning($driverWallet, $driverFee, 'ride', $ride->id, 'system', $idempotencyKey1, 'Ride earning (cancellation)');
        }
      }

      // === Mark driver idle ===
      $this->driverStatus($ride->driver_id, 0);

    }, 5); // 5 = retry attempts in case of deadlocks

    return $this->successResponse([], __('locale.Ride cancelled successfully'), 200);
  }

  public function rideStatus(Request $request)
  {
    $ride = $this->find($request->id);
    $terminalStatuses = ['Completed', 'Cancelled', 'Failed'];
    if (!in_array($ride->status, $terminalStatuses)) {
      $ride->status = $request->status;
      $ride->save();
    }
    return true;
  }

  public function waitingTime(Request $request)
  {
    // Find last waiting record for this ride
    $lastWaiting = RideWaitingTime::where('ride_id', $request->ride_id)
      ->latest('id')
      ->first();

    if (!$lastWaiting || $lastWaiting->end_time !== null) {
      // 🚦 Start a new waiting time entry
      $rideWaiting = new RideWaitingTime;
      $rideWaiting->ride_id = $request->ride_id;
      $rideWaiting->latitude = $request->latitude;
      $rideWaiting->longitude = $request->longitude;
      $rideWaiting->start_time = now();
      $rideWaiting->save();

      $message = __('locale.Ride waiting time started');
    } else {
      // 🕒 End the current waiting session
      $lastWaiting->end_time = now();
      $lastWaiting->seconds = Carbon::parse($lastWaiting->end_time)
        ->diffInSeconds(Carbon::parse($lastWaiting->start_time));
      $lastWaiting->save();

      $message = __('locale.Ride waiting time stopped');
    }

    return $this->successResponse([], $message, 200);
  }

  public function updateLocation(Request $request)
  {
    $driverId = User::find(auth()->id())?->driverId;
    $rideId = $request->ride_id ?? null;
    $lat = $request->lat;
    $lng = $request->lng;

    // 1. Get last location from cache
    $lastLoc = cacheRemember(
      "driver_last_location:$driverId",
      10, // TTL in minutes
      fn() => null,
      ['driver_location']
    );

    // 2. Throttle updates (ignore if < 10 meters)
    if ($lastLoc) {
      $distance = Helpers::haversine($lat, $lng, $lastLoc['lat'], $lastLoc['lng']);
      if ($distance < 0.010) { // 10 meters
        return $this->errorResponse([], __('ignored'), 400);
      }
    }

    // 3. Update last location in cache immediately (fast)
    cacheRemember(
      "driver_last_location:$driverId",
      10,
      fn() => [
        'lat' => $lat,
        'lng' => $lng,
        'ride_id' => $rideId,
        'updated_at' => now()
      ],
      ['driver_location']
    );

    // 4. Dispatch async DB update job
    UpdateDriverLocationJob::dispatch($driverId, $lat, $lng);

    // 5. Broadcast via Reverb WS
    broadcast(new DriverLocationUpdatedEvent($driverId, $lat, $lng, $rideId))->toOthers();
    return true;
  }

  public function lastDriverLocation($rideId)
  {
    $lastLoc = Cache::tags(['driver_location'])->get("driver_last_location:$rideId");
    return $lastLoc;
  }

  public function getRideCustomersDropdown()
  {
    return cacheRemember('dropdown_rides_user', 60, function () {
      return DB::table('rides')
        ->join('users', 'users.id', '=', 'rides.user_id')
        ->select('users.id', 'users.name')
        ->distinct()
        ->orderBy('users.name')
        ->pluck('users.name', 'users.id');
    }, ['dropdown']);
  }

  public function getRideDriversDropdown()
  {
    return cacheRemember('dropdown_rides_driver', 60, function () {
      return DB::table('rides')
        ->join('drivers', 'drivers.id', '=', 'rides.driver_id')
        ->join('users', 'users.id', '=', 'drivers.user_id')
        ->where('drivers.type', 'Ride')
        ->select('drivers.id', 'users.name')
        ->distinct()
        ->orderBy('users.name')
        ->pluck('users.name', 'drivers.id');
    }, ['dropdown']);
  }

  public function getParcelCustomersDropdown()
  {
    return User::whereExists(function ($q) {
      $q->select(DB::raw(1))
        ->from('rides')
        ->whereColumn('rides.user_id', 'users.id');
    })
      ->orderBy('name')
      ->get(['id', 'name']);
  }

  public function getParcelDriversDropdown()
  {
    return Driver::where('drivers.type', 'Delivery')
      ->whereExists(function ($query) {
        $query->select(DB::raw(1))
          ->from('rides')
          ->whereColumn('rides.driver_id', 'drivers.id');
      })
      ->join('users', 'drivers.user_id', '=', 'users.id')
      ->select('drivers.id', 'users.name')
      ->orderBy('users.name')
      ->get();
  }

  public function assignDriver($orderId, $driverId)
  {
    $order = Order::with([
      'store.user:id,name,current_country_id',
      'transaction'
    ])->findOrFail($orderId);
    $driver = Driver::with('vehicle')->find($driverId);
    $countryId = $order->store->user?->current_country_id ?? 94;
    $address = json_decode($order->address);
    $rideResult = $this->rideService->getDistanceAndTime($order->latitude, $order->longitude, $address->latitude ?? '', $address->longitude ?? '');
    $rideDistance = $rideResult['distance'];
    $estimated_time = $rideResult['time'];


    $details = [
      'estimate_strike_fare' => $order->shipping,
      'coupon_id' => $order->coupon_id,
      'payment_type' => $order->payment_type,
      'payment_id' => $order->transaction?->txn_id ?? '',
      'country_id' => $countryId,
    ];
    $otp = random_int(1000, 9999);

    $ride = Ride::create([
      'user_id' => $order->user_id,
      'driver_id' => $driverId,
      'vehicle_category_id' => $driver?->vehicle?->vehicle_category_id ?? 0,
      'pickup_location' => $order->store?->address ?? '',
      'drop_location' => $address->address . ' ' . $address->state . ', ' . $address->city . ', ' . $address->postcode,
      'pickup_lat' => $order->latitude ?? '',
      'pickup_lng' => $order->longitude ?? '',
      'drop_lat' => $address?->latitude ?? '',
      'drop_lng' => $address?->longitude ?? '',
      'trip_distance' => $rideDistance,
      'trip_time' => $estimated_time,
      'fare' => $order->shipping,
      'currency' => $order->currency,
      'details' => !empty($details) ? json_encode($details) : '',
      'type' => 'Delivery',
      'ride_otp' => $otp,
      'sender_name' => '',
      'receiver_name' => '',
      'receiver_phone' => '',
      'parcel_category_id' => null,
      'package_details' => '',
      'order_id' => $orderId,
      'status' => 'Pending',
    ]);

    $ride->ride_id = 'TPSRIDE' . $ride->id;
    $ride->save();
    // Mark driver as busy
    $this->driverStatus($ride->driver_id, 1);
    return $order;
  }

  public function driverStatus($id, $status = 0)
  {
    $driver = Driver::find($id);
    if ($driver) {
      $driver->has_ride = $status;
      $driver->save();
    }
    return;
  }

  public function driverCurrentRide()
  {
    $driver = Driver::where('user_id', auth()->id())->first();
    $ride = Ride::with('user', 'driver')->where('driver_id', $driver->id)->whereNotIn('status', ['Completed', 'Cancelled'])->first();
    if (!empty($ride)) {
      return new RideResource(Ride::with('driver', 'user', 'order.store', 'reviews')->find($ride->id));
      /*   $rideResult = $this->rideService->getDistanceAndTime($driver->latitude, $driver->longitude, $ride->pickup_lat, $ride->pickup_lng);
        $pickupDistance = $rideResult['distance'];
        $pickupTime = $rideResult['time'];
        return [
          'id' => $ride->id,
          'fare' => $ride->currency . $ride->fare,
          'trip_distance' => $ride->trip_distance,
          'trip_time' => $ride->trip_time,
          'pickup_distance' => $pickupDistance,
          'pickup_time' => $pickupTime,
          'pickup_location' => $ride->pickup_location,
          'drop_location' => $ride->drop_location,
          'pickup_lat' => $ride->pickup_lat,
          'pickup_lng' => $ride->pickup_lng,
          'drop_lat' => $ride->drop_lat,
          'drop_lng' => $ride->drop_lng,
          'status' => $ride->status,
          'status_label' => Helpers::getStatusLabel($ride->status, $ride->type),
          // 👇 Add user details here
          'user' => [
            'id' => $ride->user?->id,
            'name' => $ride->user?->name,
            'phone' => $ride->user?->dial_code . $ride->user?->phone,
            'avatar' => $ride->user?->profile_photo_url,
            'rating' => $ride->user?->rating,  // assuming 'rating' column exists in users table
          ],
        ]; */
    } else {
      return false;
    }
  }

  public function userCurrentRide()
  {
    $user = User::find(auth()->id());
    $ride = Ride::with('user', 'driver')->where('user_id', $user->id)->whereNotIn('status', ['Completed', 'Cancelled'])->first();
    if (!empty($ride)) {
      return new RideResource(Ride::with('driver', 'user', 'order.store', 'reviews')->find($ride->id));
    } else {
      return false;
    }
  }
}
