<?php

namespace App\Repositories\Eloquent;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\User;
use App\Models\CartSummary;
use App\Models\Cart;
use App\Models\StatusLog;
use App\Models\Address;
use App\Models\CouponUsage;
use App\Models\Store;
use App\Models\RideTemp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use App\Traits\ApiResponseTrait;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\DB;
use App\Events\OrderPlacedEvent;
use App\Services\RidePriceService;
use App\Repositories\Contracts\OrderRepositoryInterface;
use App\Repositories\Contracts\RideRepositoryInterface;
use App\Repositories\Contracts\RideRequestRepositoryInterface;
use Illuminate\Support\Facades\Mail;
use App\Mail\InvoiceEmail;
use Illuminate\Support\Facades\Log;

class OrderRepository extends BaseRepository implements OrderRepositoryInterface
{
  use ApiResponseTrait;
  protected $rideRepository, $rideRequestRepository, $rideService;
  public function __construct(Order $orderModel, RideRepositoryInterface $rideRepository, RideRequestRepositoryInterface $rideRequestRepository, RidePriceService $rideService)
  {
    parent::__construct($orderModel);
    $this->rideRepository = $rideRepository;
    $this->rideRequestRepository = $rideRequestRepository;
    $this->rideService = $rideService;
  }

  public function allList($request)
  {
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $storeId = $userType == 'Store' && $user->storeId ? $user->storeId : '';
    $canDelete = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('order-delete');
    $canEdit = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('order-edit');
    $canCreate = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('order-create');
    $builder = Order::with('user', 'ride', 'transaction');
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('order_id', 'LIKE', "%{$search}%");
      });
    }
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('id', 'LIKE', "%{$search}%")
          ->orWhere('order_id', 'LIKE', "%{$search}%")
          ->orWhere('payment_status', 'LIKE', "%{$search}%")
          ->orWhere('status', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if (!is_null($request->input('payment_status'))) {
      $query->where('payment_status', $request->input('payment_status'));
    }
    if (!is_null($request->input('payment_type'))) {
      $query->where('payment_type', $request->input('payment_type'));
    }
    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    if (!is_null($request->input('user_id'))) {
      $query->where('user_id', (int) $request->input('user_id'));
    }
    if (!is_null($request->input('driver_id'))) {
      $query->where('driver_id', (int) $request->input('driver_id'));
    }
    if (!is_null($request->input('store_id'))) {
      $query->where('store_id', (int) $request->input('store_id'));
    }
    if (!empty($storeId)) {
      $query->where('store_id', (int) $storeId);
    }
    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('customer', fn($order) => $order['user']?->name ?? '-')
      ->addColumn('email', fn($order) => $order['user']?->email ?? '-')
      ->addColumn('avatar', function ($order) {
        if ($order['user'] && $order['user']->profile_photo_path) {
          return asset('storage/' . $order['user']->profile_photo_path);
        }
        return ''; // or a default avatar
      })
      ->addColumn('is_ride', fn($order) => $order['ride']?->id ? true : false)
      ->addColumn('txn', fn($order) => optional($order->transaction)->txn_id ?? $order->payment_type)
      ->addColumn('user_type', $userType)
      ->addColumn('currency', fn($order) => $order->currency ?? $currency)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('prescription_image', fn($order) => asset('storage/' . $order->prescriptions))
      ->addColumn('delete-url', fn($order) => url($prefix . 'orders/' . $order->id))
      ->addColumn('status-url', fn() => url($prefix . 'orders/changestatus'))

      /*   ->with([
          'recordsTotal' => $builder->count() // ✅ total regardless of filters
        ]) */
      ->make(true);
  }

  public function findAll($id)
  {
    $data = [];
    $data['order'] = Order::with('reviews', 'user', 'store', 'items', 'transaction', 'statusLogs')->find($id);
    $data['performedStatuses'] = StatusLog::where('user_id', $data['order']['id'])->where('trackable_id', $id)->pluck('status')->toArray();
    return $data;
  }

  public function changeStatus($id, $status)
  {
    $order = Order::find($id);
    if ($order) {
      $order->status = $status;
      $order->save();
    }
    if ($status == 'Accepted') {
      $this->requestRide($order->id);
    }
    // Store status log
    $order->statusLogs()->create([
      'status' => $status,
      'user_id' => $order->id, // or null if not tracked
    ]);
    $performedStatuses = StatusLog::where('user_id', $order->id)->pluck('status')->toArray();
    return $performedStatuses;
  }


  public function store(Request $request, $Id = null)
  {
    DB::beginTransaction();
    try {
      $userId = Auth::id();
      $summary = CartSummary::where('user_id', $userId)->first();
      $taxPercent = Helpers::setting('tax', 'site_setting');
      $address = Address::find($summary->address_id);
      $store = Store::with('user')->find($summary->store_id);
      $commissionCharge = $store->commission_charge;
      $serviceTax = $store->service_tax;
      $totalPrice = $summary->total;
      $commissionPrice = $summary->subtotal;
      $platformCommission = 0;
      $serviceCharge = 0;
      if ($store && $store->user) {
        if (!empty($store->user->end_date) && $store->user->end_date > now()) {
          // Active subscription -> only platform commission
          $platformCommission = ($commissionPrice * $commissionCharge) / 100;
          //remove service charges if not need in subscribed store
          $serviceCharge = ($commissionPrice * $serviceTax) / 100;
        } else {
          // Expired or no subscription -> commission + service charge
          $platformCommission = ($commissionPrice * $commissionCharge) / 100;
          $serviceCharge = ($commissionPrice * $serviceTax) / 100;
        }
      }
      $currency = $store->user ? $store->user?->currency : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $totalCharges = $platformCommission + $serviceCharge;
      $tax = ($totalPrice * $taxPercent) / 100;
      $order = $Id ? Order::find($Id) : new Order;
      $order->user_id = $userId;
      $order->store_id = $summary->store_id;
      $order->address = json_encode($address);
      $order->latitude = $store->latitude ?? '';
      $order->longitude = $store->longitude ?? '';
      $order->subtotal = $summary->subtotal;
      $order->grand_total = $summary->subtotal + $summary->delivery_charge + $tax - $summary->coupon_amount;
      $order->tax = $tax;
      $order->discount = $summary->coupon_amount ?? 0;
      $order->currency = $currency;
      $order->coupon_id = $summary->coupon_id ?? 0;
      $order->platform_commission = $totalCharges;
      $order->shipping = $summary->delivery_charge;
      $order->payment_status = 'Pending';
      $order->payment_type = $summary->payment_type ?? 'Cash';
      if ($request->hasFile('prescriptions')) {
        $order->prescriptions = $request->file('prescriptions')->store("prescriptions", 'public');
      }
      $order->status = 'Placed';
      $order->save();
      if ($order->payment_type != 'Online' && !empty($order->coupon_id)) {
        $couponUsage = new CouponUsage;
        $couponUsage->user_id = $order->user_id;
        $couponUsage->coupon_id = $order->coupon_id;
        $couponUsage->discount = $order->discount;
        $couponUsage->save();
      }
      $order->order_id = 'TPSORD' . $order->id;
      $order->save();

      $cartItems = Cart::with('product')->where('user_id', $userId)->get();
      foreach ($cartItems as $item) {
        OrderItem::create([
          'order_id' => $order->id,
          'product_id' => $item->product_id,
          'name' => $item->product?->title ?? '',
          'quantity' => $item->quantity,
          'price' => $item->price,
          'total' => $item->price * $item->quantity,
        ]);
      }
      // 4. Clear the cart
      Cart::where('user_id', $userId)->delete();
      $summary->delete();
      if ($order->payment_type == 'Cash' || $order->payment_type == 'Wallet') {
        $message = "You have received a new order #{$order->order_id} amounting to {$order->currency}" . number_format($order->grand_total, 2);
        Helpers::notifications('Order', 'App\Notifications\Order', $message, $store->user_id, $order->id, 'orders', 'tabler-basket-check');
      }
      broadcast(new OrderPlacedEvent($order))->toOthers();

      DB::commit();
      return $order;
    } catch (\Exception $e) {
      DB::rollBack();
      throw $e;
    }
  }

  public function requestRide($id)
  {
    $order = Order::with([
      'store.user:id,name,current_country_id',
      'transaction'
    ])->findOrFail($id);
    $countryId = $order->store->user?->current_country_id ?? 94;
    $address = json_decode($order->address);
    $rideResult = $this->rideService->getDistanceAndTime($order->latitude, $order->longitude, $address->latitude, $address->longitude);
    $rideDistance = $rideResult['distance'];
    $estimated_time = $rideResult['time'];


    $details = [
      'estimate_strike_fare' => $order->shipping,
      'coupon_id' => $order->coupon_id,
      'payment_type' => $order->payment_type,
      'payment_id' => $order->transaction?->txn_id ?? '',
      'country_id' => $countryId,
    ];
    $temp = RideTemp::create([
      'user_id' => $order->user_id,
      'vehicle_category_id' => 0,
      'pickup_location' => $order->store?->address ?? '',
      'drop_location' => $address->address . ' ' . $address->state . ', ' . $address->city . ', ' . $address->postcode,
      'pickup_lat' => $order->latitude ?? '',
      'pickup_lng' => $order->longitude ?? '',
      'drop_lat' => $address->latitude ?? '',
      'drop_lng' => $address->longitude ?? '',
      'trip_distance' => $rideDistance,
      'trip_time' => $estimated_time,
      'sender_name' => '',
      'receiver_name' => '',
      'receiver_phone' => '',
      'parcel_category_id' => null,
      'package_details' => '',
      'order_id' => $order->id ?? null,
      'type' => 'Delivery',
      'fare' => $order->shipping,
      'currency' => $order->currency,
      'details' => !empty($details) ? json_encode($details) : '',
    ]);
    $driverLimit = Helpers::setting('driver_request_simultaneously', 'site_setting');
    $nearestDrivers = $this->rideRepository->findNearestDrivers($order->latitude, $order->longitude, $driverLimit, '', $countryId, []);
    $this->rideRequestRepository->createForDrivers($temp, $nearestDrivers);
    return $this->successResponse($temp, __('locale.Request sent to drivers.'), 200);
  }


  public function orderHistory(Request $request)
  {
    $user = auth()->user();

    $query = Order::query()
      ->with([
        'store.user:id,name,profile_photo_path',
        'items.product.category:id,name',
        'review:id,reviewable_id,reviewable_type,rating'
      ])
      ->where('user_id', $user->id);

    // 🔍 Search filter (order fields + store.user.name)
    if ($request->filled('search')) {
      $search = $request->search;
      $query->where(function ($q) use ($search) {
        $q->where('order_id', 'like', "%{$search}%")
          ->orWhere('payment_status', 'like', "%{$search}%")
          ->orWhere('payment_type', 'like', "%{$search}%")
          ->orWhere('status', 'like', "%{$search}%")
          ->orWhereHas('store.user', fn($q2) => $q2->where('name', 'like', "%{$search}%"));
      });
    }

    // 🔍 Category filter
    if ($request->filled('category_id')) {
      $categoryId = $request->category_id;
      $query->whereHas('items.product.category', function ($q) use ($categoryId) {
        $q->where('id', $categoryId);
      });
    }

    return $query->paginate(10);
  }


  public function orderDetail(Request $request, $id)
  {
    $user = auth()->user();
    $order = Order::with([
      'store.user:id,name,profile_photo_path',
      'items.product.category:id,name',
      'review:id,reviewable_id,reviewable_type,rating',
      'ride.driver.user:id,name,phone,profile_photo_path,dial_code',
      'transaction',
      'statusLogs'
    ])
      ->where('user_id', $user->id)
      ->findOrFail($id);
    return $order;
  }

  public function orderFeedback($id)
  {
    $user = auth()->user();
    $order = Order::with([
      'store.user:id,name,profile_photo_path',
      'items.product.category:id,name',
      'ride.driver.user:id,name,phone,profile_photo_path,dial_code'
    ])
      ->where('user_id', $user->id)
      ->findOrFail($id);
    return [
      'id' => $order->id,
      'order_number' => $order->order_number,
      'store' => [
        'id' => $order->store->id,
        'name' => $order->store->user->name,
        'address' => $order->store->address,
        'image' => $order->store->user->profile_photo_path ?? asset('assets/img/placeholder.jpg'),
      ],
      'products' => $order->items->map(function ($item) {
        return [
          'name' => $item->product->title,
          'id' => $item->product->id,
        ];
      })->toArray(),
      'driver_id' => $order->ride->driver->id,
      'driver_name' => $order->ride->driver->user->name,
      'phone' => $order->ride->driver->user->dial_code . $order->ride->driver->user->phone,
      'driver_image' => $order->ride->driver->user->profile_photo_path ?? asset('assets/img/placeholder.jpg'),
    ];
  }

  public function invoice(Request $request)
  {
    $to_email = $request->to; // auto user email
    $file = $request->file('file');
    $url = url('/');
    $data = [
      'subject' => $request->subject,
      'body' => $request->message,
      'address' => '',
      'logo' => $url . "assets/img/branding/logo.png",
    ];
    try {
      Mail::to($to_email)->send(new InvoiceEmail($data, $file));
    } catch (\Exception $e) {
      Log::error('Send Invoice Error: ' . $e->getMessage());
      return response()->json(['status' => 500, 'message' => 'Failed to send invoice.'], 500);
    }
    return true;
  }

}
