<?php
namespace App\Repositories\Eloquent;
use App\Models\Installment;
use App\Models\Rental;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use Carbon\Carbon;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\InstallmentRepositoryInterface;

class InstallmentRepository extends BaseRepository implements InstallmentRepositoryInterface
{
  public function __construct(Installment $installmentModel)
  {
    parent::__construct($installmentModel);
  }
  public function allInstallments(Request $request)
  {

    $canDelete = Auth::user()->can('rental-delete');
    $canEdit = Auth::user()->can('rental-edit');
    $canCreate = Auth::user()->can('rental-create');
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $builder = Installment::with([
      'rental.driver.user',
      'rental.car'
    ]);
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('id', 'LIKE', "%{$search}%")
          ->orWhere('amount', 'LIKE', "%{$search}%")
          ->orWhere('paid_amount', 'LIKE', "%{$search}%")
          ->orWhere('status', 'LIKE', "%{$search}%")
          // ✅ Search by driver name or email through nested relation
          ->orWhereHas('rental.driver.user', function ($q2) use ($search) {
            $q2->where('name', 'LIKE', "%{$search}%")
              ->orWhere('email', 'LIKE', "%{$search}%");
          });
      });
    }

    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if ($request->filled('rental_id')) {
      $query->where('rental_id', $request->rental_id);
    }
    if ($request->filled('driver')) {
      $driver = $request->input('driver');
      $query->whereHas('rental.driver.user', function ($q) use ($driver) {
        $q->where('id', $driver);
      });
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('payment_date', [$startDate, $endDate]);
    }

    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('currency', fn($installment) => $installment->rental?->driver?->user?->currency ?? $currency)
      ->addColumn('driver_id', fn($installment) => $installment->rental?->driver?->id ?? '-')
      ->addColumn('driver_name', fn($installment) => $installment->rental?->driver?->user?->name ?? '-')
      ->addColumn('driver_email', fn($installment) => $installment->rental?->driver?->user?->email ?? '-')
      ->addColumn('driverId', fn($installment) => $installment->rental?->driver?->user?->id ?? '')
      ->addColumn('driver_avatar', function ($installment) {
        if ($path = $installment->rental?->driver?->user?->profile_photo_path) {
          return asset('storage/' . $path);
        }
        return null;
      })
      ->addColumn('car_model', fn($installment) => $installment->rental?->car?->vehicle_model ?? '-')
      ->addColumn('car_brand', fn($installment) => $installment->rental?->car?->vehicle_brand ?? '-')
      ->addColumn('car_plate', fn($installment) => $installment->rental?->car?->license_plate ?? '-')
      ->addColumn('carId', fn($installment) => $installment->rental?->car?->id ?? '')
      ->addColumn('car_image', function ($installment) {
        if ($path = $installment->rental?->car?->vehicle_image) {
          return asset('storage/' . $path);
        }
        return null;
      })
      ->addColumn('delete-url', fn($rental) => url('admin/installments/' . $rental->id))
      ->addColumn('status-url', fn() => url('admin/installments/changestatus'))
      ->with([
        'recordsTotal' => $builder->count(), // ✅ total regardless of filters
        'create' => $canCreate // ✅ make it root-level
      ])
      ->make(true);
  }


  public function store(Request $request, $Id = null)
  {
    // 1. Find the rental
    $rental = Rental::findOrFail($request->rental_id);
    // 2. Store installment record
    $installment = $Id ? Installment::findOrFail($Id) : new Installment();
    $installment->rental_id = $rental->id;
    $installment->installment_no = $request->installment_no;
    $installment->amount = $request->amount ?? 0;
    $installment->paid_amount = $request->paid_amount ?? 0;
    $installment->payment_date = $request->payment_date ?? now();
    $installment->due_date = $rental->due_date; // for that specific installment
    $installment->status = 'Paid';
    $installment->save();

    // 3. Update rental calculations
    $paidSum = Installment::where('rental_id', $rental->id)->sum('paid_amount');
    $pendingAmount = $rental->total_amount - ($rental->down_payment + $paidSum);

    $rental->pending_amount = $pendingAmount;

    if ($pendingAmount <= 0) {
      $rental->due_date = null; // no due date if fully paid
      $rental->status = 'Completed';
    } else {
      $installmentsPaid = $rental->no_of_installments - ceil($pendingAmount / $rental->installment_amount);
      $dueDate = Carbon::parse($rental->start_date)->addMonths($installmentsPaid);
      $rental->due_date = $dueDate ?? $rental->due_date; // keep or update next due date
    }

    $rental->save();
    $rental = Rental::with(['driver.user', 'car'])->find($rental->id);
    if ($rental) {
      $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $rental->driver_name = $rental->driver?->user?->name ?? '-';
      $rental->driver_email = $rental->driver?->user?->email ?? '-';
      $rental->driver_avatar = $rental->driver?->user?->profile_photo_path
        ? asset('storage/' . $rental->driver->user->profile_photo_path)
        : null;

      $rental->car_model = $rental->car?->vehicle_model ?? '-';
      $rental->car_brand = $rental->car?->vehicle_brand ?? '-';
      $rental->car_plate = $rental->car?->license_plate ?? '-';
      $rental->car_image = $rental->car?->vehicle_image
        ? asset('storage/' . $rental->car->vehicle_image)
        : null;
      $rental->currency = $rental->driver?->user?->currency ?? $currency;
      // Add installments list
      $rental->installments_list = $rental->installments->map(function ($ins) {
        $paid = (float) $ins->paid_amount;
        $amount = (float) $ins->amount;

        // Determine status: Paid / Partial / Unpaid
        if ($paid >= $amount) {
          $status = 'Paid';
        } elseif ($paid > 0 && $paid < $amount) {
          $status = 'Partial';
        } else {
          $status = 'Unpaid';
        }

        return [
          'id' => $ins->id,
          'due_date' => $ins->due_date, // or payment_date if you prefer
          'amount' => $amount,
          'paid_amount' => $paid,
          'status' => $status,
        ];
      });
    }
    cacheForget("dropdown_installments_driver", ['dropdown']);
    return $rental;
  }

}
