<?php

namespace App\Repositories\Eloquent;
use App\Models\User;
use App\Models\Driver;
use App\Models\Vehicle;
use App\Models\VehicleCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Mail\AccountApproveEmail;
use App\Mail\AdminEmail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Storage;
use App\Services\RidePriceService;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Log;
use App\Repositories\Contracts\DriverRepositoryInterface;
class DriverRepository extends BaseRepository implements DriverRepositoryInterface
{
  protected $rideService;
  public function __construct(User $driverModel, RidePriceService $rideService)
  {
    $this->rideService = $rideService;
    parent::__construct($driverModel);
  }

  public function allList($request)
  {
    $canDelete = Auth::user()->can('driver-delete');
    $canEdit = Auth::user()->can('driver-edit');
    $canCreate = Auth::user()->can('driver-create');
    $walletList = Auth::user()->can('wallet-list');
    $builder = User::with('driver')->where('user_type', 'Driver');
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if ($search = $request->input('search.value')) {
      $query->where(function ($q) use ($search) {
        $q->where('name', 'LIKE', "%{$search}%")
          ->orWhere('email', 'LIKE', "%{$search}%")
          ->orWhere('phone', 'LIKE', "%{$search}%");
      });
    }
    // ✅ Driver type filter
    if (!is_null($request->input('type'))) {
      $query->whereHas('driver', function ($q) use ($request) {
        $q->where('type', $request->input('type'));
      });
    }
    if (!is_null($request->input('blocked'))) {
      $blocked = (int) $request->input('blocked');
      $query->whereHas('driver', function ($q) use ($blocked) {
        if ($blocked === 0) {
          $q->where('is_blocked', 0);
        } elseif ($blocked === 1) {
          $q->whereIn('is_blocked', [1, 2]);
        }
      });
    }
    if (!is_null($request->input('verified'))) {
      $query->where('is_verified', (int) $request->input('verified'));
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', (int) $request->input('status'));
    }
    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('default_currency', $currency)
      ->addColumn('verified', fn($user) => $user->is_verified)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('walletList', $walletList)
      ->addColumn('delete-url', fn($user) => url('admin/users/' . $user->id))
      ->addColumn('status-url', fn() => url('admin/users/changestatus'))
      ->addColumn('driver_id', fn($user) => optional($user->driver)->id)
      ->addColumn('rating', fn($user) => optional($user->driver)->rating)
      ->addColumn('type', fn($user) => optional($user->driver)->type)
      ->addColumn('wallet', fn($user) => optional($user->driver)->wallet)
      ->addColumn('is_blocked', fn($user) => optional($user->driver)->is_blocked)
      ->addColumn('blocked_until', fn($user) => optional($user->driver)->blocked_until)
      ->addColumn('license_expiry', fn($user) => optional($user->driver)->license_expiry)
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }


  public function allMaps(Request $request): array
  {
    $data = [];
    $type = $request->query('type');       // Ride, Delivery
    $status = $request->query('status');   // 0 or 1
    $users = cacheRemember('map_drivers', 600, function () {
      return User::with('driver')->whereHas('driver', function ($query) {
        $query->whereNotNull('latitude');
      })->get();
    }, ['drivers']);
    if ($type) {
      $users = $users->filter(function ($user) use ($type) {
        return $user->driver->type === $type;
      });
    }
    if (!is_null($status)) {
      $users = $users->filter(function ($user) use ($status) {
        return $user->status == $status;
      });
    }

    foreach ($users as $user) {
      $data[] = [
        'id' => $user->id,
        'name' => $user->name,
        'email' => $user->email,
        'phone' => $user->dial_code . $user->phone,
        'profile_photo_path' => $user->profile_photo_path,
        'status' => $user->status,
        'email_verified_at' => $user->email_verified_at,
        'driver_id' => $user['driver']->id,
        'latitude' => $user['driver']->latitude,
        'longitude' => $user['driver']->longitude,
        'is_active' => $user['driver']->is_active,
        'type' => $user['driver']->type,
      ];
    }
    return $data;
  }

  public function dropdown()
  {
    return cacheRemember("dropdown_drivers", 60, function () {
      return Driver::with(['user:id,name,phone,status']) // load only required fields
        ->active()
        ->get(['id', 'user_id']);
    }, ['drivers']);

  }

  public function store(Request $request, $Id = null): array
  {
    $data = [];

    $driver = $Id ? Driver::find($Id) : new Driver;
    foreach (['license_image'] as $field) {
      if ($request->hasFile($field)) {
        if ($Id && $driver->$field) {
          Storage::disk('public')->delete($driver->$field);
        }
        $driver->$field = $request->file($field)->store("drivers/{$field}", 'public');
      }
    }
    $country = DB::table('countries')->find($request->country);
    $state = DB::table('states')->find($request->state);
    $city = DB::table('cities')->find($request->city);
    $driver->dob = $request->dob;
    $driver->gender = $request->gender;
    $driver->type = $request->type;
    $driver->address = $request->address;
    $driver->city = $city->name ?? $request->city;
    $driver->state = $state->name ?? $request->state;
    $driver->country = $country->name ?? $request->country;
    $driver->city_id = $request->city;
    $driver->state_id = $request->state;
    $driver->country_id = $request->country;
    $driver->postcode = $request->postcode;
    $driver->landmark = $request->landmark;
    $driver->license_number = $request->license_number;
    $driver->license_expiry = $request->license_expiry;
    $driver->save();

    $user = $Id ? User::find($driver->user_id) : new User;
    $user->name = $request->name;
    $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
    $user->dial_code_iso = $request->dial_code_iso;
    $user->dial_code = $request->dial_code;
    $user->email = $request->email;
    $user->bank_details = $request->bank_details;
    if (empty($Id)) {
      $user->status = 1;
      $user->user_type = $request->user_type ?? 'Driver';
      $user->email_verified_at = now();
      $user->is_verified = 1;
      $user->password = Hash::make($request->password);
    }
    $user->currency = $country ? $country->currency_symbol : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $user->timezone = $country ? $country->timezone : config('app.timezone', 'UTC');
    $user->save();
    $driver->user_id = $user->id;
    $driver->save();
    $user->referral_code = Helpers::random_strings(6) . $user->id;
    $user->save();

    $vehicle = $Id
      ? Vehicle::where('driver_id', $Id)->first() ?? new Vehicle
      : new Vehicle;
    foreach (['vehicle_image', 'insurance_image', 'road_worthiness_image', 'vehicle_registration_image'] as $field) {
      if ($request->hasFile($field)) {
        if ($Id && $vehicle->$field) {
          Storage::disk('public')->delete($vehicle->$field);
        }
        $vehicle->$field = $request->file($field)->store("vehicles/{$field}", 'public');
      }
    }
    $vehicle->vehicle_category_id = $request->vehicle_category_id;
    $vehicle->vehicle_parent_category_id = $request->vehicle_parent_category_id;
    $vehicle->vehicle_brand_id = $request->vehicle_brand_id;
    $vehicle->vehicle_model_id = $request->vehicle_model_id;
    $vehicle->vehicle_color = $request->vehicle_color;
    $vehicle->vehicle_year = $request->vehicle_year;
    $vehicle->registration_number = $request->registration_number;

    $vehicle->license_plate = $request->license_plate;
    $vehicle->status = 1;
    $vehicle->user_id = $user->id;
    $vehicle->driver_id = $driver->id;
    $vehicle->save();
    if (empty($Id)) {
      $message = 'Your Driver Account on ' . config('app.name') . ' has been created by Admin. Now you can login to your account on ' . config('app.name') . ' App with the below credentials. <br>Following are the credentials for login <br><br> <b>Login ID: </b>' . $user->phone . '<br><b>Password: </b>' . $request->password . '<br><b>url: </b>' . url('');
      $greetings = 'Congratulations';
      $title = "Account Registration " . $user->email;
      $array['view'] = 'emails.accountApprove';
      $array['subject'] = $title;
      $array['name'] = $user->name;
      $array['greetings'] = $greetings;
      $array['content'] = $message;
      try {
        if (!empty($user->email)) {
          Mail::to($user->email)->queue(new AccountApproveEmail($array));
        }
      } catch (\Exception $e) {
      }
    }
    $data['id'] = $user->id;
    return $data;
  }

  public function register(Request $request, $Id = null)
  {
    $user = Auth::user();

    // =========================
    // DRIVER SECTION (DB only)
    // =========================
    $driver = DB::table('drivers')->where('user_id', $user->id)->first();
    $isNew = false;

    $driverData = [];
    // Handle file uploads
    foreach (['license_image'] as $field) {
      if ($request->hasFile($field)) {
        if ($Id) {
          $oldFile = DB::table('drivers')->where('user_id', $user->id)->value($field);
          if ($oldFile) {
            Storage::disk('public')->delete($oldFile);
          }
        }
        $driverData[$field] = $request->file($field)->store("drivers/{$field}", 'public');
      }
    }
    if ($request->has('dob'))
      $driverData['dob'] = $request->dob;
    if ($request->has('gender'))
      $driverData['gender'] = $request->gender;
    if ($request->has('address'))
      $driverData['address'] = $request->address;

    if ($request->has('country')) {
      $country = DB::table('countries')->find($request->country);
      $driverData['country'] = $country->name ?? $request->country;
      $driverData['country_id'] = $request->country;
    }
    if ($request->has('state')) {
      $state = DB::table('states')->find($request->state);
      $driverData['state'] = $state->name ?? $request->state;
      $driverData['state_id'] = $request->state;
    }
    if ($request->has('city')) {
      $city = DB::table('cities')->find($request->city);
      $driverData['city'] = $city->name ?? $request->city;
      $driverData['city_id'] = $request->city;
    }
    if ($request->has('postcode'))
      $driverData['postcode'] = $request->postcode;
    if ($request->has('landmark'))
      $driverData['landmark'] = $request->landmark;

    if ($request->has('license_number'))
      $driverData['license_number'] = $request->license_number;

    if ($request->has('license_expiry')) {
      $isNew = empty($driver?->license_expiry); // first time flag
      $driverData['license_expiry'] = $request->license_expiry;
    }

    $driverData['updated_at'] = now();

    if ($driver) {
      DB::table('drivers')->where('user_id', $user->id)->update($driverData);
    } else {
      $driverData['user_id'] = $user->id;
      $driverData['created_at'] = now();
      DB::table('drivers')->insert($driverData);
    }

    // =========================
    // USER UPDATE
    // =========================
    if ($request->has('name'))
      $user->name = $request->name;
    if ($request->hasFile('image')) {
      if ($user->profile_photo_path) {
        Storage::disk('public')->delete($user->profile_photo_path);
      }
      $user->profile_photo_path = $request->file('image')->store("users", 'public');
    }
    if ($request->has('bank_details'))
      $user->bank_details = $request->bank_details;
    $user->save();

    // =========================
    // VEHICLE SECTION
    // =========================
    if ($request->screen_type == 'vehicle') {
      $vehicle = $Id ? Vehicle::where('driver_id', $driver->id)->first() ?? new Vehicle : new Vehicle;

      $vehicleFields = [
        'vehicle_category_id',
        'vehicle_parent_category_id',
        'vehicle_brand_id',
        'vehicle_model_id',
        'vehicle_color',
        'vehicle_year',
        'registration_number',
        'license_plate'
      ];

      foreach ($vehicleFields as $field) {
        if ($request->has($field))
          $vehicle->$field = $request->$field;
      }

      foreach (['vehicle_image', 'insurance_image', 'road_worthiness_image', 'vehicle_registration_image'] as $field) {
        if ($request->hasFile($field)) {
          if ($Id && $vehicle->$field) {
            Storage::disk('public')->delete($vehicle->$field);
          }
          $vehicle->$field = $request->file($field)->store("vehicles/{$field}", 'public');
        }
      }

      $vehicle->status = 1;
      $vehicle->user_id = $user->id;
      $vehicle->driver_id = $driver->id ?? $vehicle->driver_id;
      $vehicle->save();
    }

    // =========================
    // NOTIFICATION FOR DOCUMENTS
    // =========================
    if ($isNew && $request->screen_type == 'documents') {
      $message = "<p>A driver {$user->name} ({$user->dial_code}{$user->phone}) has completed their profile on <strong>" . config('app.name') . "</strong>.</p><p>Review the account in admin panel.</p>";
      $greetings = 'Hello Admin';
      $title = "Driver Profile Completion " . $user->email;
      $array = [
        'view' => 'emails.adminEmail',
        'subject' => $title,
        'name' => $user->name,
        'greetings' => $greetings,
        'content' => $message
      ];
      try {
        $adminEmail = config('mail.from.address');
        if (!empty($adminEmail)) {
          Mail::to($adminEmail)->queue(new AdminEmail($array));
        }
      } catch (\Exception $e) {
        Log::error('Admin email send failed: ' . $e->getMessage());
      }
    }

    return $user;
  }

  public function changeStatus($id, $status)
  {
    $user = User::find($id);
    if ($user) {
      if ($status == 'unblock') {
        $driver = Driver::where('user_id', $user->id)->first();
        $driver->is_blocked = 0;
        $driver->save();
      } else {
        $user->is_verified = $status == '1' ? 1 : 0;
      }
      $user->save();
    }
    return $user;
  }
  public function toggleActiveStatus()
  {
    $driver = null;
    $user = Auth::user();
    if ($user) {
      $driver = Driver::where('user_id', $user->id)->first();
      $driver->is_active = $driver->is_active ? 0 : 1;
      $driver->save();
    }
    return $driver;
  }


  public function getNearestDrivers($latitude, $longitude, $limit = 50, $type, $vehicleCategoryId)
  {
    $countryId = auth()->user()->current_country_id ?? 94;
    return Driver::select(
      'drivers.id',
      'drivers.user_id',
      'drivers.latitude',
      'drivers.longitude',
      'drivers.is_active',
      'drivers.has_ride',
      'drivers.rating',
      'drivers.wallet',
      'drivers.type'
    )
      ->with(['user:id,name,email,phone,dial_code,dial_code_iso,last_seen,fcm_token,is_verified']) // eager load only required fields
      ->withDistance($latitude, $longitude, 'distance')
      ->active($countryId)
      ->when(!empty($type), function ($query) use ($type) {
        $query->where('drivers.type', $type);
      })
      ->when(!empty($vehicleCategoryId), function ($query) use ($vehicleCategoryId) {
        $query->whereHas('vehicle', function ($q) use ($vehicleCategoryId) {
          $q->where('vehicle_category_id', $vehicleCategoryId);
        });
      })
      ->orderBy('distance', 'asc')
      ->limit($limit)
      ->get();
  }

  public function getVehicleTypes(Request $request)
  {
    $user = auth()->user();
    $countryId = $user->current_country_id ?? 94;
    $pickup_lat = $request->pickup_lat;
    $pickup_lng = $request->pickup_lng;
    $drop_lat = $request->drop_lat;
    $drop_lng = $request->drop_lng;
    $type = $request->type ?? 'Ride';
    $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));

    // Step 1: Get drivers near pickup location with their vehicle category
    $drivers = Driver::select(
      'drivers.latitude',
      'drivers.longitude',
      'vehicles.vehicle_category_id'
    )
      ->join('vehicles', 'vehicles.driver_id', '=', 'drivers.id')
      ->withDistance($pickup_lat, $pickup_lng, 'distance')
      ->where('type', $type)
      ->active($countryId)
      ->get();

    // Step 2: Group drivers by vehicle category & get the nearest driver (not just distance)
    $nearestDrivers = $drivers->groupBy('vehicle_category_id')
      ->mapWithKeys(function ($drivers, $vehicleCategoryId) {
        $nearest = $drivers->sortBy('distance')->first(); // nearest driver
        return [
          $vehicleCategoryId => [
            'distance' => $nearest->distance,
            'latitude' => $nearest->latitude,
            'longitude' => $nearest->longitude,
          ]
        ];
      });
    $pickupLat = $request->pickup_lat;
    $pickupLng = $request->pickup_lng;
    // Step 3: Fetch vehicle category details
    $vehicleCategories = VehicleCategory::whereIn('id', $nearestDrivers->keys())
      ->get()
      ->map(function ($vehicle) use ($pickup_lat, $pickup_lng, $drop_lat, $drop_lng, $currency, $countryId, $nearestDrivers, $type, $pickupLat, $pickupLng) {

        // Get country-specific pricing
        $countryPrice = $vehicle->countryPrices()
          ->where('country_id', $countryId)
          ->first();
        $baseFare = $countryPrice->pivot->base_fare ?? $vehicle->base_fare;
        $perKm = $countryPrice->pivot->per_km_rate ?? $vehicle->per_km;
        $perMinute = $countryPrice->pivot->per_minute_price ?? $vehicle->per_minute;

        // Calculate distance between pickup & drop
        $rideResult = $this->rideService->getDistanceAndTime($pickup_lat, $pickup_lng, $drop_lat, $drop_lng);
        $rideDistance = $rideResult['distance'];
        $estimated_time = $rideResult['time'];

        // Get nearest driver data
        $driverData = $nearestDrivers[$vehicle->id];
        //$driverDistance = $driverData['distance'];
        $driverResult = $this->rideService->getDistanceAndTime($driverData['latitude'], $driverData['longitude'], $pickup_lat, $pickup_lng);
        $estimated_time_vehicle = $driverResult['time'];  // minutes
        $driverDistance = $driverResult['distance'];

        //Estimate Price
        if ($type == 'Delivery') {  //for Parcel
          $estimatePrice = $this->rideService->estimateDeliveryCharge($countryId, $rideDistance);
        } else {
          $estimatePrice = $this->rideService->estimatePrice($baseFare, $perKm, $perMinute, $rideDistance, $estimated_time, $pickupLat, $pickupLng);
        }


        return [
          'id' => $vehicle->id,
          'name' => $vehicle->name . ' ' . $vehicle->parent_name,
          'estimate_fare' => number_format($estimatePrice, 2),
          'estimate_strike_fare' => '',
          'distance_vehicle' => number_format($driverDistance, 2), // 👈 nearest driver distance
          'estimated_time_min_vehicle' => $estimated_time_vehicle,
          'distance' => number_format($rideDistance, 2), // pickup → drop distance
          'estimated_time_min' => $estimated_time,
          'currency' => $currency,
          'vehicle_image' => url('storage/' . $vehicle->image) ?? asset('assets/img/ride.png'),
          'coupon_id' => '',
        ];
      });

    return $vehicleCategories;
  }

  public function activeDrivers($type = 'Delivery')
  {
    return cacheRemember('active_drivers', 30, function () use ($type) {
      return Driver::with(['user:id,name,phone,status,dial_code']) // load only required fields
        ->active()
        ->where('type', $type)
        ->get(['id', 'user_id']);
    }, ['drivers']);

  }

}
