<?php
namespace App\Repositories\Eloquent;
use App\Models\Cart;
use App\Models\Product;
use App\Models\Address;
use App\Models\Store;
use App\Models\CartSummary;
use Illuminate\Http\Request;
use App\Traits\ApiResponseTrait;
use App\Repositories\BaseRepository;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Auth;
use App\Services\CouponService;
use App\Services\RidePriceService;
use Stripe\Stripe;
use Stripe\PaymentMethod;
use Illuminate\Support\Facades\DB;
use App\Repositories\Contracts\CartRepositoryInterface;
class CartRepository extends BaseRepository implements CartRepositoryInterface
{
  use ApiResponseTrait;
  protected $couponService, $rideService;
  public function __construct(Cart $cartModel, CouponService $couponService, RidePriceService $rideService)
  {
    parent::__construct($cartModel);
    $this->couponService = $couponService;
    $this->rideService = $rideService;
    Stripe::setApiKey(config('cashier.secret'));
  }
  public function addCart(Request $request)
  {

    $userId = Auth::id();
    $product = Product::findOrFail($request->product_id);
    $existingCartStore = Cart::where('user_id', $userId)->first();
    if ($existingCartStore && $existingCartStore->product->store_id !== $product->store_id) {
      return $this->errorResponse(
        __('locale.You cannot add products from different stores in the same cart.'),
        400
      );
    }
    if ($existingCartStore) {
      $existingParentId = $existingCartStore->product->category->parent_id ?? null;
      $newParentId = $product->category->parent_id ?? null;
      if ($existingParentId !== $newParentId) {
        return $this->errorResponse(
          __('locale.You cannot add products from different category in the same cart.'),
          400
        );
      }
    }
    if (($request->quantity ?? 1) == 0) {
      Cart::where('user_id', $userId)
        ->where('product_id', $request->product_id)
        ->delete();
    } else {
      // Add or update cart item
      $cartItem = Cart::updateOrCreate(
        ['user_id' => $userId, 'product_id' => $request->product_id],
        [
          'price' => $product->discounted_price ?? $product->price,
        ]
      );
      // Handle quantity separately
      if ($request->type === 'increment') {
        $cartItem->increment('quantity');
      } elseif ($request->type === 'decrement') {
        if ($cartItem->quantity > 1) {
          $cartItem->decrement('quantity');
        } else {
          Cart::where('user_id', $userId)
            ->where('product_id', $request->product_id)
            ->delete();
        }
      } else {
        // If a specific quantity is passed
        $cartItem->update(['quantity' => $request->quantity ?? 1]);
      }
    }
    // Check if cart still has items
    $cartItems = Cart::where('user_id', $userId)->get();
    if ($cartItems->isEmpty()) {
      // ❌ No items → delete summary
      CartSummary::where('user_id', $userId)->delete();
      return $this->successResponse([], __('locale.Cart is now empty.'), 200);
    }

    // Calculate subtotal
    $subtotal = $cartItems->sum(fn($item) => $item->quantity * $item->price);

    // Fetch or create summary
    $store = Store::with('user')->find($product->store_id);
    $currency = $store['user'] ? $store['user']->currency : cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $summary = CartSummary::firstOrNew(['user_id' => $userId]);
    $summary->delivery_charge = $summary->delivery_charge ?? 0;
    $summary->coupon_amount = $summary->coupon_amount ?? 0;
    $summary->store_id = $product->store_id;
    $summary->currency = $currency;

    // Recheck coupon validity if applied
    if (!empty($summary->coupon_id)) {
      $request->coupon_id = $summary->coupon_id;
      $request->store_id = $summary->store_id;

      $data = $this->couponService->applyCoupon($request);

      if ($data['status'] == 400) {
        // ❌ coupon invalid → reset
        $summary->coupon_id = 0;
        $summary->coupon_amount = 0;
      } else {
        // ✅ coupon valid → update discount
        $summary->coupon_amount = $data['discount'] ?? 0;
      }
    }

    // Final totals
    $summary->subtotal = $subtotal;
    $summary->total = $summary->subtotal + $summary->delivery_charge - $summary->coupon_amount;
    $summary->save();

    return $this->successResponse(
      $cartItem,
      __('locale.Product added to cart successfully.'),
      200
    );
  }


  public function list(Request $request)
  {
    $userId = Auth::id();
    $summary = CartSummary::where('user_id', $userId)->first();
    $cartItems = Cart::with(['product.store'])
      ->where('user_id', $userId)
      ->get();
    if ($cartItems->isEmpty()) {
      return false;
    }
    // transform only specific fields you need
    $items = $cartItems->map(function ($item) {
      $currentPrice = $item->product?->price ?? $item->price;
      $subtotal = $item->quantity * $currentPrice;
      // update DB if price changed
      if ($item->price != $currentPrice) {
        $item->update(['price' => $currentPrice]);
      }
      return [
        'id' => $item->id,
        'product_id' => $item->product_id,
        'product_name' => $item->product?->title ?? null,
        'image' => $item->product?->images[0] ? asset('storage/' . $item->product?->images[0]->file) : asset('assets/img/placeholder.jpg'),
        'store_id' => $item->product->store->id ?? null,
        'quantity' => $item->quantity,
        'price' => $currentPrice,
        'subtotal' => $subtotal,
        'prescriptions' => $item->product->prescriptions,
        'is_non_veg' => $item->product->is_non_veg ?? false,
      ];
    });

    // base totals
    $totalQuantity = $cartItems->sum('quantity');
    $subtotal = $cartItems->sum(fn($item) => $item->quantity * $item->price);
    // 👉 Check if summary exists
    if ($summary) {
      $coupon_code = '';
      // 👉 If coupon applied and summary older than 1 hour, revalidate coupon
      if (!empty($summary->coupon_id) && $summary->updated_at->lt(now()->subHour())) {

        $req = new Request([
          'coupon_id' => $summary->coupon_id,
          'store_id' => $summary->store_id,
        ]);
        $data = $this->couponService->applyCoupon($req);
        if ($data['status'] == 400) {
          // ❌ invalid → clear coupon
          $summary->coupon_id = 0;
          $summary->coupon_amount = 0;
        } else {
          // ✅ still valid → update discount
          $summary->coupon_amount = $data['coupon_cost'] ?? 0;
          $coupon_code = $data['coupon_code'] ?? '';
        }

        // recalc total and save
        $summary->subtotal = $subtotal;
        $summary->total = $summary->subtotal + $summary->delivery_charge - $summary->coupon_amount;
        $summary->save();
      }
      $totalPrice = $subtotal + $summary->delivery_charge - $summary->coupon_amount;
      $address = $summary->address_id ? Address::find($summary->address_id) : Address::where('user_id', $userId)->where('is_default', 1)->first();
      $address = !empty($address) ? $address->address . $address->state . ', ' . $address->city . ', ' . $address->postcode : '';
    } else {
      $totalPrice = $subtotal;
      $address = '';
    }
    $coupon = $summary->coupon_id ? DB::table('coupons')->find($summary->coupon_id) : '';
    $type = 'Cash';
    $cardDetails = (object) [];
    if (!empty($summary->payment_id)) {
      $paymentMethod = PaymentMethod::retrieve($summary->payment_id); // your payment_id
      $card = $paymentMethod->card;
      $cardDetails = (object) [
        'id' => $paymentMethod->id,
        'brand' => $card->brand,
        'image' => asset('assets/img/payments/' . $card->brand . '.png'),
        'last4' => $card->last4,
        'exp_month' => $card->exp_month,
        'exp_year' => $card->exp_year,
      ];
    }
    $taxPercent = Helpers::setting('tax', 'site_setting');
    $tax = ($totalPrice * $taxPercent) / 100;
    return [
      'items' => $items,
      'totalQuantity' => $totalQuantity,
      'subtotal' => $subtotal,
      'deliveryCharge' => $summary->delivery_charge ?? '',
      'couponDiscount' => $summary->coupon_amount ?? '',
      'couponId' => $summary->coupon_id ?? 0,
      'tax' => $tax,
      'totalPrice' => $totalPrice + $tax,
      'address' => $address,
      'payment_type' => $summary->payment_type ?? $type,
      'cardDetails' => (is_array($cardDetails) || is_object($cardDetails)) && !empty((array) $cardDetails)
        ? $cardDetails : null,
      'currency' => $summary->currency,
      'coupon_name' => $coupon->code ?? '',
    ];
  }


  public function clear()
  {
    $userId = Auth::id();
    Cart::where('user_id', $userId)->delete();
    return CartSummary::where('user_id', $userId)->delete();
  }

  public function setAddress(Request $request)
  {
    $userId = Auth::id();
    $address = Address::findOrFail($request->address_id);
    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $store = Store::findOrFail($summary->store_id);
    $distanceKm = $this->calculateDistance(
      (float) $store->latitude,
      (float) $store->longitude,
      (float) $address->latitude,
      (float) $address->longitude
    );
    $deliveryCharge = $this->rideService->estimateDeliveryCharge($address->country_id, $distanceKm);
    $summary->address_id = $address->id;
    $summary->delivery_charge = $deliveryCharge;
    $summary->total = $summary->subtotal + $summary->delivery_charge - $summary->coupon_amount;
    $summary->save();
    return $this->successResponse($summary, __('locale.Address set successfully.'), 200);
  }

  public function setPayment(Request $request)
  {
    $userId = Auth::id();
    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $summary->payment_id = $request->payment_id;
    $summary->payment_type = $request->payment_type;
    $summary->save();
    return $this->successResponse($summary, __('locale.Payment method added successfully.'), 200);
  }

  public function calculateDistance($lat1, $lon1, $lat2, $lon2)
  {
    $earth_radius = 6371; // km
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat / 2) * sin($dLat / 2) +
      cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
      sin($dLon / 2) * sin($dLon / 2);
    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    return $earth_radius * $c;
  }



}
