<?php
namespace App\Models;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Builder;
class Store extends BaseModel
{
  protected $fillable = ['user_id', 'address', 'description', 'country_id', 'state_id', 'city_id', 'country', 'state', 'city', 'landmark', 'postcode', 'latitude', 'longitude', 'contact_person_name', 'dial_code_contact_iso', 'dial_code_contact', 'contact_person_phone', 'service_tax', 'commission_charge', 'delivery_time', 'website', 'is_popular', 'is_active', 'is_cod', 'is_online_payment', 'orders_count', 'rating', 'rating_count', 'wallet', 'always_open', 'is_non_veg'];
  // Cast fields to proper type
  protected $casts = [
    'orders_count' => 'integer',
    'rating_count' => 'integer'
  ];
  // Set default attribute values
  protected $attributes = [
    'orders_count' => 0,
    'rating_count' => 0
  ];
  protected $appends = ['store_name'];

  public function scopeActive($query, $countryId = null)
  {
    $query->where('is_active', 1)
      ->whereHas('user', function ($q) {
        $q->where('status', 1)
          ->where('is_verified', 1);
      });
    if ($countryId) {
      $query->where('country_id', $countryId);
    }
    return $query;
  }

  public function scopeMaxDeliveryTime(Builder $query, ?float $maxTime, ?float $lat, ?float $lng): Builder
  {
    if ($lat && $lng) {
      // Interpolate numeric values directly (safe, not user input)
      $haversine = "(6371 * acos(
            LEAST(1,
                COS(RADIANS($lat)) * COS(RADIANS(stores.latitude)) *
                COS(RADIANS(stores.longitude) - RADIANS($lng)) +
                SIN(RADIANS($lat)) * SIN(RADIANS(stores.latitude))
            )
        ))";
      $avgSpeed = 0.5;
      $query->addSelect(DB::raw("
        CEIL(($haversine / $avgSpeed * 1.2) / 5) * 5 AS estimated_time,
        ROUND($haversine, 2) AS distance
    "));

      if ($maxTime) {
        $query->havingRaw("estimated_time <= ?", [$maxTime]);
      }

      $query->orderBy('distance', 'asc');
    }

    return $query;
  }



  public function products()
  {
    return $this->hasMany(Product::class);
  }
  public function user()
  {
    return $this->belongsTo(User::class, 'user_id');
  }

  public function categories()
  {
    return $this->belongsToMany(Category::class, 'category_store');
  }
  public function businessHours()
  {
    return $this->hasMany(BusinessHour::class);
  }
  public function getStoreNameAttribute()
  {
    return $this->user ? $this->user->name : null;
  }
  public function wallet()
  {
    return $this->morphOne(Wallet::class, 'owner');
  }
  public function favorites()
  {
    return $this->morphMany(Favorite::class, 'favoritable');
  }
}
