<?php
namespace App\Listeners;
use App\Traits\JsonResponseTrait;
use App\Events\SubscriptionCreated;
use App\Events\SubscriptionCharged;
use App\Events\SubscriptionCancelled;
use App\Events\SubscriptionUpdated;
use App\Events\SubscriptionPaymentFailed;
use App\Services\SubscriptionService;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Invoice;
class HandleSubscriptionEvents
{
  protected SubscriptionService $service;
  use JsonResponseTrait;
  public function __construct(SubscriptionService $service)
  {
    $this->service = $service;
  }

  public function onCreated(SubscriptionCreated $event)
  {
    $sub = $event->subscription;
    $data = $event->invoiceData;
    // 1) Record transaction
    $this->service->transactions($sub, $data);
    // 2) Audit
    $this->service->audit($sub, 'subscription_created', ['amount' => $sub->amount]);

    //Log::info("Subscription Charged event handled for subscription {$sub->id}");
  }

  public function onCharged(SubscriptionCharged $event)
  {
    $sub = $event->subscription;
    $data = $event->invoiceData;

    Stripe::setApiKey(config('cashier.secret'));
    $invoiceId = $data->id;
    if ($invoiceId) {
      $stripeInvoice = Invoice::retrieve($invoiceId);
      //Log::info("Stripe Invoice: {$stripeInvoice}");
      // 1) Update next billing & status
      $firstLine = data_get($stripeInvoice, 'lines.data.0');
      if ($firstLine && isset($firstLine->period->end)) {
        $nextBillingDate = $firstLine->period->end;
      } else {
        // 2️⃣ Fallback to the invoice’s period_end
        $nextBillingDate = $stripeInvoice->period_end;
      }
      if (!empty($nextBillingDate)) {
        $sub->update([
          'next_billing_date' => now()->setTimestamp($nextBillingDate),
          'status' => $data->status,
        ]);
      }
      // 1) Record transaction
      $this->service->transactions($sub, $stripeInvoice);
      // 2) Audit
      $this->service->audit($sub, 'subscription_charged', ['amount' => $sub->amount]);
    }
    //Log::info("SubscriptionCharged event handled for subscription {$sub->id}");
  }

  public function onUpdated(SubscriptionUpdated $event)
  {
    $subscription = $event->subscription;
    $package = \App\Models\Package::where('id', $subscription->package_id)->first();
    $data = $event->invoiceData;
    $subscription->status = 'due';
    $subscription->amount = $package->price;
    if ($subscription->stripe_subscription_id) {
      try {
        Stripe::setApiKey(config('cashier.secret'));
        $stripeSub = \Stripe\Subscription::retrieve($subscription->stripe_subscription_id);
        \Stripe\Subscription::update($subscription->stripe_subscription_id, [
          'quantity' => $data['license_count'],
          'proration_behavior' => 'create_prorations',
        ]);
        // Immediately generate and finalize the invoice
        $invoice = \Stripe\Invoice::create([
          'customer' => $stripeSub->customer,
          'subscription' => $stripeSub->id,
          'auto_advance' => true,
        ]);
        $invoice = \Stripe\Invoice::retrieve($invoice->id);
        $paidInvoice = $invoice->pay();
      } catch (\Exception $e) {
        return $this->errorResponse($e->getMessage(), 500);
      }
    }
    $subscription->save();
    // $this->service->audit($subscription, 'subscription_payment_updated',['new_license_count' => $data['license_count'],]);
  }

  public function onCancelled(SubscriptionCancelled $event)
  {
    $subscription = $event->subscription;
    // Optionally cancel the Stripe subscription (if Stripe ID exists):contentReference[oaicite:3]{index=3}
    if (!empty($subscription->stripe_subscription_id) && is_null($subscription->canceled_at)) {
      Stripe::setApiKey(config('cashier.secret'));
      try {
        $stripeSubscription = \Stripe\Subscription::retrieve($subscription->stripe_subscription_id);
        if (!empty($stripeSubscription)) {
          $stripeSubscription->cancel([
            'invoice_now' => false, // Don't create an invoice immediately
            'prorate' => false,     // Don't apply proration (i.e., don't adjust billing)
          ]);
        }
      } catch (\Exception $e) {
        Log::info("Failed to cancel Stripe subscription: {$e->getMessage()}");
        return $this->errorResponse("Failed to cancel Stripe subscription: " . $e->getMessage(), 500);
      }
      $subscription->canceled_at = now();
      $subscription->status = 'canceled';
      $subscription->save();
      $this->service->audit($subscription, 'subscription_payment_canceled', ['new_license_count' => 0,]);
    }

  }

  public function onFailed(SubscriptionPaymentFailed $event)
  {
    $sub = $event->subscription;
    $sub->update(['status' => 'payment_failed']);
    $this->service->audit($sub, 'subscription_payment_failed', [
      'error' => $event->error,
    ]);
  }
}
