<?php
namespace App\Http\Resources;
use App\Models\Ride;
use Illuminate\Http\Resources\Json\JsonResource;
use App\Helpers\Helpers;

class UserProfileResource extends JsonResource
{
  public function toArray($request)
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $data = $this;

    return [
      'id' => (int) $data->id,
      'name' => $data->name,
      'user_type' => $data->user_type,
      'email' => $data->email,
      'email_verified' => !empty($data->email_verified_at),
      'dial_code_iso' => $data->dial_code_iso ?? '',
      'dial_code' => $data->dial_code ?? '',
      'phone' => $data->phone ?? '',
      'avatar' => $data->profile_photo_url,
      'fcm_token' => $data->fcm_token ?? '',
      'referral_code' => $data->referral_code ?? '',
      'start_date' => $data->start_date ?? '',
      'end_date' => $data->end_date ?? '',
      'last_seen' => $data->last_seen ?? '',
      'is_verified' => $data->is_verified == 1,
      'created_at' => date('d-m-Y', strtotime($data->created_at)),
      'status' => $data->status == 1 ? 'Enable' : 'Disable',
      'currency' => $data->currency ?? $currency,
      ...($data->user_type === 'User'
        ? [
          'current_country_id' => $data->current_country_id ?? 0,
          'wallet' => $data->wallet ?? '',
          'addresses' => !empty($data['addresses'])
            ? $this->addressesData($data['addresses'])
            : collect([]),
        ]
        : []),

      //has Ride
      $this->mergeWhen(
        $this->user_type === 'User',
        fn() => ['has_ride' => $this->hasRide($this->id)]
      ),

      // Driver data flattened
      $this->mergeWhen(
        $data->user_type === 'Driver' && !empty($data['driver']),
        fn() => $this->driverData($data['driver'])
      ),

      // Store data flattened
      $this->mergeWhen(
        $data->user_type === 'Store' && !empty($data['store']),
        fn() => $this->storeData($data['store'])
      ),
      'token' => $this->when(isset($data->token), $data->token),
    ];
  }

  private function driverData($driver)
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    return [
      'type' => $driver->type,
      'dob' => $driver->dob,
      'gender' => $driver->gender ?? '',
      'address' => $driver->address ?? '',
      'country_id' => $driver->country_id,
      'state' => $driver->state ?? '',
      'state_id' => $driver->state_id,
      'city' => $driver->city ?? '',
      'city_id' => $driver->city_id,
      'postcode' => $driver->postcode ?? '',
      'landmark' => $driver->landmark ?? '',
      'license_number' => $driver->license_number ?? '',
      'license_expiry' => $driver->license_expiry ?? '',
      'license_image' => $driver->license_image ? asset('storage/' . $driver->license_image) : null,
      'rides_count' => $driver->rides_count ?? 0,
      'rating' => $driver->rating ?? '',
      'wallet' => $driver->wallet ?? '',
      'is_active' => $driver->is_active == 1,
      'has_ride' => $driver->has_ride == 1,
      'vehicle' => [
        'vehicle_parent_category_id' => $driver['vehicle']?->vehicle_parent_category_id ?? '',
        'vehicle_category_id' => $driver['vehicle']?->vehicle_category_id ?? '',
        'vehicle_brand_id' => $driver['vehicle']?->vehicle_brand_id ?? '',
        'vehicle_model_id' => $driver['vehicle']?->vehicle_model_id ?? '',
        'vehicle_parent_category' => $driver['vehicle']?->parent_category ?? '',
        'vehicle_category' => $driver['vehicle']?->vehicle_category ?? '',
        'vehicle_brand' => $driver['vehicle']?->vehicle_brand ?? '',
        'vehicle_model' => $driver['vehicle']?->vehicle_model ?? '',
        'vehicle_year' => $driver['vehicle']?->vehicle_year ?? '',
        'vehicle_color' => $driver['vehicle']['color']['name'] ?? '',
        'registration_number' => $driver['vehicle']?->registration_number ?? '',
        'license_plate' => $driver['vehicle']?->license_plate ?? '',
        'vehicle_image' => $driver['vehicle']?->vehicle_image ? asset('storage/' . $driver['vehicle']?->vehicle_image) : '',
        'insurance_image' => $driver['vehicle']?->insurance_image ? asset('storage/' . $driver['vehicle']?->insurance_image) : '',
        'road_worthiness_image' => $driver['vehicle']?->road_worthiness_image ? asset('storage/' . $driver['vehicle']?->road_worthiness_image) : '',
        'vehicle_registration_image' => $driver['vehicle']?->vehicle_registration_image ? asset('storage/' . $driver['vehicle']?->vehicle_registration_image) : '',
      ],
    ];
  }

  private function storeData($store)
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    return [
      'address' => $store->address ?? '',
      'description' => $store->description ?? '',
      'country_id' => $store->country_id,
      'state' => $store->state ?? '',
      'state_id' => $store->state_id,
      'city' => $store->city ?? '',
      'city_id' => $store->city_id,
      'postcode' => $store->postcode ?? '',
      'landmark' => $store->landmark ?? '',
      'contact_person_name' => $store->contact_person_name ?? '',
      'contact_person_phone' => $store->contact_person_phone ?? '',
      'service_tax' => $store->service_tax ?? '',
      'delivery_time' => $store->delivery_time ?? '',
      'website' => $store->website ?? '',
      'is_popular' => $store->is_popular == 1,
      'is_active' => $store->is_active == 1,
      'is_cod' => $store->is_cod == 1,
      'is_online_payment' => $store->is_online_payment == 1,
      'rating' => $store->rating ?? '',
      'wallet' => $store->wallet ?? '',
      'orders_count' => $store->orders_count ?? 0,
      'always_open' => $store->always_open == 1,
      'business_hours' => !empty($store['businessHours'])
        ? $this->businessHours($store['businessHours'])
        : collect([]),
    ];
  }

  private function businessHours($hours)
  {
    return collect($hours)->map(fn($hour) => [
      'day' => $hour->day,
      'is_open' => $hour->status == 1,
      'start_time' => $hour->start_time,
      'end_time' => $hour->end_time,
    ])->values();
  }

  private function addressesData($addresses)
  {
    return collect($addresses)->map(fn($address) => [
      'name' => $address->name,
      'address' => $address->address,
      'country' => $address->country,
      'state' => $address->state,
      'city' => $address->city,
      'country_id' => $address->country_id,
      'state_id' => $address->state_id,
      'city_id' => $address->city_id,
      'postcode' => $address->postcode,
      'type' => $address->type,
      'status' => $address->status == 1 ? 'Enable' : 'Disable',
      'is_default' => $address->is_default == 1,
    ])->values();
  }

  private function hasRide($userId)
  {
    $ride = Ride::where('user_id', $userId)
      ->whereNotIn('status', ['Completed', 'Cancelled', 'Failed'])
      ->first();
    if (!empty($ride)) {
      return true;
    } else {
      return false;
    }
  }

  public function with($request)
  {
    return [
      'success' => true,
      'status' => 200
    ];
  }
}
