<?php
namespace App\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;
use App\Helpers\Helpers;
use App\Models\VehicleCategory;
use App\Models\Review;
use App\Services\RidePriceService;

class RideResource extends JsonResource
{
  public function toArray($request)
  {
    try {
      $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $data = $this;
      if (!empty($data->driver)) {
        $ridePriceService = new RidePriceService();
        $rideResult = $ridePriceService->getDistanceAndTime($data->driver->latitude, $data->driver->longitude, $data->pickup_lat, $data->pickup_lng);
        $pickupDistance = $rideResult['distance'] . ' Km' ?? '';
        $pickupTime = $rideResult['time'] . ' Minutes' ?? '';
      }

      return [
        'id' => (int) $data->id,
        'rideID' => $data->ride_id ?? '',
        'pickup_location' => $data->pickup_location ?? '',
        'drop_location' => $data->drop_location ?? '',
        'pickup_lat' => (float) ($data->pickup_lat ?? 0),
        'pickup_lng' => (float) ($data->pickup_lng ?? 0),
        'drop_lat' => (float) ($data->drop_lat ?? 0),
        'drop_lng' => (float) ($data->drop_lng ?? 0),
        'fare' => $data->fare ?? '',
        'ride_otp' => $data->ride_otp ?? '',
        'currency' => $data->currency ?? $currency,
        'trip_distance' => ($data->trip_distance ?? '') . ' Km',
        'trip_time' => ($data->trip_time ?? '') . ' Minutes',
        'pickup_distance' => $pickupDistance ?? '',
        'pickup_time' => $pickupTime ?? '',
        'payment_type' => $data->payment_type ?? '',
        'status' => $data->status ?? 'Pending',
        'status_label' => Helpers::getStatusLabel($data->status ?? '', $data->type ?? ''),
        'type' => $data->type ?? '',
        'created_at' => $data->created_at ? $data->created_at->format('d-m-Y') : null,

        // Ongoing/Arrived: arriving details
        ...(
          in_array($data->status, ['Ongoing', 'Arrived'])
          ? [
            'arriving_details' => $this->arrivingTime(
              optional($data->driver)->latitude ?? 0,
              optional($data->driver)->longitude ?? 0,
              $data->pickup_lat ?? 0,
              $data->pickup_lng ?? 0
            )
          ]
          : []
        ),

        // Invoice
        ...(
          in_array($data->status, ['Invoiced', 'Completed'])
          ? ['invoice_details' => $this->invoiceDetails(json_decode($data->invoice_details ?? null))]
          : []
        ),

        // Completed: ratings
        ...(
          $data->status === 'Completed'
          ? [
            'rating' => optional($data->reviews)->rating ?? 0,
            'driver_rating' => optional($this->driverReview($data))->rating ?? 0,
          ]
          : []
        ),

        // Parcel info
        ...(
          $data->type === 'Parcel'
          ? [
            'sender_name' => $data->sender_name ?? '',
            'receiver_name' => $data->receiver_name ?? '',
            'receiver_phone' => $data->receiver_phone ?? '',
            'parcel_category_id' => $data->parcel_category_id ?? '',
            'package_details' => $data->package_details ?? '',
          ]
          : []
        ),

        // Delivery info
        ...(
          $data->type === 'Delivery' && $data->order
          ? $this->orderData($data->order)
          : []
        ),

        'cancelled_by' => $data->cancelled_by ?? null,
        'cancelled_message' => $data->cancelled_message ?? null,

        // Driver
        ...($data->driver ? $this->driverData($data->driver) : []),

        // User
        ...($data->user ? $this->userData($data->user) : []),

        // Waiting times
        ...(!empty($data->waitingTimes) ? ['waiting_times' => $this->waitingTime($data->waitingTimes)] : []),
        'is_driver_on_waiting' => $data->status != 'Completed'
          && !empty($data->waitingTimes)
          && collect($data->waitingTimes)->contains(function ($wt) {
            return $wt->start_time && !$wt->end_time;
          }),
      ];

    } catch (\Throwable $e) {
      // Catch and return minimal info to debug
      return [
        'error' => 'RideResource conversion failed',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
      ];
    }
  }

  // --- Driver ---
  private function driverData($driver)
  {
    return [
      'driver' => [
        'id' => optional($driver->user)->id,
        'name' => optional($driver->user)->name,
        'phone' => optional($driver->user)->dial_code . optional($driver->user)->phone,
        'avatar' => optional($driver->user)->profile_photo_url,
        'type' => $driver->type ?? '',
        'dob' => $driver->dob ?? null,
        'gender' => $driver->gender ?? '',
        'rides_count' => $driver->rides_count ?? 0,
        'rating' => $driver->rating ?? 0,
        'rating_count' => $driver->rating_count ?? 0,
        'vehicle_type' => optional($driver->vehicle)->parent_category ?? '',
        'vehicle' => optional($driver->vehicle)->vehicle_model . ' ' . optional($driver->vehicle)->vehicle_category ?? '',
        'vehicle_image' => $this->vehicleData($driver->vehicle),
        'vehicle_color' => optional($driver->vehicle)->color ?? '',
        'vehicle_number' => optional($driver->vehicle)->license_plate ?? '',
      ],
    ];
  }

  // --- User ---
  private function userData($user)
  {
    return [
      'user' => [
        'id' => $user?->id,
        'name' => $user->name ?? '',
        'phone' => ($user->dial_code ?? '') . ($user->phone ?? ''),
        'avatar' => $user->profile_photo_url ?? null,
        'rating' => $user->rating ?? 0,
      ],
    ];
  }

  // --- Order ---
  private function orderData($order)
  {
    $address = json_decode($order->address ?? '{}');
    $store = $order->store;

    return [
      'order' => [
        'orderID' => $order->order_id ?? '',
        'payment_type' => $order->payment_type ?? '',
        'payment_status' => $order->payment_status ?? '',
        'grand_total' => $order->grand_total ?? null,
        'latitude' => (float) ($order->latitude ?? 0),
        'longitude' => (float) ($order->longitude ?? 0),
        'user_name' => $address->name ?? '',
        'address' => $address->address ?? '',
        'store_name' => optional($store->user)->name ?? '',
        'store_phone' => (optional($store->user)->dial_code ?? '') . (optional($store->user)->phone ?? ''),
        'store_address' => $store->address ?? '',
        'store_image' => optional($store->user)->profile_image_path
          ? asset('storage/' . $store->user->profile_image_path)
          : asset('assets/img/placeholder.jpg'),
      ],
    ];
  }

  // --- Vehicle image ---
  private function vehicleData($vehicle)
  {
    if (!$vehicle)
      return asset('assets/img/placeholder.jpg');

    $vehicle_category = VehicleCategory::find($vehicle->id);
    if ($vehicle->vehicle_image) {
      return asset('storage/' . $vehicle->vehicle_image);
    } elseif ($vehicle_category && $vehicle_category->image) {
      return asset('storage/' . $vehicle_category->image);
    }
    return asset('assets/img/placeholder.jpg');
  }

  // --- Arriving time ---
  private function arrivingTime($lat1, $lng1, $lat2, $lng2)
  {
    $ridePriceService = new RidePriceService();
    return $ridePriceService->getDistanceAndTime($lat1, $lng1, $lat2, $lng2);
  }

  // --- Invoice details ---
  private function invoiceDetails($data)
  {
    if (!$data)
      return (object) [];

    return [
      'invoice_no' => $data->invoice_no ?? '',
      'invoice_date' => !empty($data->invoice_date) ? date('M d,Y', strtotime($data->invoice_date)) : null,
      'invoice_time' => !empty($data->invoice_date) ? date('H:i A', strtotime($data->invoice_date)) : null,
      'ride_cost' => isset($data->ride_cost) ? number_format($data->ride_cost, 2) : '0.00',
      'taxes' => isset($data->taxes) ? number_format($data->taxes, 2) : '0.00',
      'waiting_charges' => isset($data->waiting_charges) ? number_format($data->waiting_charges, 2) : '0.00',
      'sub_total' => isset($data->sub_total) ? number_format($data->sub_total, 2) : '0.00',
      'discount' => isset($data->discount) ? number_format($data->discount, 2) : '0.00',
      'grand_total' => isset($data->grand_total) ? number_format($data->grand_total, 2) : '0.00',
    ];
  }

  // --- Driver review ---
  public function driverReview($ride)
  {
    return Review::where('comment_id', $ride->id)
      ->where('reviewable_type', 'user')
      ->where('reviewable_id', $ride->user_id)
      ->first();
  }

  // --- Waiting times ---
  private function waitingTime($waitingTimes)
  {
    return collect($waitingTimes)->map(function ($item) {
      return [
        'id' => $item->id ?? null,
        'latitude' => $item->latitude ?? null,
        'longitude' => $item->longitude ?? null,
        'start_time' => $item->start_time ? $item->start_time->toDateTimeString() : null,
        'end_time' => $item->end_time ? $item->end_time->toDateTimeString() : null,
        'seconds' => $item->seconds ?? 0,
        'time' => Helpers::formatSecondsToMinutes($item->seconds ?? 0),
      ];
    })->toArray();
  }

  public function with($request)
  {
    return [
      'success' => true,
      'status' => 200,
    ];
  }
}
