<?php
namespace App\Http\Requests;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class DeveloperPasswordRequest extends FormRequest
{
  public function authorize()
  {
    return true;
  }

  public function rules()
  {
    return [
      'developer_password' => ['required', 'string'],
    ];
  }

  public function withValidator($validator)
  {
    $validator->after(function ($validator) {
      $key = $this->throttleKey();

      // Check rate limiting
      if (RateLimiter::tooManyAttempts($key, 3)) {
        $seconds = RateLimiter::availableIn($key);
        $validator->errors()->add('developer_password', "Too many attempts. Try again in {$seconds} seconds.");
        return;
      }

      $correctPassword = config('app.DEVELOPER_PASSWORD'); // set in .env
      if ($this->developer_password !== $correctPassword) {
        RateLimiter::hit($key, 120); // lock for 60 seconds per failed attempt
        $validator->errors()->add('developer_password', 'Incorrect Developer Password.');
      } else {
        RateLimiter::clear($key);
      }
    });
  }

  public function throttleKey()
  {
    return strtolower('developer-password|' . $this->ip());
  }
}
