<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Http\Requests\VehicleBrandRequest;
use App\Repositories\Contracts\VehicleBrandRepositoryInterface;
use App\Traits\JsonResponseTrait;
use App\Traits\ImageUploadTrait;
class VehicleBrandController extends Controller
{
  use JsonResponseTrait, ImageUploadTrait;
  protected $vehicleBrandRepository;
  public function __construct(VehicleBrandRepositoryInterface $vehicleBrandRepository)
  {
    $this->middleware('permission:vehicle-list|vehicle-create|vehicle-edit|vehicle-delete', ['only' => ['index', 'show', 'brandListAll']]);
    $this->middleware('permission:vehicle-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:vehicle-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:vehicle-delete', ['only' => ['destroy']]);

    $this->vehicleBrandRepository = $vehicleBrandRepository;
  }

  public function index()
  {
    $allBrands = $this->vehicleBrandRepository->allBrands();
    return view('admin.vehicle_brands.index', compact('allBrands'));
  }

  public function brandListAll()
  {
    $brands = $this->vehicleBrandRepository->all();
    return \DataTables::of($brands)->addIndexColumn()->make(true);
  }

  public function brandDropdown()
  {
    return $this->vehicleBrandRepository->dropdown();

  }

  public function store(VehicleBrandRequest $request)
  {
    $data = $request->validated();
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'brands');
    }
    $brand = $this->vehicleBrandRepository->store($data);
    return $this->successResponse($brand, __('locale.Brand Uploaded Successfully'));
  }

  public function update(VehicleBrandRequest $request, $id)
  {
    $brand = $this->vehicleBrandRepository->update($id, $request->validated());
    if ($request->hasFile('image')) {
      $path = $this->uploadImage($request->file('image'), 'categories', $brand->image);
      $brand->image = $path;
      $brand->save();
    }
    return $this->successResponse($brand, __('locale.Brand Edited Successfully'));
  }

  public function destroy($id)
  {
    try {
      $this->vehicleBrandRepository->delete($id);
      return $this->successResponse($id, __('locale.Brand Deleted Successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse(__('locale.Deletion Failed') . ': ' . $e->getMessage(), 500);
    }
  }

  public function changeStatus()
  {
    $brand = $this->vehicleBrandRepository->toggleStatus(request('id'), request('status'));
    return $this->successResponse($brand, __('locale.Brand Status Changes Successfully'));
  }
}
