<?php

namespace App\Http\Controllers\admin;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use App\Traits\JsonResponseTrait;
use App\Http\Requests\RoleRequest;
class RoleController extends Controller
{
  use JsonResponseTrait;
  function __construct()
  {
    $this->middleware('permission:role-list|role-create|role-edit|role-delete', ['only' => ['index', 'show', 'userList']]);
    $this->middleware('permission:role-create', ['only' => ['create', 'store', 'addUser']]);
    $this->middleware('permission:role-edit', ['only' => ['edit', 'update', 'changeStatus', 'updateUser']]);
    $this->middleware('permission:role-delete', ['only' => ['destroy']]);
  }

  public function index(Request $request)
  {
    $roles = Role::orderBy('id', 'DESC')->get();
    $data = Permission::orderBy('name', 'Asc')->get();
    $res = array();
    foreach ($data as $elem) {
      list($key, $val) = explode("-", $elem['name'], 2);
      if (array_key_exists($key, $res))
        $res[$key][] = $val;
      else
        $res[$key] = array($val);
    }
    $permission = $res;
    $users = User::with(array(
      'Roles' => function ($query) {
        $query->where('name', '!=', 'User');
      }
    ))->get();
    return view('common.app-roles', compact('roles', 'permission', 'users'));
  }

  public function store(Request $request)
  {
    $request->validate([
      'name' => 'required|unique:roles,name',
      'color' => 'required',
      'permissions' => 'required',
    ]);
    $permissions = Permission::whereIn('name', $request->input('permissions'))->pluck('id');
    $role = Role::create(['guard_name' => 'web', 'name' => $request->input('name'), 'color' => $request->input('color')]);
    $role->syncPermissions($permissions);

    return redirect('admin/roles')->with('success', __('locale.Role created successfully'));
  }

  public function edit($id)
  {
    $role = Role::find($id);
    $permission = Permission::orderBy('name', 'Asc')->get();
    $rolePermissions = DB::table("role_has_permissions")->where("role_has_permissions.role_id", $id)
      ->pluck('role_has_permissions.permission_id', 'role_has_permissions.permission_id')
      ->all();
    return view('admin.roles.edit', compact('role', 'permission', 'rolePermissions'));
  }

  public function update(Request $request, $id)
  {

    $this->validate($request, [
      'name' => 'required|unique:roles,name,' . $id,
      'permissions' => 'required',
      'color' => 'required',
    ]);
    $role = Role::find($id);
    $role->name = $request->input('name');
    $role->color = $request->input('color');
    $role->save();
    $permissions = Permission::whereIn('name', $request->input('permissions'))->pluck('id');
    $role->syncPermissions($permissions);
    return redirect('admin/roles')->with('success', __('locale.Role updated successfully'));
  }

  public function destroy($id)
  {
    DB::table("roles")->where('id', $id)->delete();
    $result['message'] = __('locale.Role Deleted Successfully');
    $result['data'] = $id;
    return $result;
  }

  public function userList()
  {
    $users = User::where('status', '!=', 2)->whereHas('roles', function ($query) {
      $query->where('user_type', 'Admin');
    })->get();

    $res['data'] = array();
    foreach ($users as $key => $value) {
      $rla = array();
      $rl = $value->user_type;
      $roles = $value->getRoleNames();
      foreach ($roles as $key => $val) {
        $rla[] = $val;
        $rl = implode(', ', $rla);
      }
      $phone = !empty($value->phone) ? $value->phone : '---';
      array_push(
        $res['data'],
        array(
          'id' => $value->id,
          'full_name' => $value->name,
          'role' => $rl,
          'username' => $value->name,
          'email' => $value->email,
          'password' => $value->password,
          'phone' => $value->phone,
          'dial_code' => $value->dial_code,
          'dial_code_iso' => $value->dial_code_iso,
          'status' => $value->status,
          'avatar' => $value->profile_photo_path,
          'url' => url(''),
          'status-url' => url('admin/users/changestatus'),
          'delete-url' => url('admin/users') . '/' . $value->id
        )
      );
    }
    return Datatables::of($res['data'])->addIndexColumn()->make(true);
  }

  public function addUser(RoleRequest $request)
  {
    $user = new User;
    $user->name = $request->name;
    $user->email = $request->email;
    $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
    $user->dial_code = $request->dial_code;
    $user->dial_code_iso = $request->dial_code_iso;
    $user->password = Hash::make($request->password);
    $user->email_verified_at = now();
    $user->status = 1;
    $user->user_type = 'Admin';
    $user->save();
    $roleId = $request->input('roles');
    $roleName = Role::find($roleId)?->name;
    if ($roleName) {
      $user->assignRole($roleName);
    }
    Session::flash('message', __('locale.Account is created successfully.'));
    return Redirect::to('admin/roles');

  }

  public function updateUser(RoleRequest $request, $id)
  {
    $user = User::findOrFail($id);
    $user->name = $request->name;
    $user->email = $request->email;
    $user->phone = preg_replace('/[\s\-\(\)]+/', '', $request->phone);
    $user->dial_code = $request->dial_code;
    $user->dial_code_iso = $request->dial_code_iso;
    $user->save();
    $roleId = $request->input('roles');
    $roleName = Role::find($roleId)?->name;
    if ($roleName) {
      $user->assignRole($roleName);
    }
    return $this->successResponse($user, __('locale.Account edited successfully.'));
  }

}
