<?php
namespace App\Http\Controllers\admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\DB;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\OrderRepositoryInterface;
use App\Repositories\Contracts\ProductRepositoryInterface;
use App\Http\Requests\InvoiceRequest;
use App\Http\Requests\OrderRequest;
class OrderController extends Controller
{
  use JsonResponseTrait;
  protected $orderRepository;
  protected $productRepository;

  function __construct(OrderRepositoryInterface $orderRepository, ProductRepositoryInterface $productRepository)
  {
    if (auth()->check() && auth()->user()->order_type === 'admin') {
      $this->middleware('permission:order-list|order-create|order-edit|order-delete', ['only' => ['index', 'show', 'orderListAll']]);
      $this->middleware('permission:order-create', ['only' => ['create', 'sendInvoice']]);
      $this->middleware('permission:order-edit', ['only' => ['edit', 'update', 'changeStatus']]);
      $this->middleware('permission:order-delete', ['only' => ['destroy']]);
    }
    $this->orderRepository = $orderRepository;
    $this->productRepository = $productRepository;
  }
  public function index()
  {
    $user = User::find(auth()->id());
    $storeId = $user->user_type == 'Store' && $user->storeId ? $user->storeId : '';
    $orderStats = Order::selectRaw("
        COUNT(*) as totalOrder,
        SUM(CASE WHEN status= 'Delivered' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'Cancelled' THEN 1 ELSE 0 END) as cancelled,
         SUM(CASE WHEN payment_status = 'Pending' THEN 1 ELSE 0 END) as pendingPayments
    ")->when(!empty($storeId), fn($q) => $q->where('store_id', $storeId))
      ->first();
    $customers = cacheRemember('dropdown_orders_user', 60, function () use ($storeId) {
      return DB::table('orders')
        ->when(!empty($storeId), fn($q) => $q->where('store_id', $storeId))
        ->join('users', 'users.id', '=', 'orders.user_id')
        ->select('users.id', 'users.name')
        ->distinct()
        ->orderBy('users.name')
        ->pluck('users.name', 'users.id');
    }, ['dropdown']);

    return view('store.orders.index', [
      'totalOrder' => $orderStats->totalOrder,
      'completed' => $orderStats->completed,
      'cancelled' => $orderStats->cancelled,
      'pendingPayments' => $orderStats->pendingPayments,
      'customers' => $customers,
    ]);
  }

  public function orderListAll(Request $request)
  {
    Helpers::markNotificationsAsRead('Order');
    return $this->orderRepository->allList($request);
  }

  public function show(Request $request, $id)
  {
    $data = $this->orderRepository->findAll($id, []);
    $order = $data['order'];
    $user = User::find(auth()->id());
    $storeId = $user->user_type == 'Store' && $user->storeId ? $user->storeId : '';
    if ($user->user_type == 'Store' && $order->store_id != $storeId) {
      abort(403);
    }
    $performedStatuses = $data['performedStatuses'];
    $prefix = $user->user_type == 'Store' ? 'store/' : 'admin/';
    $breadcrumbs = [
      'Home' => route('admin.dashboard'),
      'Order' => url($prefix . 'orders'),
      $order->order_id => null, // last one is active
    ];
    return view('/store/orders/show', compact('order', 'id', 'performedStatuses', 'breadcrumbs'));
  }

  public function update(OrderRequest $request, int $id)
  {
    $order = $this->orderRepository->store($request, $id);
    return $this->successResponse($order, __('locale.Order Updated Successfully!'));
  }

  public function changeStatus(Request $request, $id)
  {
    $status = $request->post('status');
    $performedStatuses = $this->orderRepository->changeStatus($id, $status);
    return $this->successResponse([$performedStatuses], __('locale.Order Status Changed Successfully'));
  }

  public function sendInvoice(InvoiceRequest $request)
  {
    $this->orderRepository->invoice($request);
    return $this->successResponse([], __('locale.Order Invoice Sent Successfully'));
  }

}
