<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Http\Requests\BannerRequest;
use App\Traits\JsonResponseTrait;
use App\Traits\ImageUploadTrait;
use App\Enums\BannerLinkType;
use App\Repositories\Contracts\BannerRepositoryInterface;

class BannerController extends Controller
{
  use JsonResponseTrait, ImageUploadTrait;
  protected $bannerRepository;
  public function __construct(BannerRepositoryInterface $bannerRepository)
  {
    $this->middleware('permission:banner-list|banner-create|banner-edit|banner-delete', ['only' => ['index', 'show', 'bannerListAll']]);
    $this->middleware('permission:banner-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:banner-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:banner-delete', ['only' => ['destroy']]);
    $this->bannerRepository = $bannerRepository;
  }

  public function index()
  {
    $allBanners = $this->bannerRepository->all();
    return view('admin.banners.index', compact('allBanners'));
  }

  public function show($id)
  {
    $banner = $this->bannerRepository->find($id);
    return view('admin.banners.show', compact('banner'));
  }

  public function edit($id)
  {
    $banner = $this->bannerRepository->find($id);
    return response()->json($banner);
  }

  public function bannerListAll()
  {
    $banners = $this->bannerRepository->all();
    return \DataTables::of($banners)->addIndexColumn()->addColumn('linked_name', function ($banner) {
      return $this->getLinkedName($banner);
    })->make(true);
  }

  public function store(BannerRequest $request)
  {
    $data = $request->validated();
    $data['record_id'] = $this->getRecordId($data);
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'banners');
    }
    $banner = $this->bannerRepository->store($data);
    $banner->linked_name = $this->getLinkedName($banner);
    return $this->successResponse($banner, __('locale.Banner Uploaded Successfully'));
  }

  public function update(BannerRequest $request, $id)
  {
    $data = $request->validated();
    $banner = $this->bannerRepository->find($id);
    $data['record_id'] = $this->getRecordId($data, $banner->record_id);
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'banners', $banner->image);
    }
    $data['id'] = $id;
    $banner = $this->bannerRepository->store($data);
    $banner->linked_name = $this->getLinkedName($banner);
    return $this->successResponse($banner, __('locale.Banner Edited Successfully'));
  }

  private function getRecordId(array $data, $oldRecordId = null): ?int
  {
    return match ($data['link'] ?? null) {
      BannerLinkType::PRODUCT->value => $data['product'] ?? $oldRecordId,
      BannerLinkType::STORE->value => $data['store'] ?? $oldRecordId,
      BannerLinkType::CATEGORY->value => $data['main_category'] ?? $oldRecordId,
      default => null,
    };
  }

  private function getLinkedName($banner): string
  {
    return match ($banner->link) {
      BannerLinkType::STORE->value => \App\Models\Store::find($banner->record_id)?->store_name ?? '-',
      BannerLinkType::PRODUCT->value => \App\Models\Product::find($banner->record_id)?->title ?? '-',
      BannerLinkType::CATEGORY->value => \App\Models\Category::find($banner->record_id)?->name ?? '-',
      BannerLinkType::EXTERNAL_LINK->value => $banner->link_url ?? '-',
      default => '-',
    };
  }

  public function destroy($id)
  {
    try {
      $this->bannerRepository->delete($id);
      return $this->successResponse($id, __('locale.Banner Deleted Successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse(__('locale.Deletion Failed') . ': ' . $e->getMessage(), 500);
    }
  }

  public function changeStatus()
  {
    $banner = $this->bannerRepository->toggleStatus(request('id'), request('status'));
    return $this->successResponse($banner, __('locale.Banner Status Changes Successfully'));
  }
}
