<?php
namespace App\Http\Controllers;
use App\Traits\JsonResponseTrait;
use Illuminate\Http\Request;
use App\Models\Subscription;
use Stripe\Stripe;
use Stripe\Webhook;
use App\Events\SubscriptionCharged;
use App\Events\SubscriptionPaymentFailed;
use App\Services\SubscriptionService;
use Illuminate\Support\Facades\Log;
use Stripe\Exception\SignatureVerificationException;
use UnexpectedValueException;
class StripeWebhookController extends Controller
{
  protected SubscriptionService $service;
  use JsonResponseTrait;
  public function __construct(SubscriptionService $service)
  {
    $this->service = $service;
  }

  public function handleWebhook(Request $request)
  {
    Stripe::setApiKey(config('cashier.secret'));
    $payload = $request->getContent();
    $sigHeader = $request->server('HTTP_STRIPE_SIGNATURE');
    try {
      $event = Webhook::constructEvent(
        $payload,
        $sigHeader,
        config('cashier.webhook.secret')
      );
    } catch (UnexpectedValueException $e) {
      Log::info('[StripeWebhook] Invalid payload: ' . $e->getMessage());
      return response('Invalid payload', 400);
    } catch (SignatureVerificationException $e) {
      Log::info('[StripeWebhook] Invalid signature: ' . $e->getMessage());
      return response('Invalid signature', 400);
    }

    // Handle different event types
    switch ($event->type) {
      case 'invoice.finalized':
        $invoice = $event->data->object;
        //Log::info('Stripe webhook invoice.finalized received: '. $invoice);

        break;

      case 'invoice.paid':
        $invoice = $event->data->object;
        //Log::info('Stripe webhook invoice.paid received: '. $invoice);

        break;

      case 'invoice.update':
        $invoice = $event->data->object;
        //Log::info('Stripe webhook invoice.paid received: '. $invoice);

        break;

      case 'invoice_payment.paid':
        $invoice = $event->data->object;
        //Log::info('Stripe webhook invoice_payment.paid received: '. $invoice);

        break;

      case 'invoice.payment_succeeded':
        $invoice = $event->data->object;
        //Log::info('Stripe webhook invoice.payment_succeeded received: '. $invoice);
        $stripeSubscriptionId = $invoice->subscription ?? data_get($invoice, 'parent.subscription_details.subscription');
        if (!$stripeSubscriptionId) {
          return $this->missingSubscription();
        }
        $sub = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();
        if (!$sub) {
          return $this->missingSubscription();
        }

        event(new SubscriptionCharged($sub, $invoice));
        break;

      case 'invoice.payment_failed':
        $invoice = $event->data->object;
        $stripeSubscriptionId = $invoice->subscription ?? data_get($invoice, 'parent.subscription_details.subscription');
        if (!$stripeSubscriptionId) {
          return $this->missingSubscription();
        }
        $sub = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();
        if (!$sub) {
          return $this->missingSubscription();
        }

        event(new SubscriptionPaymentFailed($sub, 'Stripe reported payment failure.'));
        break;

      case 'invoice.created':
        $invoice = $event->data->object;
        $stripeSubscriptionId = $invoice->subscription ?? data_get($invoice, 'parent.subscription_details.subscription');
        if (!$stripeSubscriptionId) {
          return $this->missingSubscription();
        }
        $sub = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();
        if (!$sub) {
          return $this->missingSubscription();
        }

        $this->service->audit($sub, 'subscription_created', [
          'amount' => $sub->amount,
        ]);
        break;

      case 'customer.subscription.updated':
        //Log::info('Stripe webhook customer.subscription.updated received: '. (array) $event->data->object);
        $invoice = $event->data->object;
        $stripeSubscriptionId = $invoice->subscription ?? data_get($invoice, 'parent.subscription_details.subscription');
        if (!$stripeSubscriptionId) {
          return $this->missingSubscription();
        }

        $sub = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();
        if (!$sub) {
          return $this->missingSubscription();
        }

        event(new SubscriptionCharged($sub, $invoice));
        break;

      default:
        Log::info('Unhandled Stripe event type: ' . $event->type);
    }


    return response('Webhook handled', 200);
  }
  /**
   * webhook for customer.subscription.updated
   */
  public function handleCustomerSubscriptionUpdated($payload)
  {
    //Log::info('Stripe webhook customer.subscription.updated received'. $payload['data']['object']);
    $stripeSub = $payload['data']['object'];

    $subscription = Subscription::where('stripe_subscription_id', $stripeSub['id'])->first();
    if (!$subscription)
      return $this->missingSubscription();
    event(new SubscriptionCharged($subscription, $stripeSub));
    return response('Webhook Handled', 200);
  }

  /**
   * webhook for invoice.payment_succeeded
   */
  public function handleInvoicePaymentSucceeded($payload)
  {
    $invoice = $payload['data']['object'];
    $subscription = Subscription::where('stripe_subscription_id', $invoice['subscription'])->first();
    if (!$subscription) {
      return $this->missingSubscription();
    }
    event(new SubscriptionCharged($subscription, $invoice));
    return response('Handled invoice.payment_succeeded', 200);
  }

  /**
   * Fallback for missing subscription
   */

  protected function missingSubscription()
  {
    return response('Subscription not found', 404);
  }
}
