<?php
namespace App\Http\Controllers\API;
use App\Http\Controllers\Controller;
use App\Models\Wallet;
use Carbon\Carbon;
use App\Traits\ApiResponseTrait;
use App\Helpers\Helpers;
use Illuminate\Http\Request;
use App\Repositories\Contracts\UserRepositoryInterface;
use App\Repositories\Contracts\WithdrawalRequestRepositoryInterface;
use App\Http\Requests\WithdrawalFormRequest;
use App\Http\Resources\WithdrawalRequestCollection;
class WalletController extends Controller
{
  use ApiResponseTrait;
  protected $userRepository;
  protected $withdrawalRequestRepository;
  public function __construct(UserRepositoryInterface $userRepository, WithdrawalRequestRepositoryInterface $withdrawalRequestRepository)
  {
    $this->userRepository = $userRepository;
    $this->withdrawalRequestRepository = $withdrawalRequestRepository;
  }
  public function walletList(Request $request)
  {
    try {
      $user = $this->userRepository->find(auth()->id());
      $userType = strtolower($user->user_type);
      $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      if ($userType === 'user') {
        $ownerId = $user->id;
      } elseif ($userType === 'driver' && $user->driver) {
        $ownerId = $user->driver->id;
      } elseif ($userType === 'store' && $user->store) {
        $ownerId = $user->store->id;
      } else {
        return $this->errorResponse([], __('locale.Owner not found!'), 400);
      }

      $wallet = Wallet::with('transactions')->where('owner_id', $ownerId)
        ->where('owner_type', $userType)
        ->first();
      //dd($wallet);

      if (!$wallet) {
        return $this->successResponse(
          ['balance' => 0, 'currency' => $currency, 'transactions' => []],
          __('locale.Wallet Empty!'),
          200
        );
      }
      // ✅ Get date filters (last 1, 3, 6 months OR custom start_date, end_date)

      $query = $wallet->transactions();
      if ($request->has('month')) {
        $range = $request->get('month'); // values: 1, 3, 6 (months)
        if (in_array($range, [1, 3, 6])) {
          $startDate = now()->subMonths($range)->startOfDay();
          $query->where('created_at', '>=', $startDate);
        }
      }
      if ($request->has('type')) {
        $type = $request->get('type');
        $query->where('order_type', $type);
      }

      if ($request->filled('start_date') && $request->filled('end_date')) {
        $query->whereBetween('created_at', [
          Carbon::parse($request->start_date)->startOfDay(),
          Carbon::parse($request->end_date)->endOfDay(),
        ]);
      }

      // ✅ Paginate (10 items per page)
      $transactions = $query->orderBy('created_at', 'desc')->paginate(10);

      // ✅ Map transaction data
      $transactions->getCollection()->transform(function ($txn) {
        $orderType = strtolower($txn->order_type);
        $image = asset('assets/img/wallet.png');
        $displayId = 'In Wallet';

        switch ($orderType) {
          case 'ride':
            $displayId = 'TPSRIDE' . $txn->order_id;
            $image = asset('assets/img/ride.png');
            break;

          case 'order':
            /*          $order = Order::find($txn->order_id);
                     $displayId = $order ? $order->order_id : 'Order'; */
            $displayId = 'TPSORD' . $txn->order_id;
            $image = asset('assets/img/order.png');
            break;

          case 'settlement':
            $displayId = 'From Settlement';
            break;

          case 'referral':
            $displayId = 'From Referral';
            break;

          case 'wallet':
            $displayId = 'Money Added';
            break;
        }

        return [
          'id' => $txn->id,
          'order_type' => ucfirst($txn->order_type),
          'order_id' => $txn->order_id,
          'display' => $displayId,
          'amount' => $txn->amount,
          'image' => $image,
          'type' => $txn->type,
          'payment_mode' => $txn->payment_mode,
          'status' => $txn->status,
          'description' => $txn->description,
          'created_at' => $txn->created_at->format('d M Y, h:i A'),
        ];
      });

      return $this->successResponse([
        'balance' => $wallet->balance,
        'currency' => $wallet->currency ?? $currency,
        'transactions' => $transactions
      ], __('locale.Wallet List!'), 200);

    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }

  public function withdrawalList(Request $request)
  {
    $withdrawalRequest = $this->withdrawalRequestRepository->myList($request);
    $collection = new WithdrawalRequestCollection($withdrawalRequest);
    $fullArray = $collection->response()->getData(true);
    return $this->successResponse(['data' => $fullArray['data'], 'links' => $fullArray['links'], 'meta' => $fullArray['meta']], __('locale.Data fetched successfully'));
  }

  public function store(WithdrawalFormRequest $request)
  {
    $wallet = $this->withdrawalRequestRepository->store($request, null);
    return $this->successResponse($wallet, __('locale.Withdrawal Request Created Successfully!'));
  }


}
