<?php

namespace App\Http\Controllers\API;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Driver;
use App\Models\EmailDetail;
use App\Http\Requests\LoginRequest;
use App\Http\Requests\ChangePasswordRequest;
use App\Http\Requests\ProfileUpdateRequest;
use App\Http\Requests\UserRequest;
use App\Http\Requests\AddressRequest;
use App\Http\Requests\PhoneEmailRequest;
use App\Http\Requests\OtpRequest;
use Illuminate\Support\Facades\Hash;
use App\Repositories\Contracts\UserRepositoryInterface;
use App\Http\Resources\UserProfileResource;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use App\Mail\AdminEmail;
use Illuminate\Support\Facades\Mail;
use App\Traits\ApiResponseTrait;
use App\Helpers\Helpers;
use Stripe\Customer;
use Stripe\PaymentMethod;
use Stripe\Stripe;
use Stripe\Exception\InvalidRequestException;
class UserController extends Controller
{
  use ApiResponseTrait;
  protected $userRepository;
  public function __construct(UserRepositoryInterface $userRepository)
  {
    $this->userRepository = $userRepository;
    Stripe::setApiKey(config('cashier.secret'));
  }
  public function login(LoginRequest $request)
  {
    try {
      $check = User::where('phone', $request->phone)->where('dial_code', $request->dial_code)->where('user_type', $request->user_type)->first();

      if (!$check) {
        return $this->errorResponse([], __('locale.This phone is not registered with us!'), 400);
      }
      // Account status check
      if ($check->status != 1) {
        return $this->errorResponse([], __("locale.Your Account is deleted or disabled. Please contact admin for more information!"), 400);
      }
      // Email verified check
      /*  if (is_null($check->email_verified_at)) {
         return $this->errorResponse([], __("locale.Please verify your email before login."), 400);
       } */
      // Password check
      if (!Hash::check($request->password, $check->password)) {
        return $this->errorResponse([], __('locale.Invalid Password!'), 400);
      }
      // Logout from all other devices (revoke all tokens)
      $check->tokens()->delete();
      // Success → Update FCM token + return token
      $check->fcm_token = $request->fcm_token;
      $check->save();
      $token = $check->createToken('tokens')->plainTextToken;
      $user = User::find($check->id);
      $user->token = $token;
      $user = new UserProfileResource($user);
      return $this->successLoginResponse($user, $token, __('locale.User login successfully.'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function register(UserRequest $request)
  {
    try {
      $user = $this->userRepository->create($request);
      $token = $user->createToken('tokens')->plainTextToken;
      $user->token = $token;
      $user = new UserProfileResource($user);
      $userType = $user->user_type;
      $url = $userType == 'Driver' ? 'drivers' : 'users';
      $message = "A new {$userType} {$user->name} has registered with the phone {$user->dial_code}{$user->phone}.";
      Helpers::adminNotification("Register {$userType}", 'App\Notifications\User', $message, $user->id, $url, $user->id);
      if ($userType == 'Driver') {
        $message = '<p>A new driver account has been registered on <strong>' . config('app.name') . '</strong> by <strong>' . $user->name . '</strong>.<br>Email: ' + $user->email + '<br>Phone: ' + $user->dial_code + $user->phone + '</p><p>You can review the account and take necessary actions by logging into your admin panel.</p>';
        $greetings = 'Hello Admin';
        $title = "Account Registration " . $user->email;
        $array['view'] = 'emails.adminEmail';
        $array['subject'] = $title;
        $array['name'] = $user->name;
        $array['greetings'] = $greetings;
        $array['content'] = $message;
        try {
          $adminEmail = config('mail.from.address');
          if (!empty($adminEmail)) {
            Mail::to($adminEmail)->queue(new AdminEmail($array));
          }
        } catch (\Exception $e) {
        }
      }
      return $this->successResponse($user, __('locale.Account Registered Successfully!'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function phoneEmailCheck(PhoneEmailRequest $request)
  {
    try {
      return $this->successResponse([], '', 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function changePassword(ChangePasswordRequest $request)
  {
    try {
      $oldpass = $request->old_password;
      $newpass = $request->password;
      $confirmpass = $request->confirm_password;
      if (Hash::check($oldpass, auth('sanctum')->user()->getAuthPassword())) {
        if ($newpass == $confirmpass) {
          User::where('id', auth('sanctum')->id())->update(['password' => Hash::make($newpass)]);

          $message = __('locale.Your password changed successfully!');
          return $this->successResponse([], $message, 200);

        } else {
          $validator = __('locale.New password and Confirm password should match.');
          return $this->errorResponse([], $validator, 400);
        }
      } else {
        $validator = __('locale.Invalid old password entered.');
        return $this->errorResponse([], $validator, 400);
      }
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }

  }
  public function splash(Request $request, $id = null)
  {
    $mode = 'P'; // P=Production, D=Development, U=Update
    $ios_version = '';     // example: latest iOS version (numeric)
    $android_version = '1.0.0'; // example: latest Android version (numeric)
    $code = preg_replace('/\D/', '', $request->code);
    $version = preg_replace('/\D/', '', $request->version);
    $platform = $request->platform;
    $message = '';
    // Check version for update mode
    if ($mode === 'P' && !empty($version) && !empty($code)) {
      $full_version = intval($code . $version);

      if ($platform === 'ios' && $full_version < $ios_version) {
        $mode = 'U';
        $message = __('locale.Please update your app to continue');
      }
      if ($platform === 'android' && $full_version < $android_version) {
        $mode = 'U';
        $message = __('locale.Please update your app to continue');
      }
    }
    if ($mode === 'D') {
      $message = __('locale.The app is on Development mode, Please wait until the app is finalized.');
    }
    $result = [
      'mode' => $mode,
      'site_url' => url('/'),
      'ios_link' => 'https://apps.apple.com',
      'android_link' => 'https://play.google.com',
      'map_key' => '', // fill if needed
    ];

    if (!empty($id)) {
      $user = User::find($id);
      if ($user) {
        $user->last_seen = now();
        $user->save();
        $result['user'] = $user;
        $result['logout'] = in_array($user->status, [0, 2]);
        if ($result['logout']) {
          $message = __('locale.Your account is disabled or deleted by admin');
        }
      } else {
        $result['logout'] = true;
        $message = __('locale.User not found');
      }
    }
    $result['message'] = $message;
    return $this->successResponse($result, $message, 200);
  }
  public function signout()
  {
    $user = request()->user();
    $userType = $user->user_type;
    if ($userType == 'Driver') {
      Driver::where('user_id', $user->id)->update(['is_active' => 0]);
    }
    $user->tokens()->where('id', $user->currentAccessToken()->id)->delete();
    return $this->successResponse([], 'locale.Logout Successfully', 200);
  }
  public function deleteAccount()
  {
    $user = User::find(auth('sanctum')->id());
    $user->status = 2;
    $user->save();
    auth('sanctum')->user()->tokens()->delete();
    return $this->successResponse([], 'locale.Your account is deleted successfully.', 200);
  }
  public function getProfile()
  {
    return new UserProfileResource(User::find(auth('sanctum')->id()));
  }
  public function userDetail($id)
  {
    return new UserProfileResource(User::find($id));
  }

  public function verify($id, $hash, Request $request)
  {
    $user = User::findOrFail($id);
    if (!hash_equals((string) $hash, sha1($user->email))) {
      return $this->errorResponse([], __('locale.Invalid verification link'), 400);
    }
    if (!$user->hasVerifiedEmail()) {
      $user->markEmailAsVerified();
    }
    return $this->successResponse([], __('locale.Email verified successfully'));
  }
  public function resend(Request $request)
  {
    $user = User::where('email', $request->email)->first();
    if ($user) {
      $user->sendEmailVerificationNotification();
      return $this->successResponse([], __('locale.Verification link sent'));
    }
    return $this->errorResponse([], __('locale.Email not found'), 400);
  }

  public function updateProfile(ProfileUpdateRequest $request)
  {
    try {
      $user = User::find(auth('sanctum')->id());
      $user->name = $request->name ?? $user->name;
      $user->email = $request->email ?? $user->email;
      $user->phone = $request->phone ?? $user->phone;
      $user->dial_code = $request->dial_code ?? $user->dial_code;
      $user->dial_code_iso = $request->dial_code_iso ?? $user->dial_code_iso;
      $user->current_country_id = $request->country_id;
      if ($request->hasFile('image')) {
        if ($user->profile_photo_path) {
          Storage::disk('public')->delete($user->profile_photo_path);
        }
        $user->profile_photo_path = $request->file('image')->store("users", 'public');
      }
      $user->save();
      return $this->successResponse([], __('locale.Profile Updated Successfully!'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }

  public function notifications()
  {
    $notify = User::find(auth('sanctum')->id());
    $emails = EmailDetail::with('email')
      ->where('user_id', '=', auth('sanctum')->id())
      ->limit(20)->orderBy('seen', 'ASC')->get();
    $total = count($notify->notifications);   //$notify->unreadNotifications
    $ck = count($emails);
    if ($ck == 0 && $total == 0) {
      return $this->successResponse(['unread' => 0, 'total_count' => 0, 'list' => []], __('No new Notification'), 200);
    } else {
      $result = array();
      $res1 = array();
      foreach ($notify->notifications->sortByDesc('created_at')->slice(0, 50) as $notification) {
        if (!empty($notification->read_at)) {
          $seen = true;
        } else {
          $seen = false;
        }
        $users = User::find($notification['data']['sender_id']);
        array_push($res1, array(
          'id' => $notification->id,
          'type' => $notification['data']['head'],
          'title' => $notification['data']['head'],
          'body' => $notification['data']['body'],
          'application_id' => $notification['data']['request_id'],
          'send_by' => $users->user_type == 'Admin' ? 'Admin' : $users->name,
          'timeago' => Carbon::parse($notification->created_at)->diffForHumans(),
          'dateTime' => strtotime($notification->created_at),
          'image' => '',
          'seen' => $seen,
        ));

      }
      $res2 = array();
      foreach ($emails as $notification) {

        array_push($res2, array(
          'id' => $notification->id,
          'type' => $notification->email->message_type,
          'title' => $notification->email->subject,
          'body' => strip_tags($notification->email->body),
          'application_id' => $notification->email->id,
          'send_by' => $notification->email->type == 'Admin' ? 'Admin' : $notification['user']->name,
          'timeago' => Carbon::parse($notification->created_at)->diffForHumans(),
          'dateTime' => strtotime($notification->created_at),
          'image' => !empty($notification->email->icon) ? url('storage/Files') . '/' . $notification->email->icon : '',
          'seen' => ($notification->seen == 1) ? true : false,
        ));
      }
      $yourArray = array_merge($res1, $res2);
      array_multisort(array_column($yourArray, "dateTime"), SORT_DESC, $yourArray);
      if ($total == 1) {
        $result['message'] = $total . ' ' . __('Notification');
      } else {
        $result['message'] = $total . ' ' . __('Notifications');
      }
    }
    return $this->successResponse($result, __('Notifications'), 200);
  }
  public function notificationsRead()
  {
    $userUnreadNotification = auth('sanctum')->user()->unreadNotifications;
    if ($userUnreadNotification) {
      $userUnreadNotification->markAsRead();
    }
    return $this->successResponse([], __('Notification read successfully'), 200);
  }
  public function singleNotification($notificationid)
  {
    $notify = User::find(auth('sanctum')->id());
    $seen = EmailDetail::find($notificationid);
    if (!empty($seen)) {
      $seen->seen = 1;
      $seen->save();
    }
    $notification = $notify->notifications()->find($notificationid);
    if ($notification) {
      $notification->markAsRead();
      $users = User::find($notification['data']['sender_id']);
      $res['id'] = $notification->id;
      $res['type'] = $notification['data']['head'];
      $res['title'] = $notification['data']['head'];
      $res['body'] = $notification['data']['body'];
      $res['application_id'] = $notification['data']['request_id'];
      $res['send_by'] = $users->user_type == 'Admin' ? 'Admin' : $users->name;
      $res['timeago'] = Carbon::parse($notification->created_at)->diffForHumans();
      $res['seen'] = 1;
    }
    return $this->successResponse($res, __('Notification read successfully'), 200);
  }

  public function addressList()
  {
    try {
      $address = $this->userRepository->addressList();
      return $this->successResponse($address, __('locale.Address List'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function addAddress(AddressRequest $request)
  {
    try {
      $user = $this->userRepository->address($request);
      if (!empty($request->id)) {
        return $this->successResponse($user, __('locale.Address edited successfully!'), 200);
      }
      return $this->successResponse($user, __('locale.Address added successfully!'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function deleteAddress(Request $request)
  {
    try {
      $user = $this->userRepository->deleteAddress($request->id);
      return $this->successResponse($user, __('locale.Address deleted successfully!'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function setDefaultAddress(Request $request)
  {
    try {
      $user = $this->userRepository->setDefaultAddress($request->id);
      return $this->successResponse($user, __('locale.Address set as default successfully!'), 200);
    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }

  public function paymentMethods()
  {
    $user = $this->userRepository->find(auth()->id());
    if ($user->user_type === 'User') {
      $wallet = $user->wallet ?? 0;
    } elseif ($user->user_type === 'Driver') {
      $wallet = $user->driver?->wallet ?? 0;
    }
    $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    try {
      $paymentMethodsList = []; // default empty
      try {
        $paymentMethods = PaymentMethod::all([
          'customer' => $user->stripe_id,
          'type' => 'card',
        ]);
      } catch (InvalidRequestException $e) {
      }
      if (!empty($paymentMethods)) {
        foreach ($paymentMethods as $card) {
          $paymentMethodsList[] = [
            'image' => asset('assets/img/payments/' . strtolower($card->card->brand) . '.png'),
            'brand' => ucfirst($card->card->brand),
            'last4' => $card->card->last4,
            'is_default' => $card->id === $user->default_payment_method,
            'name' => $card->billing_details->name,
            'card_number' => '**** **** **** ' . $card->card->last4,
            'expires_month' => $card->card->exp_month,
            'expires_year' => $card->card->exp_year,
            'cvc_check' => $card->card?->checks?->cvc_check === 'pass',
            'phone' => $card->billing_details->phone,
            'email' => $card->billing_details->email,
            'origin' => $card->billing_details?->address?->country,
            'card_id' => $card->id,
          ];
        }
      }
      return $this->successResponse([
        'wallet_balance' => $wallet,
        'currency' => $currency,
        'payment_methods' => $paymentMethodsList
      ], __('locale.Payment Methods'), 200);

    } catch (\Throwable $th) {
      return $this->errorResponse([], $th->getMessage(), 500);
    }
  }
  public function deletePaymentMethod(Request $request)
  {
    try {
      $paymentMethodId = $request->paymentMethodId;
      if (!$paymentMethodId)
        return $this->errorResponse([], __('locale.Payment Method ID is required.'), 400);
      $paymentMethod = PaymentMethod::retrieve($paymentMethodId);
      $paymentMethod->detach(); // Detaches from customer
      return $this->successResponse([], __('locale.Card deleted successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse($e->getMessage(), 500);
    }
  }
  public function setDefaultPaymentMethod(Request $request)
  {
    $user = auth()->user();
    $paymentMethodId = $request->paymentMethodId;
    if (!$paymentMethodId)
      return $this->errorResponse([], __('locale.Payment Method ID is required.'), 400);
    Customer::update($user->stripe_id, [
      'invoice_settings' => [
        'default_payment_method' => $paymentMethodId,
      ],
    ]);
    $user->default_payment_method = $paymentMethodId;
    $user->save();
    return $this->successResponse([], __('locale.Card added as default successfully.'));
  }

  public function addCard(Request $request)
  {
    $user = auth()->user();
    $paymentMethodId = $request->paymentMethodId;
    if (!$paymentMethodId)
      return $this->errorResponse([], __('locale.Payment Method ID is required.'), 400);
    if (!$user->stripe_id) {
      $user->createAsStripeCustomer();
    }
    // Attach to customer
    PaymentMethod::retrieve($paymentMethodId)->attach([
      'customer' => $user->stripe_id,
    ]);

    // Set as default
    Customer::update($user->stripe_id, [
      'invoice_settings' => [
        'default_payment_method' => $paymentMethodId,
      ],
    ]);
    $user->default_payment_method = $paymentMethodId;
    $user->save();
    return $this->successResponse([], __('locale.Card added as default successfully.'));
  }

  public function sendOtp(OtpRequest $request)
  {
    $request->validate([
      'email' => 'required|email',
      'mobile' => 'required|string|min:10',
    ]);
    $user = $this->userRepository->sendOtp($request);
    return $this->successResponse([], __('locale.Email OTP sent. Send mobile OTP via Firebase on client.'));
  }
}
