<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\Ride;
use App\Models\Store;
use App\Models\Country;
use App\Models\CouponUsage;
use App\Models\Driver;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use App\Models\WalletTransaction;
use App\Events\PaymentSuccessEvent;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use App\Services\WalletService;
use Illuminate\Support\Str;
use App\Traits\ApiResponseTrait;
use App\Http\Requests\OrderPaymentRequest;
use App\Http\Requests\RidePaymentRequest;
use App\Http\Requests\WalletPaymentRequest;
use App\Http\Requests\OrderWalletRequest;
use App\Http\Requests\RideWalletRequest;
use App\Repositories\Contracts\UserRepositoryInterface;
use App\Repositories\Contracts\TransactionRepositoryInterface;
class PaymentController extends Controller
{
  use ApiResponseTrait;
  protected $transactionRepository;
  protected $userRepository;
  public function __construct(UserRepositoryInterface $userRepository, TransactionRepositoryInterface $transactionRepository)
  {
    Stripe::setApiKey(config('cashier.secret'));
    $this->transactionRepository = $transactionRepository;
    $this->userRepository = $userRepository;
  }
  /**
   * Pay for an order
   */
  public function payOrder(OrderPaymentRequest $request)
  {
    try {
      $user = Auth::user();
      $currency = Helpers::setting('currency', 'currency');
      $order = Order::find($request->order_id);
      $store = Store::find($order->store_id);
      $country = Country::find($store->country_id);

      // ✅ Create Stripe PaymentIntent
      $paymentIntent = PaymentIntent::create([
        'amount' => round($order->grand_total * 100), // cents
        'currency' => strtolower($country->currency_code ?? $currency),
        'customer' => $user->stripe_id,
        'payment_method' => $request->payment_method_id,
        'off_session' => true,
        'confirm' => true,
      ]);

      // ✅ Handle 3D Secure (if required)
      if ($paymentIntent->status === 'requires_action') {
        return $this->successResponse([
          'requires_action' => true,
          'client_secret' => $paymentIntent->client_secret
        ], '', 200);
      }

      // ✅ Success: update DB
      DB::transaction(function () use ($user, $order, $request, $paymentIntent, $store) {
        $this->transactionRepository->store([
          'user_id' => $user->id,
          'transactionable_type' => 'order',
          'transactionable_id' => $order->id,
          'amount' => $order->grand_total,
          'sub_amount' => $order->subtotal + $order->shipping,
          'tax' => $order->tax,
          'discount' => $order->discount,
          'currency' => $order->currency,
          'txn_id' => $paymentIntent->id,
          'txn_details' => json_encode($paymentIntent),
          'payment_mode' => 'stripe',
          'status' => 'success',
          'contact_details' => json_encode([
            'email' => $user->email ?? '',
            'phone' => $user->phone ?? '',
            'message' => 'Order Payment',
          ]),
        ]);

        if ($order->payment_status == 'Pending') {
          $message = "You have received a new order #{$order->order_id} amounting to {$order->currency}" . number_format($order->grand_total, 2);
          Helpers::notifications('Order', 'App\Notifications\Order', $message, $store->user_id, $order->id, 'orders', 'tabler-basket-check');
        }

        Order::where('id', $request->order_id)
          ->update([
            'payment_status' => 'Paid',
            'payment_type' => 'Online',
          ]);
        if (!empty($order->coupon_id)) {
          $couponUsage = new CouponUsage;
          $couponUsage->user_id = $order->user_id;
          $couponUsage->coupon_id = $order->coupon_id;
          $couponUsage->discount = $order->discount;
          $couponUsage->save();
        }
        if ($order->status == 'Out for Delivery') {
          $ride = Ride::where('order_id', $order->id)->first();
          if ($ride) {
            broadcast(new PaymentSuccessEvent($ride))->toOthers();
          }
        }
      });
      return $this->successResponse([], __('locale.Payment successful'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }
  public function addWallet(WalletPaymentRequest $request)
  {

    try {
      $user = auth()->user();
      $currency = Helpers::setting('currency', 'currency');
      $country = DB::table('countries')->find($user->current_country_id);
      $paymentIntent = PaymentIntent::create([
        'amount' => round($request->amount * 100), // in cents
        'currency' => strtolower($country->currency_code ?? $currency),
        'customer' => $user->stripe_id,
        'payment_method' => $request->payment_method_id,
        'off_session' => true,
        'confirm' => true,
      ]);

      if ($paymentIntent->status === 'requires_action') {
        // 3D Secure required
        return $this->successResponse([
          'requires_action' => true,
          'client_secret' => $paymentIntent->client_secret
        ], '', 200);
      }

      // Payment succeeded, add amount to wallet
      $wallets = new WalletService;
      $idempotencyKey = Str::uuid()->toString();
      $userType = strtolower($user->user_type);
      $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
      $userWallet = $wallets->getOrCreate($userType, $ownerId);
      $wallets->creditEarning($userWallet, $request->amount, 'wallet', $user->id, 'online', $idempotencyKey, 'Money Added to Wallet');
      $wallets->tryAutoSettleDues($userWallet, 'After online wallet payment');
      $lastTxnId = WalletTransaction::where('wallet_id', $userWallet->id)
        ->latest('id')
        ->first()?->id ?? 0;
      DB::transaction(function () use ($user, $lastTxnId, $request, $paymentIntent) {
        $this->transactionRepository->store([
          'user_id' => $user->id,
          'transactionable_type' => 'wallet',
          'transactionable_id' => $lastTxnId,
          'amount' => $request->amount,
          'sub_amount' => $request->amount,
          'tax' => 0,
          'discount' => 0,
          'currency' => $user->currency,
          'txn_id' => $paymentIntent->id,
          'txn_details' => json_encode($paymentIntent),
          'payment_mode' => 'stripe',
          'status' => 'success',
          'contact_details' => json_encode([
            'email' => $user->email ?? '',
            'phone' => $user->phone ?? '',
            'message' => 'Money added to wallet',
          ]),
        ]);
      });
      return $this->successResponse([], __('locale.Wallet credited successfully'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }

  public function payRide(RidePaymentRequest $request)
  {
    try {
      $user = Auth::user();
      $currency = Helpers::setting('currency', 'currency');
      $ride = Ride::find($request->ride_id);
      $driver = Driver::find($ride->driver_id);
      $country = Country::find($driver->country_id);

      // ✅ Create Stripe PaymentIntent
      $paymentIntent = PaymentIntent::create([
        'amount' => round($ride->fare * 100), // cents
        'currency' => strtolower($country->currency_code ?? $currency),
        'customer' => $user->stripe_id,
        'payment_method' => $request->payment_method_id,
        'off_session' => true,
        'confirm' => true,
      ]);

      // ✅ Handle 3D Secure (if required)
      if ($paymentIntent->status === 'requires_action') {
        return $this->successResponse([
          'requires_action' => true,
          'client_secret' => $paymentIntent->client_secret
        ], '', 200);
      }

      // ✅ Success: update DB
      DB::transaction(function () use ($user, $ride, $request, $paymentIntent) {
        $invoice = json_decode($ride->invoice_details);
        $this->transactionRepository->store([
          'user_id' => $user->id,
          'transactionable_type' => 'ride',
          'transactionable_id' => $ride->id,
          'amount' => $ride->fare,
          'sub_amount' => $invoice->sub_total ?? $ride->fare,
          'tax' => 0,
          'discount' => $invoice->discount ?? 0,
          'currency' => $ride->currency,
          'txn_id' => $paymentIntent->id,
          'txn_details' => json_encode($paymentIntent),
          'payment_mode' => 'stripe',
          'status' => 'success',
          'contact_details' => json_encode([
            'email' => $user->email ?? '',
            'phone' => $user->phone ?? '',
            'message' => $ride->type == 'Ride' ? 'Ride Payment' : 'Parcel Ride Payment',
          ]),
        ]);

        Ride::where('id', $request->ride_id)
          ->update([
            'payment_type' => 'online',
            'payment_status' => 'Paid',
          ]);
        broadcast(new PaymentSuccessEvent($ride))->toOthers();
        // coupon usage alrealy in endRide
      });
      return $this->successResponse([], __('locale.Payment successfull'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }


  //wallet payments
  public function walletOrder(OrderWalletRequest $request)
  {
    try {
      $user = Auth::user();
      $order = Order::find($request->order_id);
      $wallets = new WalletService;
      $idempotencyKey = Str::uuid()->toString();
      $userType = strtolower($user->user_type);
      $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
      $userWallet = $wallets->getOrCreate($userType, $ownerId);
      if ($order->grand_total > $userWallet->balance) {
        return $this->errorResponse([], __('locale.Wallet amount is less than the requested payment amount.'), 400);
      }
      Order::where('id', $request->order_id)
        ->update([
          'payment_status' => 'Paid',
          'payment_type' => 'Wallet',
        ]);
      $wallets->walletPayment($userWallet, $order->grand_total, 'wallet payment for order', $idempotencyKey, $request->order_id);
      if (!empty($order->coupon_id)) {
        $couponUsage = new CouponUsage;
        $couponUsage->user_id = $order->user_id;
        $couponUsage->coupon_id = $order->coupon_id;
        $couponUsage->discount = $order->discount;
        $couponUsage->save();
      }
      if ($order->status == 'Out for Delivery') {
        $ride = Ride::where('order_id', $order->id)->first();
        if ($ride) {
          broadcast(new PaymentSuccessEvent($ride))->toOthers();
        }
      }
      return $this->successResponse([], __('locale.Wallet Payment done successfull'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }

  public function walletRide(RideWalletRequest $request)
  {
    try {
      $user = Auth::user();
      $ride = Ride::find($request->ride_id);
      $wallets = new WalletService;
      $idempotencyKey = Str::uuid()->toString();
      $userType = strtolower($user->user_type);
      $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
      $userWallet = $wallets->getOrCreate($userType, $ownerId);
      if ($ride->fare > $userWallet->balance) {
        return $this->errorResponse([], __('locale.Wallet amount is less than the requested payment amount.'), 400);
      }
      Ride::where('id', $request->ride_id)
        ->update([
          'payment_type' => 'wallet',
          'payment_status' => 'Paid',
        ]);
      broadcast(new PaymentSuccessEvent($ride))->toOthers();
      $wallets->walletPayment($userWallet, $request->amount, 'wallet payment for ride', $idempotencyKey, $request->ride_id);
      // coupon usage alrealy in endRide
      return $this->successResponse([], __('locale.Wallet Payment done successfull'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }


}
