<?php

namespace App\Http\Controllers\API;
use Illuminate\Http\Request;
use App\Models\RideTemp;
use App\Models\Ride;
use App\Http\Controllers\Controller;
use App\Traits\ApiResponseTrait;
use App\Http\Resources\RideCollection;
use App\Http\Resources\RideResource;
use App\Http\Requests\RideRequestRequest;
use App\Helpers\Helpers;
use App\Repositories\Contracts\RideRepositoryInterface;
use App\Repositories\Contracts\RideRequestRepositoryInterface;
use App\Repositories\Contracts\ReviewRepositoryInterface;
class ParcelController extends Controller
{

  use ApiResponseTrait;
  protected $rideRepository, $rideRequestRepository, $reviewRepository;

  public function __construct(RideRepositoryInterface $rideRepository, RideRequestRepositoryInterface $rideRequestRepository, ReviewRepositoryInterface $reviewRepository)
  {
    $this->rideRepository = $rideRepository;
    $this->rideRequestRepository = $rideRequestRepository;
    $this->reviewRepository = $reviewRepository;
  }

  public function parcelSpecifications()
  {
    $messages = [
      'Maximum parcel weight allowed is 20kg.',
      'Fragile items must be marked clearly on the package.',
      'Parcels must not exceed 1m x 1m x 1m in size.',
      'Prohibited items (liquids, explosives, perishables) are not allowed.'
    ];
    return $this->successResponse($messages, '', 200);
  }

  public function requestParcelRide(RideRequestRequest $request)
  {
    $user = auth()->user();
    $countryId = $user->current_country_id ?? 94;
    $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $details = [
      'estimate_strike_fare' => $request->estimate_strike_fare,
      'coupon_id' => $request->coupon_id,
      'payment_type' => $request->payment_type,
      'payment_id' => $request->payment_id,
      'country_id' => $countryId,
    ];
    $packageDetails = [
      'sender_name' => $request->sender_name ?? '',
      'sender_phone' => $request->sender_phone ?? '',
      'receiver_name' => $request->receiver_name ?? '',
      'receiver_phone' => $request->receiver_phone ?? '',
      'pay_at' => $request->pay_at ?? 'Pick Up',
    ];
    $temp = RideTemp::create([
      'user_id' => auth()->id(),
      'vehicle_category_id' => $request->vehicle_id,
      'pickup_location' => $request->pickup_location,
      'drop_location' => $request->drop_location,
      'pickup_lat' => $request->pickup_lat,
      'pickup_lng' => $request->pickup_lng,
      'drop_lat' => $request->drop_lat,
      'drop_lng' => $request->drop_lng,
      'trip_distance' => $request->trip_distance,
      'trip_time' => $request->trip_time,
      'sender_name' => $request->sender_name ?? '',
      'receiver_name' => $request->receiver_name ?? '',
      'receiver_phone' => $request->receiver_phone ?? '',
      'parcel_category_id' => $request->parcel_category_id ?? null,
      'package_details' => !empty($packageDetails) ? json_encode($packageDetails) : '',
      'order_id' => $request->order_id ?? 0,
      'type' => 'Parcel',
      'fare' => $request->estimate_fare,
      'currency' => $currency,
      'details' => !empty($details) ? json_encode($details) : '',
    ]);
    $driverLimit = Helpers::setting('driver_request_simultaneously', 'site_setting');
    $nearestDrivers = $this->rideRepository->findNearestDrivers($request->pickup_lat, $request->pickup_lng, $driverLimit, $request->vehicle_id, $countryId, []);
    $this->rideRequestRepository->createForDrivers($temp, $nearestDrivers);
    return $this->successResponse($temp, __('locale.Request sent to drivers.'), 200);
  }
  public function parcelHistory(Request $request)
  {
    $user = auth()->user();
    $query = Ride::with('driver')->where('user_id', $user->id)->where('type', 'Parcel')->orderByDesc('created_at')->paginate(10);
    return new RideCollection($query);
  }

  public function parcelDetail(Request $request, $id)
  {
    return new RideResource(Ride::with('driver', 'user', 'reviews')->find($id));
  }

}
