<?php

namespace App\Http\Controllers\API;
use Illuminate\Http\Request;
use App\Models\Store;
use App\Models\Product;
use App\Models\Favorite;
use App\Repositories\Contracts\BannerRepositoryInterface;
use App\Repositories\Contracts\CategoryRepositoryInterface;
use App\Repositories\Contracts\StoreRepositoryInterface;
use App\Repositories\Contracts\ProductRepositoryInterface;
use App\Http\Controllers\Controller;
use App\Traits\ApiResponseTrait;
use App\Http\Resources\CategoryCollection;
use App\Http\Resources\BannerCollection;
use App\Http\Resources\StoreCollection;
use App\Http\Resources\StoreResource;
use App\Http\Resources\ProductCollection;
use App\Http\Resources\ProductResource;
class DeliveryController extends Controller
{

  use ApiResponseTrait;
  protected $categoryRepository, $bannerRepository, $storeRepository, $productRepository;

  public function __construct(CategoryRepositoryInterface $categoryRepository, BannerRepositoryInterface $bannerRepository, StoreRepositoryInterface $storeRepository, ProductRepositoryInterface $productRepository)
  {
    $this->categoryRepository = $categoryRepository;
    $this->bannerRepository = $bannerRepository;
    $this->storeRepository = $storeRepository;
    $this->productRepository = $productRepository;
  }

  public function categories()
  {
    $query = $this->categoryRepository->categories();
    return new CategoryCollection($query, true);
  }
  public function dashboard(Request $request)
  {
    $category = $this->categoryRepository->find($request->category_id);
    $banners = $this->bannerRepository->banners('category', $request->category_id);
    $categories = $this->categoryRepository->subCategoriesDashboard($request->category_id);
    $recommended = $this->storeRepository->storeDashboard($request, 'recommended', '4');
    $featured = $this->storeRepository->storeDashboard($request, 'featured', '4');
    return $this->successResponse([
      'heading' => $category->description ?? '',
      'banners' => (new BannerCollection($banners))->resolve()['data'],
      'categories' => (new CategoryCollection($categories))->resolve()['data'],
      'recommended_store' => (new StoreCollection($recommended))->resolve()['data'],
      'featured_store' => (new StoreCollection($featured))->resolve()['data'],
    ], '', 200);
  }

  public function categoryDashboard(Request $request)
  {
    $recommended = $this->storeRepository->storeDashboard($request, 'recommended', '4');
    $featured = $this->storeRepository->storeDashboard($request, 'all', '10');
    $category = $this->categoryRepository->find($request->category_id);
    return $this->successResponse([
      'recommended_store' => (new StoreCollection($recommended))->resolve()['data'],
      'all_store' => (new StoreCollection($featured))->response()->getData(true),
      'category' => $category->name ?? '',
    ], '', 200);
  }


  public function storeDetail(Request $request, $id)
  {
    $query = $this->storeRepository->storeDetail($request, $id);
    return (new StoreResource($query))->additional(['popup' => false]);
  }

  public function storePopupDetail(Request $request, $id)
  {
    $query = $this->storeRepository->storeDetail($request, $id);
    return (new StoreResource($query))->additional(['popup' => true]);
  }

  public function toggleStoreFavorite(Request $request)
  {
    $user = auth()->user();
    $type = $request->type; // 'store' or 'product'
    $id = $request->id;
    $model = $type === 'store' ? Store::class : Product::class;
    $item = $model::findOrFail($id);
    $favorite = Favorite::where([
      'user_id' => $user->id,
      'favoritable_type' => $model,
      'favoritable_id' => $id,
    ])->first();
    if ($favorite) {
      $favorite->delete();
      return $this->successResponse([], __('locale.Removed from favorites'), 200);
    }
    Favorite::create([
      'user_id' => $user->id,
      'favoritable_type' => $model,
      'favoritable_id' => $id,
    ]);
    return $this->successResponse([], __('locale.Added to favorites'), 200);
  }

  public function productList(Request $request, $storeId)
  {
    $query = $this->productRepository->productList($request, $storeId);
    $collection = new ProductCollection($query);
    $fullArray = $collection->response()->getData(true);
    return $this->successResponse(['data' => $fullArray['data'], 'links' => $fullArray['links'], 'meta' => $fullArray['meta']], __('locale.Data fetched successfully'));
  }

  public function productDetail($id)
  {
    $query = $this->productRepository->productDetail($id);
    return new ProductResource($query);
  }

  public function toggleProductFavorite(Request $request)
  {
    $user = auth()->user();
    $type = $request->type; // 'store' or 'product'
    $id = $request->id;
    $model = $type === 'store' ? Store::class : Product::class;
    $item = $model::findOrFail($id);
    $favorite = Favorite::where([
      'user_id' => $user->id,
      'favoritable_type' => $model,
      'favoritable_id' => $id,
    ])->first();
    if ($favorite) {
      $favorite->delete();
      return $this->successResponse([], __('locale.Removed from favorites'), 200);
    }
    Favorite::create([
      'user_id' => $user->id,
      'favoritable_type' => $model,
      'favoritable_id' => $id,
    ]);
    return $this->successResponse([], __('locale.Added to favorites'), 200);
  }

}
