<?php

namespace App\Http\Controllers;

use App\Models\Brands;
use App\Models\Category;
use App\Models\Inventory;
use App\Models\ProductGallery;
use App\Models\Products;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class ProductsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
         $sortEntity = (new Products())->sortEntity;
         $sortOrder = (new Products())->sortOrder;

        $result = null;
        if ($request->ajax()) {
            $sortEntity = $request->sortEntity;
            $sortOrder = $request->sortOrder;

            $result = (new Products)->pagination($request);
           
            return view('admin.products.pagination', compact('result', 'sortOrder', 'sortEntity'));
        }
        $url = url()->full();
        return view('admin.products.index',compact('url','result', 'sortOrder', 'sortEntity'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = (new Category())->service();
        $brands = (new Brands())->service();
        return view('admin.products.create',compact('categories','brands'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
      
        $inputs = $request->all();
        $rules = [
            'category'  => 'required|integer|exists:category,id',
            'subcategory' => 'required|integer|exists:subcategory,id',
            'brand' => 'required|integer|exists:brands,id',
            'name'   => 'required|string',
            'price'   => 'required|numeric|min:1',
            'offer_price'   => 'required|numeric|lt:price',
            'shipping_fee'   => 'required|numeric|min:1',
            'description' => 'nullable',
            'pieces_available' => 'required|numeric',
            'sku_number' => 'required',
            'barcode_number' => 'required',
            'estimated_delivery_time' => 'required|numeric',
            'status'  =>  'in:0,1',
            'cover_image' => 'required|mimes:jpg,jpeg,png,svg',
             'gallery_images'   => 'required|array|min:1',  //For testing purpose
            //  'gallery_images'   => 'required|array|min:5',
            'gallery_images.*'=> 'mimes:jpg,jpeg,png,svg'
        ];
        $validation = validator($inputs, $rules);
        if ($validation->fails()) {
             return back()->withErrors($validation->getMessageBag());
        }

        try{
            DB::beginTransaction();
        
        $coverImg = null;
        $path = public_path(PRODUCTS_PATH);
        if ($request->hasFile('cover_image')) {
            $file = $request->file('cover_image');
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move($path, $fileName);

            $coverImg = $fileName;
        }
       $product = Products::create([
            'category_id' => $inputs['category'],
            'subcategory_id' => $inputs['subcategory'],
            'brand_id' => $inputs['brand'],
            'name' => $inputs['name'],
            'price' => $inputs['price'],
            'offer_price' => $inputs['offer_price'],
            'shipping_fee' => $inputs['shipping_fee'],
            'description' => $inputs['description'],
            'no_of_pieces_available' => $inputs['pieces_available'],
            'sku_number' => $inputs['sku_number'],
            'barcode_number' => $inputs['barcode_number'],
            'estimated_delivery_time' => $inputs['estimated_delivery_time'],
            'status' => $inputs['status'],
             'cover_image' => $coverImg
        ]);
         /* ---------------- Gallery Images ---------------- */
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                 $file = $image;
                $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move($path, $fileName);
                ProductGallery::create([
                    'product_id' => $product->id,
                    'image' => $fileName,
                ]);
            }
        }
        DB::commit();
        return redirect()->route('products.index')->with('success', __('admin.product_created_successfully'));

        
        } catch (\Exception $e) {
        DB::rollBack();
          return jsonResponse(true, 207, __('internal_server_error'));
        }
    }

    /**
     * Display the specified resource.
     */
     public function show(Products $product)
    {
         $product->load('gallery','category','subcategory','brands','inventory');
        $categories = (new Category())->service();

        return view('admin.products.show',compact('product','categories'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Products $product)
    {
        $product->load('category','subcategory','brands','gallery');
         $categories = (new Category())->service();
        $brands = (new Brands())->service();
        return view('admin.products.edit',compact('product','categories','brands'));
    }

    /**
     * Update the specified resource in storage.
     */
 public function update(Request $request, Products $product)
{
    $rules = [
        'category'  => 'required|integer|exists:category,id',
        'subcategory' => 'required|integer|exists:subcategory,id',
        'brand' => 'required|integer|exists:brands,id',
        'name'   => 'required|string',
        'price'   => 'required|numeric|min:1',
        'offer_price'   => 'required|numeric|lt:price',
        'shipping_fee'   => 'required|numeric|min:1',
        'description' => 'nullable',
        // 'pieces_available' => 'required|numeric',
        'sku_number' => 'required',
        'barcode_number' => 'required',
        'estimated_delivery_time' => 'required|numeric',
        'status'  => 'required|in:0,1',
        'cover_image' => 'nullable|mimes:jpg,jpeg,png,svg',
        'gallery_images' => 'nullable|array|min:1',  //For Testing purpose
        // 'gallery_images' => 'nullable|array|min:5',
        'gallery_images.*'=> 'mimes:jpg,jpeg,png,svg'
    ];

    $request->validate($rules);

    try {
        DB::beginTransaction();

        
            $product->category_id = $request->category;
             $product->subcategory_id = $request->subcategory;
             $product->brand_id = $request->brand;
            $product->name = $request->name;
             $product->price = $request->price;
             $product->offer_price = $request->offer_price;
             $product->shipping_fee = $request->shipping_fee;
             $product->description = $request->description;
            //  $product->no_of_pieces_available = $request->pieces_available;
             $product->sku_number = $request->sku_number;
             $product->barcode_number = $request->barcode_number;
             $product->estimated_delivery_time = $request->estimated_delivery_time;
             $product->status = $request->status;
        

        /* Cover image */
        if ($request->hasFile('cover_image')) {
            $path = public_path(PRODUCTS_PATH);
            $file = $request->file('cover_image');
            $fileName = time().'_'.uniqid().'.'.$file->getClientOriginalExtension();
            $file->move($path, $fileName);
            $product->cover_image = $fileName;
        }

       
        /* UPDATE PRODUCT */
        $product->save();

        /* Remove gallery images */
        if ($request->removed_gallery_images) {
            $ids = explode(',', $request->removed_gallery_images);
            ProductGallery::whereIn('id', $ids)->delete();
        }

        /* Add new gallery images */
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $fileName = time().'_'.uniqid().'.'.$image->getClientOriginalExtension();
                $image->move(public_path(PRODUCTS_PATH), $fileName);

                ProductGallery::create([
                    'product_id' => $product->id,
                    'image' => $fileName,
                ]);
            }
        }

        DB::commit();

        return redirect()->back()->with('success', __('admin.product_updated_successfully'));

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->with('error', $e->getMessage());
    }
}


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Products $product)
{
    try {
        DB::beginTransaction();

        /*  Delete cover image */
        if ($product->cover_image) {
            $coverPath = public_path(PRODUCTS_PATH . $product->cover_image);
            if (File::exists($coverPath)) {
                File::delete($coverPath);
            }
        }

        /*  Delete gallery images */
        if ($product->gallery) {
            foreach ($product->gallery as $gallery) {
                $galleryPath = public_path(PRODUCTS_PATH . $gallery->image);
                if (File::exists($galleryPath)) {
                    File::delete($galleryPath);
                }
            }

            /* Delete gallery records */
            $product->gallery()->delete();
        }

        /*  Delete product */
        $product->delete();

        DB::commit();

        return response()->json([
            'success' => true,
            'status'  => 201,
            'message' => __('admin.product_deleted_successfully'),
            'extra'   => ['redirect' => route('products.index')]
        ]);

    } catch (\Exception $e) {
        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}

    public function toggleAllStatus($status, Request $request)
    {
        
        $inputs = $request->only('ids');

        try {
            DB::beginTransaction();
            $inputs = $request->only('ids');

            (new Products())->toggleStatus($status, $inputs['ids']);
            DB::commit();

           return response()->json(['success' => true,'status'  => 201,'message' => __('admin.status_updated_successfully'),'extra'   => ['redirect' => route('products.index')]]);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 207, __('admin.server_error'));
        }
    }

    public function status($id)
    {
        $result = Products::findorFail($id);
        if (!$result) {
            $message = __('admin.invalid_detail');
            return jsonResponse(false, 207, $message);
        }

        try {
            DB::beginTransaction();
            $result->update(['status' => !$result->status]);
            DB::commit();
              return response()->json(['success' => true,'status'  => 201,'message' => __('admin.status_updated_successfully'),'extra'   => ['redirect' => route('products.index')]]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 207, __('admin.server_error'));
        }
    }
     public function service(Request $request)
    {
        $inputs = $request->all();
        $rules = [
            'subcategory_id' => 'required|numeric|min:1'
        ];
        $validation =  validator($inputs, $rules);
        if ($validation->fails()) {
            return jsonResponse(false, 206, $validation->getMessageBag());
        }

        $title = $inputs['title'] ?? __('admin.select');
        $options = '<option value="" selected disabled>'.__('admin.select').'</option>';

        $result = (new Products())->service(false, $title, ['subcategory_id' => $inputs['subcategory_id']]);
        if (isset($result) && count($result) > 0) {
            foreach ($result as $key => $option) {
                $options .= '<option value="' . $key . '">' . $option . '</option>';
            }
        }
       
        return response()->json(['success' => true, 'options' => $options]);
    }
      public function getPrice(Request $request)
    {
        $inputs = $request->all();
        $rules = [
            'product_id' => 'required|numeric|min:1'
        ];
        $validation =  validator($inputs, $rules);
        if ($validation->fails()) {
            return jsonResponse(false, 206, $validation->getMessageBag());
        }

       $productId = $inputs['product_id'];
       $result = Products::find($productId);
       $price = $result->price;
      
       
        return response()->json(['success' => true, 'price' => $price]);
    }
     public function updateInventory(Request $request)
    {
         $inputs = $request->all();
        $rules = [
            'product_id' => 'required|numeric|min:1|exists:products,id',
            'stock_type' => 'required|in:in,out',
            'quantity' => 'required|numeric|min:1'
        ];
        $validation =  validator($inputs, $rules);
        if ($validation->fails()) {
            return jsonResponse(false, 206, $validation->getMessageBag());
        }

       $productId = $inputs['product_id'];
       $product = Products::find($productId);
       $updatedStock = 0;
       $currentStock = $product->no_of_pieces_available;
       try{
            DB::beginTransaction();
            if($inputs['stock_type'] == 'in')
                {
                        $updatedStock = $currentStock + $inputs['quantity'];
                }else if($inputs['stock_type'] == 'out')
                {
                        $updatedStock = ($currentStock > 0) ? $currentStock - $inputs['quantity'] : $currentStock + $inputs['quantity'];
                }
                Inventory::create([
                    'product_id' => $productId,
                    'stock_type' => $inputs['stock_type'],
                    'available_stock' => $currentStock,
                    'quantity' => $inputs['quantity'],
                    'updated_stock' => $updatedStock
                ]);
                //Update Inventory in products table
                $product->no_of_pieces_available = $updatedStock;
                $product->save();
                DB::commit();
                return redirect()->back()->with('success', __('admin.inventory_updated_successfully'));

            } catch (\Exception $e) {
                DB::rollBack();
                return back()->with('error', $e->getMessage());
            }
       
    }
}
