<?php

namespace App\Http\Controllers;

use App\Models\Customers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class CustomersController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
         $sortEntity = (new Customers())->sortEntity;
         $sortOrder = (new Customers())->sortOrder;

        $result = null;
        if ($request->ajax()) {
            $sortEntity = $request->sortEntity;
            $sortOrder = $request->sortOrder;

            $result = (new Customers)->pagination($request);
           
            return view('admin.customers.pagination', compact('result', 'sortOrder', 'sortEntity'));
        }
        $url = url()->full();
        return view('admin.customers.index',compact('url','result', 'sortOrder', 'sortEntity'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.customers.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
       $clean_phone = preg_replace('/^' . preg_quote($request->dial_code, '/') . '/','',$request->phone);
        $request->merge(['phone' => $clean_phone]);
        $inputs = $request->all();
        $rules = [
            'name'              => 'required|string|max:150',
            'email'             => 'required|email|unique:customers,email',
            'dial_code'         => 'required',
            'dial_code_iso'     => 'required',
            'phone'             => 'required|digits:10||unique:customers,phone',
            'password'          => 'required|min:8',
            'status'            =>  'in:0,1'
        ];
        $validation = validator($inputs, $rules);
        if ($validation->fails()) {
             return back()->withErrors($validation->getMessageBag());
        }
        $customerImg = null;
        $path = public_path(CUSTOMERS_PATH);
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move($path, $fileName);

            $customerImg = $fileName;
        }
        Customers::create([
            'name'           => $inputs['name'],
            'email'          => $inputs['email'],
            'dial_code'      => $inputs['dial_code'],
            'dial_code_iso'  => $inputs['dial_code_iso'],
            'phone'          => $inputs['phone'],
            'password'       => Hash::make($inputs['password']),
            'status'      => $inputs['status'],
            'image' => $customerImg
        ]);
        return redirect()->route('customers.index')->with('success', __('admin.customer_created_successfully'));
    }

    /**
     * Display the specified resource.
     */
     public function show(Customers $customer)
    {
        return view('admin.customers.show',compact('customer'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Customers $customer)
    {
       $clean_phone = preg_replace('/^' . preg_quote($request->dial_code, '/') . '/','',$request->phone);
        $request->merge(['phone' => $clean_phone]);
        $validated = $request->validate([
            'name'          => 'nullable|string|max:150',
            'email'         => 'nullable|email|unique:customers,email,' . $customer->id,
            'dial_code'     => 'required',
            'dial_code_iso' => 'required',
            'phone'         => 'required|digits:10|unique:customers,phone,' . $customer->id,
            'phone' => [
                            'required',
                            'digits:10',
                            Rule::unique('customers')->where(function ($query) use ($request) {
                                return $query->where('dial_code', $request->dial_code)->where('dial_code_iso', $request->dial_code_iso);
                            })->ignore($customer->id)
                        ],
            'password'      => 'nullable|min:8',
            'status'        => 'required|boolean',
        ]);
        $customer->name  = $validated['name'];
        $customer->email = $validated['email'];
        $customer->dial_code = $validated['dial_code'];
        $customer->dial_code_iso = $validated['dial_code_iso'];
        $customer->phone = $validated['phone'];
        $customer->status = $validated['status'];
        if (!empty($validated['password'])) {
            $customer->password = Hash::make($validated['password']);
        }
        if($validated['status'] == 0) {
            $customer->tokens()->delete();
        }
        if ($request->hasFile('image')) {
            $path = public_path(CUSTOMERS_PATH);
            $file = $request->file('image');
            $fileName = time().'_'.uniqid().'.'.$file->getClientOriginalExtension();
            $file->move($path, $fileName);
            $customer->image = $fileName;
        }
        $customer->save();

        return redirect()->back()->with('success', __('admin.customer_updated_successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Customers $customer)
    {
        $customer->delete();
        return response()->json(['success' => true,'status'  => 201,'message' => __('admin.customer_deleted_successfully'),
        'extra'   => ['redirect' => route('customers.index')]]);
    }

    public function toggleAllStatus($status, Request $request)
    {
        
        $inputs = $request->only('ids');

        try {
            DB::beginTransaction();
            $inputs = $request->only('ids');

            (new Customers)->toggleStatus($status, $inputs['ids']);
            DB::commit();

           return response()->json(['success' => true,'status'  => 201,'message' => __('admin.status_updated_successfully'),'extra'   => ['redirect' => route('customers.index')]]);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 207, __('admin.server_error'));
        }
    }

    public function status($id)
    {
        $result = Customers::findorFail($id);
        if (!$result) {
            $message = __('admin.invalid_detail');
            return jsonResponse(false, 207, $message);
        }

        try {
            DB::beginTransaction();
            $result->update(['status' => !$result->status]);
            DB::commit();
              return response()->json(['success' => true,'status'  => 201,'message' => __('admin.status_updated_successfully'),'extra'   => ['redirect' => route('customers.index')]]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 207, __('admin.server_error'));
        }
    }
}
