<?php

namespace App\Models;

use App\Helpers\CommonHelper;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class order extends Model
{
    use HasFactory;
    protected $table = 'orders';
    protected $fillable = [
        'customer_id',
        'shipping_address_id',
        'billing_address_id',
        'order_number',
        'order_status',
        'subtotal',
        'total_price',
        'currency_code',
        'conversion_rate',
        'tax_percent',
        'tax_amount',
        'total_items',
        'refund_status',
        'is_approved_by',
        'approved_date',
        'payment_status'
    ];
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customer_id', 'id')->withTrashed();
    }
    // public function shippingAddress()
    // {
    //     return $this->belongsTo(shipping_address::class);
    // }
    public function orderDetails()
    {
        return $this->hasMany(orderDetail::class);
    }
    public function orderStatusLogs()
    {
        return $this->hasMany(OrderStatusLog::class);
    }
    public function transaction()
    {
        return $this->hasOne(transaction::class);
    }
    public function billingAddress()
    {
        return $this->belongsTo(BillingAddress::class, 'billing_address_id');
    }
    public function shippingAddress()
    {
        return $this->belongsTo(ShippingAddress::class, 'shipping_address_id');
    }
    public function orderCancel()
    {
        return $this->hasOne(OrderCancellation::class, 'order_id');
    }
    public function orderReturn()
    {
        return $this->hasOne(OrderReturn::class, 'order_id');
    }
    //get payment status atribute
    public function getPaymentStatusTextAttribute()
    {
        return CommonHelper::paymentStatusText($this->status);
    }
    public function getPaginatedData($request)
    {
        $query = self::with('customer')->leftJoin('customers', 'customers.id', '=', 'orders.customer_id')
            ->select('orders.*', 'customers.full_name as customer_name');


        $perPage = $request->input('per_page', 10);
        $allowedSortColumns = [
            'id'            => 'orders.id',
            'order_number'  => 'orders.order_number',
            'total_price'   => 'orders.total_price',
            'order_status'  => 'orders.order_status',
            'total_items'   => 'orders.total_items',
            'created_at'    => 'orders.created_at',
            'customer_name' => 'customers.full_name',
        ]; // add only valid DB columns here

        // Default to 'orders.id' if invalid or empty
        $sortByInput = $request->get('sort_by', 'id');
        $sortBy = $allowedSortColumns[$sortByInput] ?? 'orders.id';

        $sortOrder = $request->get('sort_order', 'desc');
        $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? $sortOrder : 'desc';

        if ($request->filled('keyword')) {
            $keyword = strtolower($request->keyword);
            $query->where(function ($q) use ($keyword) {
                $q->where('orders.order_number', 'like', "%$keyword%")
                    ->orWhere('orders.id', 'like', "%$keyword%");
                //also serach by order status text
                foreach (CommonHelper::$orderSatuses as $key => $label) {
                    if (str_contains(strtolower($label), $keyword)) {
                        $q->orwhere('order_status', $key);
                    }
                }
            });
        }
        if ($request->filled('status_filter')) {
            $query->where('order_status', $request->status_filter);
        }
        if ($request->filled('customer_id')) {
            $customerId = $request->customer_id;
            $query->where('customer_id', $customerId);
        }

        return $query->orderBy($sortBy, $sortOrder)->paginate($perPage);
    }
}
