<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerWholesalePlan;
use App\Models\WholesaleIdPlan;
use App\Services\CashBackService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Throwable;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    public function getProfile()
    {
        try {

            $customer = Auth::user();
            $image = null;
            if ($customer) {
                if ($customer->image == '' || $customer->image == 'defualt.png') {
                    $image = asset('uploads/default.png');
                } else {
                    $image = asset($customer->image);
                }
            }

            $data = [
                'full_name' => $customer->full_name,
                'email' => $customer->email,
                'dial_code' => $customer->dial_code,
                'dial_code_iso' => $customer->dial_code_iso,
                'mobile_number' => $customer->mobile_number,
                'currency_code' => $customer->currency,
                'currency_symbol' => $customer->currency_symbol,
                'timezone' => $customer->timezone,
                'image' => $image,
                'is_profile_updated' => $customer->is_profile_updated
            ];
            return CommonHelper::apiResponse(200, __("Profile successfully retrieved"), $data);
        } catch (Throwable $e) {

            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function updateProfile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'full_name' => 'required|min:3|max:20'
            ]);
            if ($validator->fails()) {
                return CommonHelper::apiResponse(400, $validator->errors());
            }
            $customer = Auth::user();
            if (!$customer) {
                return CommonHelper::apiResponse(404, __("Customer not found"));
            }
            $customer->full_name = $request->full_name;
            $customer->save();
            return CommonHelper::apiResponse(200, __("Profile updated successfully"));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function updateProfileImage(Request $request)
    {
        try {
            $rules = [
                'image' => [
                    'required',
                    'mimes:jpg,jpeg,png,gif,webp,svg',
                    'max:2048',
                    'image'
                ]

            ];
            $message = [
                'image.mimes' => __("Only image file is allowed"),
            ];
            $validator = Validator::make($request->all(), $rules, $message);
            if ($validator->fails()) {
                return CommonHelper::apiResponse(400, $validator->errors());
            }
            $customer = Auth::user();
            if (!$customer) {
                return CommonHelper::apiResponse(404, __("Customer not found"));
            }
            //update image
            $image_path = $customer->image;
            if ($request->hasFile('image')) {
                $image_path = CommonHelper::fileUpload($request->file('image'), CUSTOMER_PATH);
            }
            $customer->image = $image_path;
            $customer->save();
            return CommonHelper::apiResponse(200, __("Profile image updated successfully"));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //change password
    public function changePassword(Request $request)
    {
        try {
            $rules = [
                'old_password' => [
                    'required',

                ],
                'new_password' => [
                    'required',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/',

                ],
                'password_confirmation' => ['required', 'same:new_password'],
            ];
            $message = [
                'new_password.regex' => __('invalid password'),

            ];

            $validator = Validator::make($request->all(), $rules, $message);
            if ($validator->fails()) {
                return CommonHelper::apiResponse(400, $validator->errors());
            }
            $customer = Auth::user();
            if (!$customer) {
                return CommonHelper::apiResponse(404, __("Customer not found"));
            }
            if (!Hash::check($request->old_password, $customer->password)) {
                return CommonHelper::apiResponse(400, __("Old password is not correct"));
            }
            if (Hash::check($request->new_password, $customer->password)) {
                return CommonHelper::apiResponse(400, __("New password is same as old password"));
            }
            $customer->password = Hash::make($request->new_password);
            $customer->save();
            return CommonHelper::apiResponse(200, __("Password changed successfully"));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function deleteAccount()
    {
        $customer = Auth::user();
        $customer = Customer::find($customer->id);
        $customer->delete();
        $customer->tokens()->delete();
        return CommonHelper::apiResponse(200, __("Customer deleted successfully"));
    }
    public function getAllCurrencies()
    {
        $currencies = DB::table('currencies_master')
            ->select('currency_code', 'currency_symbol')
            ->get();

        return CommonHelper::apiResponse(200, '', $currencies);
    }
    //update currency
    public function updateCurrency(Request $request)
    {
        try {
            $validation = Validator::make($request->all(), [
                'currency_code' => 'required|string|size:3|uppercase',
            ]);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            //update currency code:
            $currencyInfo = CommonHelper::getCurrencyInfo($request->currency_code);
            if (!$currencyInfo) {
                return CommonHelper::apiResponse(404, __("curency info not found"));
            }
            $customer = auth()->user();
            $customer->currency = $currencyInfo->currency_code;
            $customer->currency_name = $currencyInfo->currency_name;
            $customer->currency_symbol = $currencyInfo->currency_symbol;
            $customer->save();
            return CommonHelper::apiResponse(200, __("currency updated"));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }

    //get wholesale discount plans
    public function getCashbackPlans()
    {
        $customer = auth()->user();
        $plans = WholesaleIdPlan::where('status', 1)->get();
        if (!$plans) {
            return CommonHelper::apiResponse(400, __('Cashback plan not found'));
        }
        try {
            $currencyInfo = CashBackService::currencyWiseInfo($customer->currency);

            $priceField = $currencyInfo['priceField'];
            $minOrderAmountField = $currencyInfo['minOrderAmountField'];
            $plansWithFileteredColumns =  $plans->map(function ($plan) use ($priceField, $minOrderAmountField) {
                return [
                    'plan_id' => $plan->id,
                    'title' => $plan->title,
                    'price' => $plan->$priceField,
                    'min_order_amt' => $plan->$minOrderAmountField,
                    'discount_in_percent' => $plan->discount,
                    'is_default' => $plan->is_default,
                    'status' => $plan->status

                ];
            });

            return CommonHelper::apiResponse(200, __('cash back plans retrieved'), $plansWithFileteredColumns);
        } catch (Throwable $e) {

            return CommonHelper::apiErrorResponse($e);
        }
    }

    //assign or buy the cashback plan by customer
    public function assignCashbackPlan(Request $request, CashBackService $cashBackService)
    {
        DB::beginTransaction();
        $customer = auth()->user();
        $customerId = $customer->id;
        $validation = Validator::make($request->all(), [
            'plan_id' => 'required|integer|exists:wholesale_id_plans,id',
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        $planId = $request->plan_id;

        $plan = WholesaleIdPlan::find($planId);
        if (!$plan) {
            return CommonHelper::apiResponse(400, __('Cashback plan not found'));
        }
        try {

            //check existing active plan with cashback add in wallet
            $existingPlan = $customer->activePlanWholesale;
            if ($existingPlan) {
                $fromDate = Carbon::parse($existingPlan->cashback_from_date);
                $toDate = Carbon::now();

                CashBackService::processCashBack($existingPlan, $customer, $fromDate, $toDate);

                // Close out old plan
                $cashBackService->deactivateExistingPlans($customerId);
            }

            //check if any existing plan id created for customer
            $customPlanId = $cashBackService->getOrCreateCustomPlanIdForCustomer($customerId);
            //every plan have one year duration

            $planStartDate = Carbon::now();
            $endDate = $planStartDate->copy()->addYear();

            CustomerWholesalePlan::create([
                'customer_id' => $customerId,
                'original_plan_id' => $plan->id,
                'custom_plan_id' => $customPlanId,
                'title' => $plan->title,
                'price_in_usd' => $plan->price_in_usd,
                'price_in_mmk' => $plan->price_in_mmk,
                'price_in_rmb' => $plan->price_in_rmb,
                'discount' => $plan->discount,
                'number_of_days' => 0,
                'min_order_amt_usd' => $plan->min_order_amt_usd,
                'min_order_amt_mmk' => $plan->min_order_amt_mmk,
                'min_order_amt_rmb' => $plan->min_order_amt_rmb,
                'start_date' => $planStartDate->toDateTimeString(),
                'end_date' => $endDate->toDateTimeString(),
                'status' => ACTIVE,
                'cashback_from_date' => $planStartDate->toDateTimeString(),
                'cashback_to_date'   => $planStartDate->copy()->endOfMonth()->endOfDay()->toDateTimeString(),
            ]);
            Customer::where('id', $customerId)->update(['wholesale_plan_id' => $customPlanId]);
            DB::commit();

            return CommonHelper::apiResponse(200, "You’ve successfully subscribed to a cashback plan");
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
