<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Mail\SendOtpMail;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Auth;
use Kreait\Firebase\Exception\Auth\FailedToVerifyToken;
use ArrayObject;
use Illuminate\Support\Facades\Auth as CustomerAuth;

class AuthController extends Controller
{

    public function register(Request $request)
    {

        try {
            DB::beginTransaction();
            $data = new ArrayObject();
            $rules = [
                'full_name' => 'required|string|min:3|max:20',
                'email' => [
                    'required',
                    'email',
                    Rule::unique('customers')->whereNull('deleted_at')
                ],
                'dial_code' => 'required',
                'dial_code_iso' => 'required',
                'mobile_number' => 'required|numeric',
                'password' => [
                    'required',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/',

                ]
            ];
            $messages = [
                'password.regex' => __('invalid password'),
                'email.unique' => __('already taken'),
            ];

            $validation = Validator::make($request->all(), $rules, $messages);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            //mobile invalid
            if (!Customer::validateMobileNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                return CommonHelper::apiResponse(400, __("Invalid Mobile Number"));
            }
            $fullMobileNumber = $request->dial_code . $request->mobile_number;

            //mobile exist
            /*$existMb = Customer::existMobileNumber($request->dial_code, $request->dial_code_iso,  $request->mobile_number);
            if (!empty($existMb)) {
                return CommonHelper::apiResponse(400, __("mobile exist"));
            }*/


            //mobile number already exist
            $customerExists = Customer::where('dial_code', $request->dial_code)
                ->where('mobile_number', $request->mobile_number)
                ->withoutTrashed()->first();
            if ($customerExists) {
                return CommonHelper::apiResponse(400, __("mobile number exist"));
            }

            //email id already exist
            $emailExists = Customer::where('email', $request->email)
                ->withoutTrashed()->first();
            if ($emailExists) {
                return CommonHelper::apiResponse(400, __("email exist"));
            }



            //generate otp
            $emailOtp = rand(111111, 999999);
            //fetch currency info
            $currencyInfo = CommonHelper::getCurrencyInfo('USD');
            //fecth and store country data
            $countryInfo = CommonHelper::getCountryInfo($request->dial_code_iso);
            if ($countryInfo) {
                $timezone = $countryInfo['timezone'];
            }
            //save data
            $customer =   Customer::create([
                'full_name' => $request->full_name,
                'email' => $request->email,
                'dial_code' => $request->dial_code,
                'dial_code_iso' => $request->dial_code_iso,
                'mobile_number' => $request->mobile_number,
                'password' => Hash::make($request->password),
                'email_otp' => $emailOtp,
                'currency' => $currencyInfo->currency_code ?? null,
                'currency_name' => $currencyInfo->currency_name ?? null, //$request->currency,
                'currency_symbol' => $currencyInfo->currency_symbol ?? null,
                'timezone' => $timezone ?? null,
                'status' => 1,

            ]);

            //send email otp
            try {
                Mail::to($request->email)->send(new SendOtpMail($request->full_name, $emailOtp));
            } catch (\Exception $e) {
                if (str_contains($e->getMessage(), '550')) {
                    return CommonHelper::apiResponse(400, __('email invalid'));
                }
                return CommonHelper::apiResponse(500, __('email send error'));
            }
            DB::commit();
            return CommonHelper::apiResponse(200, __("email otp sent"));
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //verify mobile otp -  verifying a Firebase ID token 
    public function verifyMobileOtp(Request $request)
    {
        $validation =  Validator::make($request->all(), [
            'firebase_token' => 'required|string',
            'dial_code' => 'required|string',
            'dial_code_iso' => 'required|string',
            'mobile_number' => 'required|numeric',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        try {
            $servicePath = storage_path('firebase.json');
            $factory = (new Factory)->withServiceAccount($servicePath);
            $auth = $factory->createAuth();
            $verifiedIdToken = $auth->verifyIdToken($request->firebase_token);
            $uid = $verifiedIdToken->claims()->get('sub');
            $user = $auth->getUser($uid);

            //check
            $customer = Customer::where('dial_code', $request->dial_code)->where('mobile_number', $request->mobile_number)->first();
            if (!$customer) {
                //dd('here');
                return CommonHelper::apiResponse(400, __('mobile number not match'));
            }
            $fullMobileNumber = $customer->dial_code . $customer->mobile_number;
            if ($user->phoneNumber != $fullMobileNumber) {
                return CommonHelper::apiResponse(400, __('mobile number not match'));
            }
            $customer->is_mobile_no_verified = 1;
            $customer->mobile_verified_at = now();
            $customer->save();
            DB::commit();
            $message = __('successfully verified');
            $message_description = __('verification completed');
            return CommonHelper::apiResponse(200, $message, [], $message_description);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //email otp verification
    public function verifyEmailOtp(Request $request)
    {

        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'email' => 'required|email|exists:customers,email',
            'email_otp' => 'required|numeric',
            'fcm_token' => 'required|string',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        $customer = Customer::where('email', $request->email)->first();
        if (!$customer) {
            return CommonHelper::apiResponse(400, __('email not found'));
        }
        if ($customer->email_otp != $request->email_otp) {
            return CommonHelper::apiResponse(400, __('email otp is not correct'));
        }
        Customer::where('id', $customer->id)->update(['is_email_verified' => 1, 'email_verified_at' => now(), 'fcm_token' => $request->fcm_token, 'email_otp' => '']);
        Customer::checkCurrentFcm($request->fcm_token, $customer->id);
        $customer->fcm_token = $request->fcm_token;
        $customer->save();
        DB::commit();
        $data = [
            'auth_token'   => $customer->createToken($customer->email)->plainTextToken, //create token
        ];
        $message = __('email verified successfully');
        $messageDescription = __('verification completed');
        return CommonHelper::apiResponse(200, $message, $data, $messageDescription,);
        try {
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //resend otp
    public function resendOtp(Request $request)
    {
        try {
            DB::beginTransaction();
            $validation = Validator::make($request->all(), [
                'email' => 'required|email|exists:customers,email',

            ]);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            $customer = Customer::where('email', $request->email)->first();
            if (!$customer) {
                return CommonHelper::apiResponse(400, __('email not found'));
            }

            //generate otp
            $emailOtp = rand(111111, 999999);
            $customer->email_otp = $emailOtp;
            $customer->save();
            DB::commit();
            try {
                Mail::to($request->email)->send(new SendOtpMail($request->full_name, $emailOtp));
            } catch (\Exception $e) {
                if (str_contains($e->getMessage(), '550')) {
                    return CommonHelper::apiResponse(400, __('email invalid'));
                }
                return CommonHelper::apiResponse(500, __('email send error'));
            }
            return CommonHelper::apiResponse(200, __("email otp sent"));
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //login
    public function login(Request $request)
    {
        try {
            DB::beginTransaction();
            $validation = Validator::make(
                $request->all(),
                [
                    'mobile_number' => 'required|string',
                    'dial_code' => 'required|string',
                    'password' => 'required|string',
                    'fcm_token' => 'required|string',
                ]
            );

            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors(), [], true);
            }
            $result = Customer::where('dial_code', $request->dial_code)->where('mobile_number', $request->mobile_number)->first();
            if (!$result || !Hash::check($request->password, $result->password)) {
                return CommonHelper::apiResponse(400, __('invalid_credentials'));
            }
            //inactive customer
            if ($result->status == 0) {
                return  CommonHelper::apiResponse(400, __('account inactive'));
            }
            $data = [
                'is_mobile_no_verified' => $result->is_mobile_no_verified,
                'is_email_verified' =>  $result->is_email_verified,
                'email' =>  $result->email,
            ];
            //number verified
            if ($result->is_mobile_no_verified == 0) {
                return CommonHelper::apiResponse(200, __('mobile number not verified'), $data);
            }
            //email verified
            if ($result->is_email_verified == 0) {
                $emailOtp = rand(111111, 999999);
                $result->email_otp =  $emailOtp;
                //$result->save();
                try {
                    Mail::to($result->email)->send(new SendOtpMail($result->full_name, $emailOtp));
                } catch (\Exception $e) {
                    if (str_contains($e->getMessage(), '550')) {
                        return CommonHelper::apiResponse(400, __('email invalid'));
                    }
                    return CommonHelper::apiResponse(500, __('email send error'));
                }
                return CommonHelper::apiResponse(200, __("email otp sent"), $data);
            }
            Customer::checkCurrentFcm($request->fcm_token, $result->id);
            $result->fcm_token = $request->fcm_token;
            $result->save();
            //check existing token - for prevent multilogin the same user from more than one device
            $result->tokens()->delete();
            DB::commit();
            $data = [
                'auth_token' => $result->createToken($result->email)->plainTextToken,
                'id' => $result->id,
                'is_email_verified' => $result->is_email_verified,
                'is_mobile_no_verified' => $result->is_mobile_no_verified,
                'is_profile_updated' => $result->is_profile_updated,
            ];
            return CommonHelper::apiResponse(200, __('login success'), $data);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function logout()
    {
        $customer = Customer::find(CustomerAuth::user()?->id);
        if ($customer) {
            //null fcm token
            $customer->update(['fcm_token' => '']);
            $customer->tokens()->delete();
            return CommonHelper::apiResponse(200, __('logout success'));
        }
        return CommonHelper::apiResponse(400, __('not auhtenticated'));
    }
    //forgot password
    public function forgotPassword(Request $request)
    {
        try {
            DB::beginTransaction();
            $validations = Validator::make($request->all(), [
                'email' => 'required|email|min:3'
            ]);
            if ($validations->fails()) {
                return CommonHelper::apiResponse(400, $validations->errors()); // return validation errors
            }
            $customer = Customer::where('email', $request->email)->where('status', 1)->first();
            if (!$customer) {
                return CommonHelper::apiResponse(400, __('email not found'));
            }
            $emailOtp = mt_rand(111111, 999999);
            try {
                Mail::to($customer->email)->send(new SendOtpMail($customer->full_name, $emailOtp));
            } catch (\Exception $e) {
                if (str_contains($e->getMessage(), '550')) {
                    return CommonHelper::apiResponse(400, __('email invalid'));
                }
                return CommonHelper::apiResponse(500, __('email send error'));
            }

            $customer->email_otp = $emailOtp;
            $customer->save();
            DB::commit();
            return CommonHelper::apiResponse(200, __("email otp sent"));
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //verify forgot password otp
    public function verifyForgotPasswordOtp(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'email' => 'required|email|exists:customers,email',
            'email_otp' => 'required|numeric',

        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        $customer = Customer::where('email', $request->email)->where('status', 1)->first();
        if (!$customer) {
            return CommonHelper::apiResponse(400, __('email not found'));
        }
        if ($customer->email_otp != $request->email_otp) {
            return CommonHelper::apiResponse(400, __('email otp is not correct'));
        }

        $message = __('email verified successfully');

        return CommonHelper::apiResponse(200, $message);
        try {
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //change forgot password
    public function changeForgotPassword(Request $request)
    {
        try {
            DB::beginTransaction();

            $rules = [
                'email' => [
                    'required',
                    'email'
                ],

                'password' => [
                    'required',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/',
                    'confirmed'

                ]
            ];
            $messages = [
                'password.regex' => __('invalid password'),

            ];

            $validation = Validator::make($request->all(), $rules, $messages);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            $customer = Customer::where('email', $request->email)->where('status', 1)->first();
            if (!$customer) {
                return CommonHelper::apiResponse(400, __('email not found'));
            }
            $password = Hash::make($request->password);
            $customer->password = $password;
            $customer->email_otp = '';
            $customer->save();
            DB::commit();
            $message = __('password changed');
            $message_description = __('password changed successfully');
            return CommonHelper::apiResponse(200, $message, [], $message_description);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
