<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VehicleType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use App\Helpers\CommonHelper;
use App\Models\VehicleMake;
use App\Models\VehicleModel;
use Illuminate\Database\QueryException;
use Throwable;

class VehicleMakeController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-vehicle_make')->only('index');
        $this->middleware('permission:add-vehicle_make')->only(['create', 'store']);
        $this->middleware('permission:edit-vehicle_make')->only(['edit', 'update']);
        $this->middleware('permission:edit-vehicle_make')->only(['edit', 'show']);
        $this->middleware('permission:delete-vehicle_make')->only(['destroy']);
        $this->middleware('permission:edit-vehicle_make')->only(['checkVehicleModels']);
        $this->middleware('permission:edit-vehicle_make')->only(['deleteVehicleModels']);
    }
    public function index(Request $request)
    {
        $vehicleTypes = VehicleType::where('status', 1)->get();
        $vehicleMakes = (new VehicleMake())->getPaginatedData($request);

        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.vehicleMake.pagination', compact('vehicleMakes', 'vehicleTypes'))->render()
            ]);
        }

        return view('admin.vehicleMake.index', compact('vehicleMakes', 'vehicleTypes'));
    }
    public function create()
    {
        $vehicleTypes = VehicleType::where('status', 1)->get();
        return view('admin.vehicleMake.create', compact('vehicleTypes'));
    }
    public function store(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|string|min:3|max:200',
            'vehicle_types' => 'required|array',
            'vehicle_types.*' => 'exists:vehicle_types,id',
            'image' => 'required|image|mimes:jpeg,jpg,png,gif,webp|max:2048',
            'status' => 'required|in:0,1'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        $image = DEFAULT_PATH;
        if ($request->hasFile('image')) {
            $image = CommonHelper::fileUpload($request->image, MAKE_PATH);
        }
        try {
            $vehicleMake = VehicleMake::create([
                'name' => $request->name,
                'status' => $request->status,
                'image_path' => $image
            ]);
            if ($request->filled('vehicle_types')) {
                $vehicleMake->types()->attach($request->vehicle_types);
            }
            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-make.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle make added successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function edit($id)
    {
        $vehicleMake = VehicleMake::with('types')->findOrFail($id);
        $vehicleTypes = VehicleType::where('status', 1)
            ->orWhereHas('makes', function ($q) use ($id) {
                $q->where('vehicle_make_id', $id);
            })->get();
        return view('admin.vehicleMake.edit', compact('vehicleMake', 'vehicleTypes'));
    }
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|string|min:3|max:200',
            'vehicle_types' => 'required|array',
            'vehicle_types.*' => 'exists:vehicle_types,id',
            'image' => 'nullable|image|mimes:jpeg,jpg,png,gif,webp|max:2048'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }


        try {
            $vehicleMake = VehicleMake::findOrFail($id);
            $image = $request->image ?? $vehicleMake->image_path;
            if ($request->hasFile('image')) {
                $image = CommonHelper::fileUpload($request->image, MAKE_PATH,  $vehicleMake->image_path);
            }
            $data = [
                'name' => $request->name,
                'status' => $request->status,
                'image_path' => $image
            ];

            $vehicleMake->types()->sync($request->vehicle_types);

            $vehicleMake->update($data);
            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-make.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle make updated successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function destroy($id)
    {

        $vehicleMake = VehicleMake::findOrFail($id);

        try {
            if ($vehicleMake->image_path && file_exists(public_path($vehicleMake->image_path))) {
                unlink(public_path($vehicleMake->image_path));
            }
            //$vehicleMake->types()->detach();
            $vehicleMake->delete();
            $vehicleMake->update(['status' => 2]);
            $extra = [
                'redirect' => route('admin.vehicle-make.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle Make deleted successfully.', [], $extra);
        } catch (QueryException $e) {
            $msg = $e->getMessage();
            if (strpos($msg, '1451') !== false) { // fk check
                $message = 'Please delete entries first from its relevant sections';

                if (preg_match('/foreign key.*references `(\w+)`/', $message, $matches)) {
                    $message = 'Please delete entries first from ' . $matches[1] . ' sections';
                }

                return response()->json(['status' => false, 'status_code' => 500, 'message' => $message], 500);
            }
        } catch (Throwable $e) {
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    //check vehicle models whle editing, uncheck the type
    public function checkVehicleModels(Request $request)
    {
        $hasModels = VehicleModel::where('vehicle_type_id', $request->vehicle_type_id)
            ->where('vehicle_make_id', $request->vehicle_make_id)
            ->count();
        return response()->json([
            'hasModels' => $hasModels,
        ]);
    }
    public function deleteVehicleModels(Request $request)
    {
        VehicleModel::where('vehicle_type_id', $request->vehicle_type_id)
            ->where('vehicle_make_id', $request->vehicle_make_id)
            ->update(['status' => 2]);
        VehicleModel::where('vehicle_type_id', $request->vehicle_type_id)
            ->where('vehicle_make_id', $request->vehicle_make_id)
            ->delete();

        return response()->json([
            'success' => true,
        ]);
    }
}
