<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Department;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;
use App\Helpers\CommonHelper;
use App\Models\Role;
use Illuminate\Validation\Rule;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (auth()->user() && auth()->user()->id != 1) {
                if (in_array($request->route()->getActionMethod(), ['index', 'create', 'store', 'edit', 'update', 'destroy'])) {
                    abort(403, "You are not allowed to access this page.");
                }
            }
            return $next($request);
        });
    }
    public function index()
    {
        $roles = Role::with('department')->where('name', '!=', SUPER_ADMIN)->orderBy('name', 'asc')->get();
        return view('admin.roles.index', [
            'roles' => $roles
        ]);
    }
    public function create()
    {
        $departments = Department::orderBy('name', 'asc')->get();
        $permissions = Permission::orderBy('module', 'ASC')->get()->groupBy('module');
        return view('admin.roles.create', compact('permissions', 'departments'));
    }
    public function store(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:roles,name',
            'department_id' => 'required|exists:departments,id',
            'permissions' => 'required|array|min:1',
            'permissions.*' => 'string|exists:permissions,name',
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            $permissions = $request->input('permissions', []);
            foreach ($permissions as $permission) {
                $module = Permission::where('name', $permission)->first();
                $mod = ($module) ? $module->module : '';
                $listPermission = "list-$mod";
                if (str_contains($permission, 'edit') || str_contains($permission, 'delete')) {

                    if (!in_array($listPermission, $permissions)) {
                        return  CommonHelper::jsonResponseWeb(400, '', ["permissions.{$listPermission}" => ["You must assign the list permission of module {$mod} before assigning its related permission."]]);
                    }
                }

                if ($permission == 'status-update-order') {
                    // Enforce dependency on both view and list permissions
                    if (!in_array('view-order-detail-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.view-order-detail-order" => ["You must assign 'view order detail' and 'order list' permission before assigning 'status update' permission."]
                        ]);
                    }

                    if (!in_array('list-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.list-order" => ["You must assign 'list' permission before assigning 'status update' permission."]
                        ]);
                    }
                }

                if ($permission == 'view-order-detail-order') {
                    // Enforce dependency on list permission only
                    if (!in_array('list-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.list-order" => ["You must assign 'list' permission before assigning 'view order detail' permission."]
                        ]);
                    }
                }
            }


            $department = Department::findOrFail($request->department_id);
            //get default permissions
            $defaultPermissions = config("role_defaults.{$department->id}", []);

            $permissions =  array_unique(array_merge($permissions, $defaultPermissions));
            $role =  Role::create([
                'name' => $request->name . ' - ' . $department->name,
                'actual_name' => $request->name,
                'department_id' => $request->department_id,
            ]);
            //assign permissions one by one

            if (!empty($request->permissions)) {
                $role->syncPermissions($permissions);
            }

            DB::commit();
            $message = 'Role added successfully.';
            $extra = ['redirect' => route('admin.role.index')];
            return CommonHelper::jsonResponseWeb(200, 'Role added successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function edit($id)
    {
        $role = Role::with('department')->findOrFail($id);
        if ($role->name === "superadmin") {
            return redirect()->route('admin.role.index')->with('error', 'You are not allowed to edit this role.');
        }
        $departments = Department::orderBy('name', 'asc')->get();

        $defaultPermissions = (array) config('role_defaults.' . $role->department_id, []);
        $hasPermissions = $role->permissions->pluck('name');
        $permissions = Permission::orderBy('module', 'ASC')->get()->groupBy('module');
        return view('admin.roles.edit', [
            'defaultPermissions' => $defaultPermissions,
            'hasPermissions' => $hasPermissions,
            'permissions' => $permissions,
            'role' => $role,
            'departments' => $departments,
        ]);
    }
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        // Find the role to update
        $role = Role::findOrFail($id);

        // Validate request, exclude current role name for unique validation
        $validation = Validator::make($request->all(), [
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('roles', 'name')->ignore($role->id),
            ],
            'department_id' => 'required|exists:departments,id',
            'permissions' => 'required|array|min:1',
            'permissions.*' => 'string|exists:permissions,name',
        ]);

        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }

        try {
            $permissions = $request->input('permissions', []);
            foreach ($permissions as $permission) {
                $module = Permission::where('name', $permission)->first();
                $mod = ($module) ? $module->module : '';
                $listPermission = "list-$mod";
                if (str_contains($permission, 'edit') || str_contains($permission, 'delete')) {

                    if (!in_array($listPermission, $permissions)) {
                        return  CommonHelper::jsonResponseWeb(400, '', ["permissions.{$listPermission}" => ["You must assign the list permission of module {$mod} before assigning its related permission."]]);
                    }
                }

                if ($permission == 'status-update-order') {
                    // Enforce dependency on both view and list permissions
                    if (!in_array('view-order-detail-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.view-order-detail-order" => ["You must assign 'view order detail' and 'order list' permission before assigning 'status update' permission."]
                        ]);
                    }

                    if (!in_array('list-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.list-order" => ["You must assign 'list' permission before assigning 'status update' permission."]
                        ]);
                    }
                }

                if ($permission == 'view-order-detail-order') {
                    // Enforce dependency on list permission only
                    if (!in_array('list-order', $permissions)) {
                        return CommonHelper::jsonResponseWeb(400, '', [
                            "permissions.list-order" => ["You must assign 'list' permission before assigning 'view order detail' permission."]
                        ]);
                    }
                }
            }
            $department = Department::findOrFail($request->department_id);
            $name = $role->name;
            $actual_name = $role->actual_name;
            if ($request->filled('name') && $request->name !== $role->name) {
                $name =  $request->name . ' - ' . $department->name;
                $actual_name =  $request->name;
            }
            $role->update([
                'name' => $name,
                'actual_name' => $actual_name,
                'department_id' => $request->department_id,
            ]);

            $defaultPermissions = (array) config('role_defaults.' . $department->id, []);
            $permissions = array_unique(array_merge($permissions, $defaultPermissions));
            $role->syncPermissions($permissions);

            DB::commit();

            return CommonHelper::jsonResponseWeb(200, 'Role updated successfully', [], [
                'redirect' => route('admin.role.index'),
            ]);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $role = Role::find($id);
            $message = "";
            if (empty($role)) {
                $message = "Invalid Role";

                return CommonHelper::jsonResponseWeb(400, $message);
            }
            if ($role->name === "superadmin") {


                return CommonHelper::jsonResponseWeb(400, 'You are not allowed to delete this role');
            }
            $role->delete();
            DB::commit();
            $message = 'Role deleted successfully.';
            return CommonHelper::jsonResponseWeb(200, $message, [], [
                'redirect' => route('admin.role.index'),
            ]);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
