<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductImage;
use App\Models\ProductStock;
use App\Models\Setting;
use App\Models\VehicleMake;
use App\Models\VehicleModel;
use App\Models\VehicleType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-product')->only('index');
        $this->middleware('permission:show-product')->only('show');
        $this->middleware('permission:add-product')->only(['create', 'store', 'getVehicleMakes', 'getVehicleModels']);
        $this->middleware('permission:edit-product')->only(['edit', 'update', 'getVehicleMakes', 'getVehicleModels']);
        $this->middleware('permission:delete-product')->only(['edit', 'update', 'deleteImage']);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $products = (new Product())->getPaginatedData($request);
        $vehicleModels = VehicleModel::where('status', 1)->get();
        $vehicleMakes = VehicleMake::where('status', 1)->get();
        $vehicleTypes = VehicleType::where('status', 1)->get();
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.product.pagination', compact('products', 'vehicleModels', 'vehicleMakes', 'vehicleTypes'))->render(),
            ]);
        }
        return view('admin.product.index', compact('products', 'vehicleModels', 'vehicleMakes', 'vehicleTypes'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $vehicleModels = VehicleModel::with(['vehicleType', 'vehicleMake'])
            ->where('status', 1)
            ->whereHas('vehicleType', function ($query) {
                $query->where('status', 1);
            })
            ->whereHas('vehicleMake', function ($query) {
                $query->where('status', 1);
            })
            ->get();
        $currrncyRate = Setting::getValue(['usd_to_mmk', 'usd_to_rmb']);
        $currrncyRateMmk = $currrncyRate['usd_to_mmk'] ?? null;
        $currrncyRateRmb = $currrncyRate['usd_to_rmb'] ?? null;
        return view('admin.product.create', compact('vehicleModels', 'currrncyRateMmk', 'currrncyRateRmb'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'min:3|max:200|required|string',
            'vehicle_model' => 'required|array',
            'vehicle_model.*' => 'exists:vehicle_models,id',
            'feature_image' => 'required|mimes:jpg,jpeg,webp,gif,png|max:2048',
            'price_in_usd' => 'required|numeric|min:0',
            'price_in_mmk' => 'required|numeric|min:0',
            'price_in_rmb' => 'required|numeric|min:0',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpg,jpeg,webp,gif,png|max:2048',
            'description' => 'nullable|string',
            'status' => 'required|in:0,1',
            'is_featured' => 'nullable|boolean'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            $currrncyRate = Setting::getValue(['usd_to_mmk', 'usd_to_rmb']);
            $currrncyRateMmk = $currrncyRate['usd_to_mmk'];
            $currrncyRateRmb = $currrncyRate['usd_to_rmb'];
            $priceInUsd = $request->price_in_usd;
            $priceInMmk = $request->price_in_mmk;
            $priceInRmb = $request->price_in_rmb;

            $priceInUsd = $priceInMmk / $currrncyRateMmk;
            $priceInMmk = $priceInUsd * $currrncyRateMmk;
            $priceInRmb = $priceInUsd * $currrncyRateRmb;
            //dd($priceInMmk);
            $product = Product::create([
                'name' => $request->name,
                'price_in_usd' => $priceInUsd,
                'price_in_mmk' => $priceInMmk,
                'price_in_rmb' => $priceInRmb,
                'description' => $request->description,
                'status' => $request->status,
                'is_featured' => $request->has('is_featured')
            ]);
            //model entry
            $product->compatibleModels()->attach($request->vehicle_model);
            //featured image entry
            if ($request->hasFile('feature_image')) {
                $imagePath = CommonHelper::fileUpload($request->file('feature_image'), PRODUCT_PATH);
                $product->images()->create([
                    'image' => $imagePath,
                    'product_id' => $product->id,
                    'is_featured' => 1
                ]);
            }

            //images entry
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $imagePath = CommonHelper::fileUpload($image, PRODUCT_PATH);
                    $product->images()->create([
                        'image' => $imagePath,
                        'product_id' => $product->id,
                        'is_featured' => 0
                    ]);
                }
            }

            DB::commit();
            $extra = [
                'redirect' => route('admin.product.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Product added successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $product = Product::with(['images', 'compatibleModels', 'stock'])->findOrFail($id);
        $assignedModelIds = $product->compatibleModels->pluck('id')->toArray();
        $currrncyRate = Setting::getValue(['usd_to_mmk', 'usd_to_rmb']);
        $currrncyRateMmk = $currrncyRate['usd_to_mmk'] ?? null;
        $currrncyRateRmb = $currrncyRate['usd_to_rmb'] ?? null;


        $vehicleModels = VehicleModel::with(['vehicleType', 'vehicleMake'])
            ->where(function ($query) use ($assignedModelIds) {
                $query->where(function ($subQuery) {
                    $subQuery->where('status', 1)
                        ->whereHas('vehicleType', fn($q) => $q->where('status', 1))
                        ->whereHas('vehicleMake', fn($q) => $q->where('status', 1));
                })
                    ->orWhereIn('id', $assignedModelIds);
            })
            ->get();

        return view('admin.product.edit', compact('product', 'vehicleModels',  'currrncyRateMmk', 'currrncyRateRmb'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'min:3|max:200|required|string',
            'vehicle_model' => 'required|array',
            'vehicle_model.*' => 'exists:vehicle_models,id',
            'price_in_usd' => 'required|numeric|min:0',
            'price_in_mmk' => 'required|numeric|min:0',
            'price_in_rmb' => 'required|numeric|min:0',
            'feature_image' => 'nullable|mimes:jpg,jpeg,webp,gif,png|max:2048',
            'images' => 'nullable|array|min:1',
            'images.*' => 'image|mimes:jpg,jpeg,webp,gif,png|max:2048',
            'descirption' => 'nullable|string',
            'status' => 'required|in:0,1',
            'is_featured' => 'nullable|boolean'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            $product = Product::findOrFail($id);
            $priceInUsd = $product->price_in_usd;
            $priceInMmk = $product->price_in_mmk;
            if ($request->has('price_in_usd') || $request->has('price_in_mmk') || $request->has('price_in_rmb')) {
                $currrncyRate = Setting::getValue(['usd_to_mmk', 'usd_to_rmb']);
                $currrncyRateMmk = $currrncyRate['usd_to_mmk'];
                $currrncyRateRmb = $currrncyRate['usd_to_rmb'];
                $priceInUsd = $request->price_in_usd;
                $priceInMmk = $request->price_in_mmk;
                $priceInRmb = $request->price_in_rmb;

                $priceInUsd = $priceInMmk / $currrncyRateMmk;
                $priceInMmk = $priceInUsd * $currrncyRateMmk;
                $priceInRmb = $priceInUsd * $currrncyRateRmb;
            }
            $product->update([
                'name' => $request->name,
                'price_in_usd' => $priceInUsd,
                'price_in_mmk' => $priceInMmk,
                'price_in_rmb' => $priceInRmb,
                'description' => $request->description,
                'status' => $request->status,
                'is_featured' => $request->has('is_featured')
            ]);
            if ($request->hasFile('feature_image')) {
                $existingFeatured = $product->featuredImage()->first();
                if ($existingFeatured) {
                    //dd('hrere');
                    $imagePath = CommonHelper::fileUpload($request->file('feature_image'), PRODUCT_PATH, $product->image);
                    $existingFeatured->update([
                        'image' => $imagePath,
                        'is_featured' => 1,
                    ]);
                } else {
                    $imagePath = CommonHelper::fileUpload($request->file('feature_image'), PRODUCT_PATH);
                    $product->images()->create([
                        'image' => $imagePath,
                        'is_featured' => 1
                    ]);
                }
            }
            if ($request->hasFile('images')) {
                foreach (($request->file('images')) as $image) {
                    $imagePath = CommonHelper::fileUpload($image, PRODUCT_PATH);
                    ProductImage::create([
                        'product_id' => $product->id,
                        'image' => $imagePath,
                        'is_featured' => 0
                    ]);
                }
            }
            $product->compatibleModels()->sync($request->vehicle_model);
            //ProductStock::where('product_id', $product->id)->update(['quantity' => $request->stock,]);
            DB::commit();
            $extra = ['redirect' => route('admin.product.index')];
            return CommonHelper::jsonResponseWeb(200, 'Product updated successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction();
            $product = Product::findOrFail($id);
            foreach ($product->images as $image) {
                if (file_exists(public_path($image->image))) {
                    unlink(public_path($image->image));
                }
                $image->delete();
            }
            //$product->compatibleModels()->detach();
            $product->update(['status' => 2]);
            $product->delete();
            DB::commit();
            $extra = [
                'redirect' => route('admin.product.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Product deleted successfully.', [], $extra);
        } catch (QueryException $e) {
            $msg = $e->getMessage();
            if (strpos($msg, '1451') !== false) { // fk check
                $message = 'Please delete entries first from its relevant sections';
                return response()->json(['status' => false, 'status_code' => 500, 'message' => $message], 500);
            }
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function deleteImage($id)
    {
        $image = ProductImage::findOrFail($id);
        if (file_exists(public_path($image->image))) {
            unlink(public_path($image->image));
        }
        $image->delete();
        return response()->json(['message' => 'Image deleted successfully']);
    }
    //filter functions
    public function getVehicleMakes(Request $request)
    {

        $typeId = $request->parent_id;
        $vehicleTypes = VehicleType::with('makes')->find($typeId);
        if (!$vehicleTypes) {
            return response()->json([
                'data' => []
            ]);
        }
        $makes = $vehicleTypes->makes()->get();

        return response()->json([
            'data' => $makes
        ]);
    }
    public function getVehicleModels(Request $request)
    {
        $makeId = $request->parent_id;

        $makes = VehicleMake::with('models')->find($makeId);

        if (!$makes) {
            return response()->json([
                'data' => [],
                'message' => 'Make not found'
            ]);
        }

        $models = $makes->models()->get();

        if ($models->isEmpty()) {
            Log::info('No models found for make ID: ' . $makeId);
        } else {
            Log::info('Models found:', $models->toArray());
        }

        return response()->json([
            'data' => $models
        ]);
    }
    //manage inventory
    public function manageInventory($id)
    {

        $product = Product::findOrFail($id);
        return view('admin.product.manage-inventory', compact('product'));
    }
    public function updateInventory(Request $request, $id)
    {
        DB::beginTransaction();
        $validation =  Validator::make($request->all(), [
            'type' => 'required|in:in,out',
            'quantity' => 'required|integer',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        //check stock available more than out entry
        if ($request->type === 'out') {
            if (!Product::validateAvailableStock($id, $request->quantity)) {
                return CommonHelper::jsonResponseWeb(400, 'Not enough stock available');
            }
        }



        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        //make entry
        $inventory = ProductStock::create([
            'product_id' => $id,
            'type' => $request->type,
            'quantity' => $request->quantity,
            'comment' => $request->comment
        ]);

        //calculATE update product's in hand stock
        Product::updateStockInHand($id);
        DB::commit();
        $extra = [
            'redirect' => route('admin.product.edit', $id)
        ];
        return CommonHelper::jsonResponseWeb(200, 'Inventory updated successfully.', [], $extra);
        try {
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    // show product
    public function show($id)
    {
        $product = Product::with(['images', 'compatibleModels', 'stock'])->findOrFail($id);
        $assignedModelIds = $product->compatibleModels->pluck('id')->toArray();
        $vehicleModels = VehicleModel::with(['vehicleType', 'vehicleMake'])
            ->where(function ($query) use ($assignedModelIds) {
                $query->where(function ($subQuery) {
                    $subQuery->where('status', 1)
                        ->whereHas('vehicleType', fn($q) => $q->where('status', 1))
                        ->whereHas('vehicleMake', fn($q) => $q->where('status', 1));
                })
                    ->orWhereIn('id', $assignedModelIds);
            })
            ->get();

        return view('admin.product.show', compact('product', 'vehicleModels'));
    }
}
