<?php

namespace App\Services;

use App\Models\Product;
use Illuminate\Support\Facades\DB;

class ProductService
{
    public function getLatestProducts($limit = null, $pagination = null, $filters = null)
    {

        $query = Product::with(['compatibleModels.vehicleMake', 'compatibleModels.vehicleType', 'featuredImage'])->where('status', 1);
        if ($limit) {
            $query->limit($limit);
        }
        if ($pagination && isset($pagination['skip'], $pagination['take'])) {
            $query->skip($pagination['skip'])->take($pagination['take']);
        }
        //aplly sorting and filter
        if ($filters) {
            $query = $this->applyFilters($query, $filters);
        } else {
            $query->orderBy('created_at', 'desc');
        }


        return $query->get();
    }
    public function getFeaturedProducts($limit = null, $pagination = null, $filters = null)
    {
        $query = Product::with(['compatibleModels.vehicleMake', 'compatibleModels.vehicleType', 'featuredImage'])->where('is_featured', 1)->where('status', 1);
        if ($limit) {
            $query->limit($limit);
        }
        if ($pagination && isset($pagination['skip'], $pagination['take'])) {
            $query->skip($pagination['skip'])->take($pagination['take']);
        }
        //aplly sorting and filter
        if ($filters) {
            $query = $this->applyFilters($query, $filters);
        } else {
            $query->orderBy('created_at', 'desc');
        }
        return $query->get();
    }
    public function getTopSellingProducts($limit = null, $pagination = null, $filters = null)
    {
        $query = Product::with(['compatibleModels.vehicleMake', 'compatibleModels.vehicleType', 'featuredImage'])->selectRaw('products.*,(SELECT COUNT(od.id) from order_details od where od.product_id = products.id) as total_sold')
            ->where('products.status', 1)
            ->orderBy('total_sold', 'desc');
        if ($limit) {
            $query->limit($limit);
        }
        if ($pagination && isset($pagination['skip'], $pagination['take'])) {
            $query->skip($pagination['skip'])->take($pagination['take']);
        }
        //aplly sorting and filter
        if ($filters) {
            $query = $this->applyFilters($query, $filters);
        }
        return $query->get();
    }
    //get all products
    public function getAllProducts($limit = null, $pagination = null, $filters = null)
    {
        $query = Product::with(['compatibleModels.vehicleMake', 'compatibleModels.vehicleType', 'featuredImage'])
            ->where('status', 1);

        if ($limit) {
            $query->limit($limit);
        }

        if ($pagination && isset($pagination['skip'], $pagination['take'])) {
            $query->skip($pagination['skip'])->take($pagination['take']);
        }

        if ($filters) {
            $query = $this->applyFilters($query, $filters);
        }

        return $query->get();
    }
    //function to get correct currency
    public function getCurrency(Product $product, string $currencyCode)
    {
        //dd($currencyCode);
        if (in_array($currencyCode, ['CNY', 'RMB'])) {
            $price = $product->price_in_rmb;
        } elseif (in_array($currencyCode, ['MMK'])) {
            $price = $product->price_in_mmk;
        } else {
            $price = $product->price_in_usd;
        }
        $outOfStock = (int) 0;
        if ($product->stock_in_hand == 0) {
            $outOfStock = (int) 1;
        }
        return [
            'id' => $product->id,
            'name' => $product->name,
            'description' => $product->description,
            'price' => round($price, 2),
            'price_formatted' => number_format($price, 2),
            'currency_symbol' => auth()->user()->currency_symbol,
            'featured_image' => $product->featuredImage?->image
                ? asset($product->featuredImage->image)
                : asset('assets/img/product_no_image.jpg'),
            'deleted_at' => $product->deleted_at,
            'status' => $product->status,
            'stock_in_hand' => $product->stock_in_hand,
            'out_of_stock' => $outOfStock,
            'rating' => number_format($product->rating, 1),
            //vehicle compatibility
            'vehicle_compatibility' => $product->compatibleModels->map(function ($model) {
                return [
                    'model' => $model->name,
                    'make'  => $model->vehicleMake?->name,
                    'type'  => $model->vehicleType?->name,

                ];
            })->toArray()

        ];
    }
    //this function use for get price of product customer's currenct wise
    public function getLatestProductsForCustomer(string $currencyCode, $limit = null, $pagination = null, $filters = null)
    {
        $products = $this->getLatestProducts($limit, $pagination, $filters);
        return  $products->map(function ($product) use ($currencyCode) {
            return   $this->getCurrency($product, $currencyCode);
        })->toArray();
    }
    public function getFeaturedProductsForCustomer(string $currencyCode, $limit = null, $pagination = null, $filters = null)
    {
        $products = $this->getFeaturedProducts($limit, $pagination, $filters);
        return $products->map(function ($product) use ($currencyCode) {
            return $this->getCurrency($product, $currencyCode);
        })->toArray();
    }
    public function getTopSellingProductsForCustomer(string $currencyCode, $limit = null, $pagination = null, $filters = null)
    {
        $products = $this->getTopSellingProducts($limit, $pagination, $filters);
        return $products->map(function ($product) use ($currencyCode) {
            return $this->getCurrency($product, $currencyCode);
        })->toArray();
    }
    public function getAllProductsForCustomer(string $currencyCode, $limit = null, $pagination = null, $filters = null)
    {
        $products = $this->getAllProducts($limit, $pagination, $filters);

        return $products->map(function ($product) use ($currencyCode) {
            return $this->getCurrency($product, $currencyCode);
        })->toArray();
    }
    public function applyFilters($query, $filters)
    {
        //search by keywrod
        if (!empty($filters['keyword'])) {
            $keyword = $filters['keyword'];
            $query->where('status', 1)
                ->where(function ($q) use ($keyword) {
                    $q->where('name', 'like', "{$keyword}%")
                        ->orwhere('description', 'like', "%{$keyword}%");
                });
        }
        // filter by vehicle type / make / model (combined in one whereHas)
        if (!empty($filters['vehicle_type_id']) || !empty($filters['vehicle_make_id']) || !empty($filters['vehicle_model_id'])) {
            $query->whereHas('compatibleModels', function ($q) use ($filters) {
                $q->where('status', 1);

                if (!empty($filters['vehicle_type_id'])) {
                    $q->where('vehicle_type_id', $filters['vehicle_type_id']);
                }

                if (!empty($filters['vehicle_make_id'])) {
                    $q->where('vehicle_make_id', $filters['vehicle_make_id']);
                }

                if (!empty($filters['vehicle_model_id'])) {
                    $q->where('id', $filters['vehicle_model_id']);
                }
            });
        }
        if (!empty($filters['sort_by'])) {

            switch ($filters['sort_by']) {
                case 'price_low_to_high':
                    $query->orderBy('price_in_usd', 'asc');
                    break;
                case 'price_high_to_low':
                    $query->orderBy('price_in_usd', 'desc');
                    break;
                case 'rating_low_to_high':
                    $query->orderBy('rating', 'asc');
                    break;
                case 'rating_high_to_low':
                    $query->orderBy('rating', 'desc');
                    break;
                case 'latest':
                default:
                    $query->orderBy('created_at', 'desc');
            }
        } else {
            // default sorting when sort_by not provided
            $query->orderBy('created_at', 'desc');
        }
        return $query;
    }
}
