<?php

namespace App\Services;

use App\Helpers\CommonHelper;
use App\Services\ProductService;
use App\Services\CashBackService;
use App\Models\Cart;
use App\Models\Product;
use Throwable;
use Illuminate\Support\Facades\DB;

class CartService
{


    protected $productService;
    protected $cashBackService;
    public function __construct(ProductService $productService, CashBackService $cashBackService)
    {
        $this->productService = $productService;
        $this->cashBackService = $cashBackService;
    }
    public function getCartOverview($customer)
    {
        $customerId = $customer->id;
        $currencyCode = $customer->currency;
        $currencySymbol = $customer->currency_symbol;
        try {
            //get out of stock, deleted, inactive products and display warning
            $removedItems = Cart::with('product')
                ->where('customer_id', $customerId)
                ->whereHas('product', function ($q) {
                    $q->where(function ($q2) {
                        $q2->where('status', '!=', 1)
                            ->orWhereNotNull('deleted_at')
                            ->orWhere('stock_in_hand', '<=', 0);
                    });
                })
                ->get()
                ->map(function ($item) {
                    $reason = null;
                    $isInactive = 0;
                    $isDeleted = 0;
                    $outOfStock  = 0;
                    if ($item->product->deleted_at) {
                        $isDeleted = 1;
                        $reason = __('product_deleted');
                    } elseif ($item->product->status == 0) {
                        $isInactive = 1;
                        $reason = __('product_inactive');
                    } elseif ($item->product->stock_in_hand <= 0) {
                        $outOfStock  = 1;
                        $reason = __('out of stock', ['product' => $item->product->name]);
                    }

                    return [
                        'product_id' => $item->product->id,
                        'product_name' => $item->product->name,
                        'reason' => $reason,
                        'is_deleted' => $isDeleted,
                        'is_inactive' => $isInactive,
                        'out_of_stock' => $outOfStock,
                    ];
                });

            //fetch all porducts according to cart but exclude inactive, out of stock and inactive

            $cartItems = Cart::with('product')
                ->where('customer_id', $customerId)
                ->whereHas('product', function ($q) {
                    $q->where('status', 1)->whereNull('deleted_at')->where('stock_in_hand', '>', 0);
                })
                ->get();

            $emptyCart = false;
            if ($cartItems->isEmpty()) {
                $emptyCart = true;
            }

            //fecth product price column data according to currecny
            $cartData = $cartItems->map(function ($item) use ($currencyCode) {
                $productData = $this->productService->getCurrency($item->product, $currencyCode);
                $subtoatl = $productData['price'] * $item->quantity;
                $warning     = [];
                //$outOfStock  = 0;
                $lessStock   = 0;
                $priceChange = 0;
                $finalQty = $item->quantity;
                /*if ($productData['stock_in_hand'] <= 0) {
                    $finalQty = 0;
                    $outOfStock  = 1;
                    $warning[] =  __('only items available', [
                        'stock' => $productData['stock_in_hand'],
                        'product' => $productData['name']
                    ]);
                }*/
                if ($item->quantity > $productData['stock_in_hand']) {
                    $lessStock   = 1;
                    $finalQty = $productData['stock_in_hand'];
                    $warning[] =  __('only items available', [
                        'stock' => $productData['stock_in_hand'],
                        'product' => $productData['name']
                    ]);
                }

                if ($item->price != $productData['price']) {
                    $priceChange = 1;
                    $warning[] = __('price changed', ['product' => $productData['name']]);
                }


                return [
                    'product_id' => $productData['id'],
                    'product_name' => $productData['name'],
                    'product_image' => $productData['featured_image'],
                    'unit_price' => $productData['price'],
                    'unit_price_formatted' => number_format($productData['price'], 2),
                    'currency_symbol' => $productData['currency_symbol'],
                    'requested_quantity' => $item->quantity,
                    'final_quantity' => $finalQty,
                    'available_stock' => $productData['stock_in_hand'],
                    'status' => $productData['status'],
                    'deleted_at' => $productData['deleted_at'],
                    //'out_of_stock'    => $outOfStock,
                    'less_stock'      => $lessStock,
                    'price_changed'   => $priceChange,
                    'warning' => $warning,
                    'sub_total' => $subtoatl,
                    'sub_total_formatted' => number_format($subtoatl, 2),

                ];
            });

            //prepare summary for payment
            $totalProductAmount = (float)$cartData->sum('sub_total');
            $summary = CommonHelper::taxCalculate($totalProductAmount);

            //fetch wholesale paln if not active any then apply basic one
            //$appliedPlan = $customer->activePlanWholesale ??  WholesaleIdPlan::where('is_default', 1)->first();
            $paymentSummeryData = [
                'product_cost' => $totalProductAmount,
                'product_cost_formatted' => number_format($totalProductAmount, 2),
                'tax_rate' => $summary['tax_rate'],
                'tax_amount' => $summary['tax_amount'],
                'tax_amount_formatted' => number_format($summary['tax_amount'], 2),
                'order_total' => $summary['total_with_tax'],
                'order_total_formatted' => number_format($summary['total_with_tax']),
                'currency_symbol' => $currencySymbol,

            ];

            //get active plan info
            $cashback = $customer->activePlanWholesale;
            DB::commit();
            $cashBackData = [
                'cash_back_paln_id' => $customer->wholesale_plan_id,
                'cash_back_title' => $cashback->title,
                'is_default' => (int) optional($cashback->originPaln)->is_default,

            ];

            $flags = [
                'has_out_of_stock' => $removedItems->contains('out_of_stock', 1),
                'has_deleted' => $removedItems->contains('is_deleted', 1),
                'has_inactive' => $removedItems->contains('is_inactive', 1),
                'has_less_stock'   => $cartData->contains('less_stock', 1),
                'has_price_change' => $cartData->contains('price_changed', 1),
            ];
            return [
                'empty_cart' => $emptyCart,
                'products_data' => $cartData,
                'removed_items' => $removedItems,
                'payment_summary' => $paymentSummeryData,
                'cash_back_detail' => $cashBackData,
                'flags' => $flags,
            ];
        } catch (Throwable $e) {
            \Log::error('CartService getCartOverview error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
