<?php

namespace App\Models;

use App\Helpers\NotificationHelper;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use SoftDeletes;
    use HasFactory;
    protected $fillable = [
        'name',
        'stock_in_hand',
        'price_in_usd',
        'price_in_mmk',
        'price_in_rmb',
        'rating',
        'description',
        'status',
        'is_featured'

    ];
    public function images()
    {
        return $this->hasMany(ProductImage::class)->where('is_featured', 0);
    }
    public function reviews()
    {
        return $this->hasMany(ProductReview::class);
    }
    public function compatibleModels()
    {
        return $this->belongsToMany(VehicleModel::class, 'product_vehicle_compatibility');
    }
    public function featuredImage()
    {
        return $this->hasOne(ProductImage::class)->where('is_featured', 1);
    }
    public function carts()
    {
        return $this->hasMany(Cart::class, 'product_id');
    }
    public function stock()
    {
        return $this->hasMany(ProductStock::class);
    }
    /*public function getPaginatedData($request)
    {
        $query = self::withTrashed();

        $perPage = $request->input('per_page', 10);
        $sortBy = $request->get('sort_by');
        $allowedSortColumns = ['id', 'name', 'status', 'created_at']; // add only valid DB columns here

        // Default to 'products.id' if invalid or empty
        $sortBy = in_array($sortBy, $allowedSortColumns) ? 'products.' . $sortBy : 'products.id';

        $sortOrder = $request->get('sort_order', 'desc');
        $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? $sortOrder : 'desc';

        if ($request->has('keyword') && $request->get('keyword')) {
            $keyword = $request->keyword;
            $query->where(function ($q) use ($keyword) {
                $q->where('name', 'like', "%$keyword%")
                    ->orWhere('status', 'like', "%$keyword%")
                    ->orWhere('id', 'like', "%$keyword%");
            });
        }

        if ($request->has('status') && $request->get('status') != '') {
            $query->where('status', $request->status);
        }


        return $query->orderBy($sortBy, $sortOrder)->paginate($perPage);
    }*/
    public function getPaginatedData($request)
    {
        $query = self::withTrashed()->whereIn('status', [0, 1]);

        $perPage = $request->input('per_page', 10);
        $sortBy = $request->get('sort_by');
        $allowedSortColumns = ['id', 'name', 'status', 'created_at'];
        $sortBy = in_array($sortBy, $allowedSortColumns) ? 'products.' . $sortBy : 'products.id';

        $sortOrder = $request->get('sort_order', 'desc');
        $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? $sortOrder : 'desc';

        if ($request->filled('keyword')) {
            $keyword = $request->keyword;
            $query->where(function ($q) use ($keyword) {
                $q->where('name', 'like', "%$keyword%")
                    ->orWhere('status', 'like', "%$keyword%")
                    ->orWhere('id', 'like', "%$keyword%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Vehicle Type filter
        if ($request->filled('vehicle_type')) {
            $query->whereHas('compatibleModels', function ($q) use ($request) {
                $q->where('vehicle_type_id', $request->vehicle_type);
            });
        }

        // Vehicle Make filter
        if ($request->filled('vehicle_make')) {
            $query->whereHas('compatibleModels', function ($q) use ($request) {
                $q->where('vehicle_make_id', $request->vehicle_make);
            });
        }

        // Vehicle Model filter
        if ($request->filled('vehicle_model')) {
            $query->whereHas('compatibleModels', function ($q) use ($request) {
                $q->where('id', $request->vehicle_model);
            });
        }

        return $query->orderBy($sortBy, $sortOrder)->paginate($perPage);
    }
    //update stock in hand
    public static function updateStockInHand($productId)
    {
        $stockInHand = ProductStock::calculateInHandStock($productId);
        self::where('id', $productId)->update(['stock_in_hand' => $stockInHand]);

        // notify admin if stock is 0
        if ($stockInHand <= 0) {
            // fetch product details
            $product = self::find($productId);
            if (!$product) {
                return; // safety check
            }
            $title = __('out_of_stock');
            $message = __('product_out_of_stock', ['product_name' => $product->name]);
            NotificationHelper::saveNotification(
                null,
                'system',
                1,
                ADMIN,
                OUT_OF_STOCK,
                $title,
                $message,
                ['product_id', $productId],
                0,
            );
            //fire event
            event(new \App\Events\AdminNotificationEvent([
                'message' => $message,
                'title'   => $title,
                'url' => route('admin.product.manageInventory', [$productId])
            ]));
        }
    }
    //validate available stock
    public static function validateAvailableStock($productId, $reqQuantity)
    {
        $product = self::find($productId);
        return $product && $product->stock_in_hand >= $reqQuantity;
    }
}
