<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use \Carbon\Carbon;
use libphonenumber\PhoneNumberUtil;
use libphonenumber\NumberParseException;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Customer extends Authenticatable
{
    use HasFactory, HasApiTokens, Notifiable, SoftDeletes;
    protected $table = "customers";
    protected $fillable = [
        'full_name',
        'email',
        'mobile_number',
        'dial_code',
        'dial_code_iso',
        'currency',
        'currency_name',
        'currency_symbol',
        'timezone',
        'is_email_verified',
        'is_mobile_no_verified',
        'password',
        'address',
        'status',
        'description',
        'image',
        'email_otp',
        'wholesale_plan_id',
        'is_profile_updated',
        'fcm_token'
    ];
    protected $hidden = [
        'password',
    ];
    protected $casts = [
        'mobile_verified_at',
        'email_verified_at',
    ];

    public function orders()
    {
        return $this->hasMany(order::class);
    }
    public function addresses()
    {
        return $this->hasMany(CustomerAddress::class);
    }
    public function wholesalePlans()
    {
        return $this->hasMany(CustomerWholesalePlan::class)->orderByRaw("
        FIELD(status, 'active') DESC, end_date DESC
    ");
    }
    public function activePlanWholesale()
    {
        return $this->hasOne(CustomerWholesalePlan::class)->where('status', 'active');
    }
    public function carts()
    {
        return $this->hasMany(Cart::class);
    }
    public function conversation()
    {
        return $this->hasMany(Conversation::class);
    }
    protected $appends = ['avatar_url', 'name'];
    public function getAvatarUrlAttribute()
    {
        // Use asset if image exists and file exists, otherwise fallback
        if ($this->image && file_exists(public_path($this->image))) {
            return asset($this->image);
        }

        return asset(config('constants.default_avatar'));
    }
    public function getNameAttribute()
    {
        return $this->full_name; // map full_name → name
    }
    public function getPaginatedData($request)
    {
        $query = self::with('activePlanWholesale')
            ->leftJoin('wholesale_id_plans', 'customers.wholesale_plan_id', '=', 'wholesale_id_plans.id')
            ->whereIn('customers.status', [0, 1])
            ->where('is_email_verified', 1)
            ->where('is_mobile_no_verified', 1)
            ->select('customers.*');

        $perPage = $request->input('per_page', 10);
        $sortBy = $request->get('sort_by');
        $allowedSortColumns = ['id', 'full_name', 'email', 'mobile_number', 'status', 'created_at',];
        $sortBy = in_array($sortBy, $allowedSortColumns) ? 'customers.' . $sortBy : 'customers.id';

        $sortOrder = $request->get('sort_order', 'desc');
        $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? $sortOrder : 'desc';

        if ($request->filled('keyword')) {
            $keyword = $request->keyword;
            $query->where(function ($q) use ($keyword) {
                $q->where('customers.full_name', 'like', "%$keyword%")
                    ->orWhere('customers.email', 'like', "%$keyword%")
                    ->orWhere('customers.mobile_number', 'like', "%$keyword%")
                    ->orWhere('customers.status', 'like', "%$keyword%")
                    ->orWhere('customers.id', 'like', "%$keyword%")
                    ->orWhere('wholesale_id_plans.title', 'like', "%$keyword%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('plans_filter')) {
            $query->where('wholesale_plan_id', $request->plans_filter);
        }

        return $query->orderBy($sortBy, $sortOrder)->paginate($perPage);
    }
    public static function checkAndExpireMyWholesalePlan()
    {
        return CustomerWholesalePlan::where('status', 'active')->whereDate('end_date', '<', Carbon::now())->update(['status' => 'expired']);
    }
    //invalid mobile number validation
    public static function validateMobileNumber($dialCode, $dialCodeISO,  $mobileNumber)
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        try {
            $fullMobileNumber = $dialCode . $mobileNumber;
            $swissNumberProto = $phoneUtil->parse($fullMobileNumber, $dialCodeISO);
            $isValid = $phoneUtil->isValidNumber($swissNumberProto);
            if ($isValid) {
                return true;
            }
        } catch (NumberParseException $e) {
            return false;
        }
        return false;
    }
    //exist whole mobile number
    public static function existMobileNumber($dial_code, $dial_code_iso, $mobile_number, $skip_id = null)
    {
        $result = self::where('dial_code', $dial_code)
            ->where('dial_code_iso', $dial_code_iso)
            ->where('mobile_number', $mobile_number);

        //edit case
        if ($skip_id) {
            $result = $result->where('id', '!=', $skip_id);
        }

        return $result->first();
    }
    //exist whole email
    public static function existEmailId($email)
    {
        return self::where('email', $email)->withoutTrashed()->first();
    }
    //check current fcm , if exist in others records then empty those records's fcm
    public static function checkCurrentFcm($fcmToken, $currentCustomerId)
    {
        $fcmToken = trim($fcmToken);
        self::withTrashed()->where('fcm_token', $fcmToken)->where('id', '!=', $currentCustomerId)->update(['fcm_token' => '']);
        return true;
    }
}
