<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\BillingAddress;
use App\Models\Cart;
use App\Models\CheckoutAddress;
use App\Models\order;
use App\Models\OrderCancellation;
use App\Models\orderDetail;
use App\Models\OrderReturn;
use App\Models\OrderStatusLog;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\Setting;
use App\Models\ShippingAddress;
use App\Services\CartService;
use App\Services\ProductService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Support\Str;
use Illuminate\Validation\Validator as ValidationValidator;
use App\Helpers\NotificationHelper;
use App\Helpers\OrderNotificationHelper;
use App\Models\ProductReview;
use App\Models\Transaction;
use App\Models\User;

class OrderController extends Controller
{
    protected $productService;
    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }
    public function placeOrder(Request $request, CartService $cartService)
    {
        DB::beginTransaction();
        try {
            //fetch customer detail
            $customer = auth()->user();
            $customerId = $customer->id;
            $customerCurrency = $customer->currency;
            $customerSymbol = $customer->currency_symbol;

            //fetch cart detail - original cart count
            $customer = auth()->user();
            $cartOverview =  $cartService->getCartOverview($customer);

            if ($cartOverview['empty_cart']) {
                return CommonHelper::apiResponse(400, __('cart empty'));
            }

            if (
                $cartOverview['flags']['has_out_of_stock']
                || $cartOverview['flags']['has_deleted']
                || $cartOverview['flags']['has_inactive']
                || $cartOverview['flags']['has_price_change']
                || $cartOverview['flags']['has_less_stock']
            ) {

                return CommonHelper::apiResponse(
                    400,
                    __('Your cart has changed. Please review your order before placing.'),

                );
            }

            $currentConverionRate = CommonHelper::getCurrentConversionRate($customerCurrency);
            $customOrderNumber =  substr(time() . mt_rand(10, 99), -10);
            $orderStatus = 0; // Pending
            $paymentStatus = 0; // Unpaid
            $orderStatusInText = CommonHelper::$orderSatuses[0] ?? 'unknown';
            $paymentStatusInText =  CommonHelper::paymentStatusText(0);
            //save orders in order table
            $order = order::create([
                'customer_id' => $customerId,
                'order_number' => $customOrderNumber,
                'order_status' => $orderStatus,
                'payment_status' => $paymentStatus,
                'subtotal' => $cartOverview['payment_summary']['product_cost'],
                'total_price' =>  $cartOverview['payment_summary']['order_total'],
                'currency_code' => $customerCurrency,
                'conversion_rate' => $currentConverionRate,
                'tax_percent' => $cartOverview['payment_summary']['tax_rate'],
                'tax_amount' => $cartOverview['payment_summary']['tax_amount'],
                'total_items' => collect($cartOverview['products_data'])->sum('final_quantity'),
                'refund_status'       => 0,
                'notes' => $request->notes,


            ]);

            //order data
            $orderData = [
                'order_id' => $order->id,
                'order_number' => $customOrderNumber,
                'order_status' => $orderStatusInText,
                'payment_status_key' => $order->payment_status_data['key'],
                'payment_status_label' => $order->payment_status_data['label'],
                'product_cost' => $cartOverview['payment_summary']['product_cost'],
                'product_cost_formatted' => number_format($cartOverview['payment_summary']['product_cost'], 2),
                'tax_rate' =>  $cartOverview['payment_summary']['tax_rate'],
                'tax_amount' => $cartOverview['payment_summary']['tax_amount'],
                'tax_amount_formatted' => number_format($cartOverview['payment_summary']['tax_amount'], 2),
                'order_total' =>  $cartOverview['payment_summary']['order_total'],
                'order_total_formatted' => number_format($cartOverview['payment_summary']['order_total'], 2),
                'currency_symbol' => $customerSymbol,
                'ready_for_payment' => (int) 0
            ];

            $addresses = CommonHelper::getCheckoutAddresses($customer->id);
            $shippingTemp = $addresses['shipping'];
            //billing validation
             $rules = [
            'billing.full_name'     => 'required|string|max:255',
            'billing.mobile_number' => 'required|string|max:20|digits_between:6,14',
            'billing.address' => 'required|string|max:255',
            'billing.city_id'          => 'required|integer|exists:cities,id',
            'billing.state_id'         => 'required|integer|exists:states,id',
            'billing.country_id'       => 'required|integer|exists:countries,id',
            'billing.city'          => 'required|string|max:100',
            'billing.state'         => 'required|string|max:100',
            'billing.country'       => 'required|string|max:100',
            'billing.dial_code'       => 'required|regex:/^\+\d{1,4}$/',
        ];
        //for billing pincode
        $isBillingPincodeOptional = CommonHelper::isPincodeOptional($request->billing['country_id'] ?? null);
        $rules['billing.pincode'] = ($isBillingPincodeOptional ? 'nullable' : 'required') . '|string|max:20';

        $attributes = [
            'billing.full_name'     => __('Billing Full Name'),
            'billing.mobile_number' => __('Billing Mobile Number'),
            'billing.address'       => __('Billing Address'),
            'billing.country'       => __('Billing Country'),
            'billing.state'         => __('Billing State'),
            'billing.city'          => __('Billing City'),
            'billing.pincode'       => __('Billing Pincode'),
        ];

        $validation = Validator::make($request->all(), $rules, [], $attributes);

        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
            $billingTemp = ($request->has('billing')) ? $request->billing : $addresses['billing'];
            //dd($billingTemp);
            // Save shipping snapshot
            $shipping = ShippingAddress::create([
                'full_name'     => $shippingTemp['full_name'] ?? $customer->full_name,
                'mobile_number' => $shippingTemp['mobile_number'] ?? $customer->mobile_number,
                'address'       => $shippingTemp['address'] ?? null,
                'city'          => $shippingTemp['city'] ?? null,
                'state'         => $shippingTemp['state'] ?? null,
                'country'       => $shippingTemp['country'] ?? null,
                'pincode'       => $shippingTemp['pincode'] ?? null,
                'dial_code'     => $shippingTemp['dial_code'] ?? null,
                'logistics_name' => $shippingTemp['logistics_name'] ?? null,
                'gate_name' => $shippingTemp['gate_name'] ?? null,

            ]);

            // Save billing snapshot
            $billing = BillingAddress::create([
                'full_name'     => $billingTemp['full_name'] ?? $customer->full_name,
                'mobile_number' => $billingTemp['mobile_number'] ?? $customer->mobile_number,
                'address'       => $billingTemp['address'] ?? null,
                'city'          => $billingTemp['city'] ?? null,
                'state'         => $billingTemp['state'] ?? null,
                'country'       => $billingTemp['country'] ?? null,
                'pincode'       => $billingTemp['pincode'] ?? null,
                'dial_code'     => $billingTemp['dial_code'] ?? null,
            ]);
            //update addresses in orders table
            $order->update([
                'shipping_address_id' => $shipping->id,
                'billing_address_id' => $billing->id
            ]);
            //clear checkout address
            CheckoutAddress::where('customer_id', $customer->id)->delete();
            //save orders details
            foreach ($cartOverview['products_data'] as $item) {
                orderDetail::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'product_name' => $item['product_name'],
                    'price' => $item['unit_price'],
                    'quantity' => $item['final_quantity'],
                    'total' => $item['sub_total'],
                    'is_returned'          => (int)0,
                    'payment_status' => (int)0,

                ]);

                //save payment transaction
                $transaction =  Transaction::create([
                    'order_id' => $order->id,
                    'payment_status' => PENDING,
                    'payment_method' => 'UPI QR',
                    'transaction_id' => null,
                    'amount' => $order->total_price,
                    'txn_datentime' => null,
                ]);
                //update stock
                /*Product::where('id', $item['product_id'])
                    ->decrement('stock_in_hand', $item['final_quantity']);*/
                //make entry
                $inventory = ProductStock::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'type' => 'out',
                    'quantity' => $item['final_quantity'],
                    'comment' => __('Deduct the stock after place the order #' . $customOrderNumber),
                ]);

                //calculATE update product's in hand stock
                Product::updateStockInHand($item['product_id']);
            }
            //notification to admin for new order placed
            OrderNotificationHelper::sendOrderStatusNotification($order, 0, CUSTOMER);

            //save order status logs
            OrderStatusLog::create([
                'order_id' => $order->id,
                'status' => 0,
                'status_date_time'  => now(),
            ]);
            //clear the cart
            Cart::where('customer_id', $customerId)->delete();
            $orderData['transaction_id'] =   $transaction->id;
            DB::commit();

            $data = [

                'order_data' => $orderData,

            ];
            return CommonHelper::apiResponse(200, __('order placed'),  $data);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //order history fetch
    public function orderHistory(Request $request)
    {
        //fetch customer detail
        $customer = auth()->user();
        $customerId = $customer->id;
        $customerTimezone = $customer->timezone;
        $validation = validator::make($request->all(), [
            'current_page' => 'required|integer',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        //pagination
        $perPage = 10;
        $currentPage = $request->current_page;
        $skip = ($currentPage - 1) * $perPage;
        try {


            $orders = Order::where('customer_id', $customerId)
                ->with('orderDetails.product.featuredImage')
                ->orderBy('created_at', 'desc')
                ->skip($skip)
                ->take($perPage)
                ->get();

            if ($orders->isEmpty()) {
                return CommonHelper::apiResponse(200, __('no order history available'), [], '', true);
            }

            $data = $orders->map(function ($order) use ($customerTimezone) {
                $orderCurrencyInfo = CommonHelper::getCurrencyInfo($order->currency_code);
                $orderCurrencySymbol = $orderCurrencyInfo->currency_symbol;
                //dd($orderCurrencySymbol);
                $firstProductImage = '';
                //count of products
                $productCount = 0;
                if ($order->orderDetails->isNotEmpty()) {
                    $firstProduct = $order->orderDetails->first()->product;
                    if ($firstProduct && $firstProduct->featuredImage) {
                        $firstProductImage = $firstProduct->featuredImage->image;
                    }
                    //count of products
                    $productCount = $order->orderDetails->count();
                }
                //get redable status especailly for special cases
                $statusData = CommonHelper::getReadableOrderStatus($order);
                return [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'order_quantity' => $order->total_items,
                    'product_count' => $productCount,
                    'order_status' => $statusData['display_status'],
                    'order_total' => $order->total_price,
                    'currency_symbol' => $orderCurrencySymbol,
                    'order_date' => CommonHelper::formatDate($order->created_at, $customerTimezone),
                    'first_product_image' => asset($firstProductImage),
                ];
            });
            return CommonHelper::apiResponse(200, __('order history retrieved'), $data, '', true);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function orderHistoryDetail(Request $request)
    {
        $validation = validator::make($request->all(), [
            'order_id' => 'required|exists:orders,id',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        $customer = auth()->user();
        $customerId = $customer->id;
        $customerName = $customer->full_name;
        $customerContactNumber = $customer->dial_code . $customer
            ->mobile_number;
        $customerTimezone = $customer->timezone;
        $orderCancel = 0;
        $orderReturn = 0;

        try {
            $order = order::where('id', $request->order_id)
                ->where('customer_id', $customerId)
                ->with(['orderDetails.product.featuredImage', 'shippingAddress', 'billingAddress', 'transaction.files'])
                ->first();
            if (!$order) {
                return CommonHelper::apiResponse(404, __('Order not found'));
            }
            $orderStatus = $order->order_status;
            $orderId = $order->id;
            $orderCurrencyInfo = CommonHelper::getCurrencyInfo($order->currency_code);
            $orderCurrencySymbol = $orderCurrencyInfo->currency_symbol;
            $productData = $order->orderDetails->map(function ($item) use ($orderCurrencySymbol, $customerId, $orderId, $orderStatus) {
                $firstProductImage = '';
                $readyForReview = 0;
                if ($item->product && $item->product->featuredImage) {
                    $firstProductImage = $item->product->featuredImage->image;
                }
                //ready for review
                if ($orderStatus === 3) {
                    $readyForReview = 1;
                }
                //get review data to check review existence
                $review = ProductReview::where('product_id', $item->product_id)
                    ->where('customer_id', $customerId)
                    ->where('order_id', $orderId)
                    ->first();

                return [
                    'product_id' => $item->product_id,
                    'product_name' => $item->product_name,
                    'quantity' => $item->quantity,
                    'product_unit_price' => $item->price,
                    'product_total_cost' => $item->total,
                    'product_image' => asset($firstProductImage),
                    'currency_symbol' => $orderCurrencySymbol,
                    'is_review_submitted' => $review ? 1 : 0,
                    'is_ready_for_review' => $readyForReview

                ];
            });
            if (in_array($order->order_status, [0, 1, 2])) {
                $orderCancel = 1;
            }
            //set return button according to return days
            if (in_array($order->order_status, [3])) {
                $returnDays = Setting::getValue('return_days');
                $orderPlacedDate = $order->created_at;
                $lastReturnDate = $orderPlacedDate->copy()->addDays($returnDays);
                $orderReturn = now()->lessThanOrEqualTo($lastReturnDate) ? 1 : 0;
            }

            $statusData = CommonHelper::getReadableOrderStatus($order);
            $orderData = [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'order_status' =>  $statusData['display_status'],
                'payment_status_key' => $order->payment_status_data['key'],
                'payment_status_label' => $order->payment_status_data['label'],
                'transaction_id' =>   $order->transaction?->id ?? null,
                'customer_name' => $customerName,
                'customer_contact_number' => $customerContactNumber,
                'tax_percent' => $order->tax_percent,
                'tax_amount' => $order->tax_amount,
                'currency_symbol' => $orderCurrencySymbol,
                'product_cost' => $order->subtotal,
                'order_total' => $order->total_price,
                'order_date' => CommonHelper::formatDate($order->created_at, $customerTimezone),
                'order_cancel' => $orderCancel,
                'order_return' => $orderReturn,
                'special_case_message' => $statusData['special_message'],
                'order_refund_status' => $statusData['refund_status'],
                'ready_for_payment' => CommonHelper::readyForPayment($order->order_status, $order->transaction?->files ?? collect()),
                'transaction_proof' => optional($order->transaction?->files?->first())->file_path ? asset(optional($order->transaction?->files?->first())->file_path) : null,
                'notes' => $order->notes ?? null
            ];
            $data = [
                'product_data' => $productData,
                'shipping_address' => $order->shippingAddress,
                'billing_address' => $order->billingAddress,
                'order_data' => $orderData,

            ];
            return CommonHelper::apiResponse(200, __('Order detail retrieved'), $data);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function orderUpdateStatus(Request $request)
    {
        DB::beginTransaction();
        $action = $request->action;
        try {
            $validation = Validator::make($request->all(), [
                'order_id' => 'required|integer|exists:orders,id',
                'action' => 'required|string|in:cancel,return',
                'reason' => 'required|string|max:255',
            ]);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            $order = Order::with('transaction')->find($request->order_id);

            //important security check
            if ($order->customer_id != auth()->id()) {
                return CommonHelper::apiResponse(400, __('unauthorized customer'));
            }
            //return
            if ($request->action == 'return') {

                //return only when
                if ($order->order_status != 3) {
                    return CommonHelper::apiResponse(400, __('order_return_only'));
                }
            }

            //cancel
            if ($request->action == 'cancel') {

                //cancel only when
                if (!in_array($order->order_status, [0, 1, 2])) {
                    return CommonHelper::apiResponse(400, __('order_cancel_only'));
                }
            }
            $statusMap = [
                'cancel' => 4,
                'return' => 5
            ];

            //now enter record in corresponding table
            if ($request->action == ACTION_CANCEL) {
                OrderCancellation::firstOrCreate(
                    ['order_id' => $order->id],
                    [
                        'order_id' => $order->id,
                        'cancel_by_id' => auth()->id(),
                        'cancel_by_type' => CUSTOMER,
                        'reason' => $request->reason,
                        'status' => 0,
                        'refund_status' => 0,
                    ]
                );
                //if cancel before confirm order
                if ($order->order_status == 0) {
                    $order->order_status = 9;
                    $order->payment_status = 0;
                    $order->payment_status = 0;
                    $order->transaction->payment_status = 'unpaid';
                    $order->transaction->save();
                } else {
                    $order->order_status = 4;
                }
                $order->save();
            } else if ($request->action == ACTION_RETURN) {
                OrderReturn::firstOrCreate([
                    'order_id' => $order->id,
                    'reason' => $request->reason,
                    'status' => 0,
                    'refund_status' => 0,
                ]);
                $order->order_status = 5;
                $order->save();
            }
            //save notification
            OrderNotificationHelper::sendOrderStatusNotification($order, $statusMap[$action], CUSTOMER);

            DB::commit();
            //$actionText = CommonHelper::$pastTenseMap[$action] ?? "{action}ed";
            $actionText = CommonHelper::$pastTenseMap[$action] ?? "{action}ed";
            return CommonHelper::apiResponse(200, __('order updated after approval', ['updated' => $actionText]));
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
