<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Helpers\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class MessageController extends Controller
{
    //save message
    public function store(Request $request)
    {
        $senderName = auth()->user()->full_name;
        $allowedImages = CommonHelper::allowedImageExtension();
        $mimes = implode(',', $allowedImages);
        $validation = Validator::make(
            $request->all(),
            [
                'conversation_id' => 'required|exists:conversations,id',
                'message' => 'nullable|string',
                'attachement' => 'nullable|mimes:' . $mimes . '|max:2048',
            ],
            [
                'attachement.mimes' => __('file_type_vald'),
                'attachement.max' => __('each_file_size_vald_2mb')
            ]

        );
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        $message = new Message();
        $message->conversation_id = $request->conversation_id;
        $message->sender_id = auth()->id();
        $message->sender_type = \App\Models\Customer::class;
        $message->message = $request->message;
        if ($request->has('attachement')) {
            $message->attachements = CommonHelper::fileUpload($request->file('attachement'), MESSAGE_MEDIA);
        }
        $message->save();
        //send notification
        $notifyRoles = CommonHelper::$generalNotifyRoles;
        $admins = User::role($notifyRoles)->get();
        $convo_id = $message->conversation_id;
        foreach ($admins as $admin) {
            NotificationHelper::saveNotification(
                auth()->id(),
                CUSTOMER,
                $admin->id,
                ADMIN,
                CHAT,
                __('new_message'),
                __('new_message_from', ['name' => $senderName]),
                ['url' => route('admin.chat'), 'conversation_id' => $convo_id],
            );
        }
        //message notification
        event(new \App\Events\AdminChatMessageEvent([
            'title' => __('new_message'),
            'message' => __('new_message_from', ['name' => $senderName]),
            'conversation_id' => $convo_id,
            'url' => route('admin.chat'),
        ]));

        //fire event via pusher
        $message = Message::with(['sender'])->find($message->id);
        event(new \App\Events\ChatMessageEvent($message));
        return CommonHelper::apiResponse(200, __('message_sent'));
    }
    //get message list
    public function index(Request $request)
    {
        //fetch customer detail
        $customer = auth()->user();
        $customerTimezone = $customer->timezone;
        $messages = Message::where('conversation_id', $request->conversation_id)
            ->with('sender')
            ->get()
            ->map(function ($msg) use ($customerTimezone) {
                $senderClass =  class_basename($msg->sender_type);
                $freindlyType = match ($senderClass) {
                    'User' => 'Admin',
                    'Customer' => 'Customer',
                    default => $senderClass,
                };
                return [
                    'conversation_id' => $msg->conversation_id,
                    'sender_type' => $freindlyType, // gives "Customer" or "Admin"
                    'message' => $msg->message,
                    'attachements' => $msg->attachements ?  asset($msg->attachements) : null,
                    'is_read' => $msg->is_read,
                    'created_at' => CommonHelper::formatDate($msg->created_at, $customerTimezone),
                    'sender' => [
                        'id' => $msg->sender?->id,
                        'name' => $msg->sender?->full_name ?? $msg->sender?->name, // handle Customer vs Admin
                        'image' => $msg->sender?->image ? asset($msg->sender?->image) : asset(DEFAULT_PATH),
                    ]
                ];
            });
        if ($messages->isEmpty()) {
            return CommonHelper::apiResponse(200, __('no_message_found'), $messages, '', true);
        }
        return CommonHelper::apiResponse(200, __('messages_retrieved'), $messages, '', true);
    }
}
