<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\CustomerAddress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class CustomerAddressController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        /**
         * Show the form for creating a new resource.
         *
         * @return \Illuminate\Http\Response
         */
        try {
            $customerId = auth()->user()->id;
            //fetch addresses
            $addresses = CustomerAddress::where('customer_id', $customerId)
                //->orderBy()
                ->get();
            $addresses =  $addresses->map(function ($address) {
                $country = CommonHelper::getCountries($address->country);
                $state = CommonHelper::getStates($address->country, $address->state);
                $city = CommonHelper::getCities($address->state, $address->city);
                return [
                    'address_id' => $address->id,
                    'address_line' => $address->address,
                    'country_id' => $address->country,
                    'country_name' => $country->name,
                    'state_id' => $address->state ?? null,
                    'state_name' => $state->name ?? null,
                    'city_id' => $address->city ?? null,
                    'city_name' => $city->name ?? null,
                    'pincode' => $address->pincode,
                    'is_default' => $address->is_default,
                    'logistics_name' => $address->logistics_name,
                    'gate_name' => $address->gate_name,
                    'full_name' => $address->full_name,
                    'dial_code' => $address->dial_code,
                    'mobile_number' => $address->mobile_number,
                ];
            });
            return CommonHelper::apiResponse(200, __('addresses retrieved'), $addresses, '', true);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $customerId = auth()->user()->id;
        $validation =  CustomerAddress::addressValidation($request->all(), 'store');
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }

        try {
            //if is_defualt true in request
            if ($request->has('is_default') && $request->is_default == 1) {
                CustomerAddress::where('customer_id', $customerId)->update(['is_default' => false]);
            }

            //store
            CustomerAddress::create([
                'customer_id' => $customerId,
                'address' => $request->address_line1,
                'country' => $request->country_id,
                'state' => $request->state_id,
                'city' => $request->city_id,
                'pincode' => $request->pincode,
                'is_default' => $request->is_default ?? 0,
                'logistics_name' => $request->logistics_name,
                'gate_name' => $request->gate_name,
                'full_name' => $request->full_name,
                'dial_code' => $request->dial_code,
                'mobile_number' => $request->mobile_number,
            ]);
            return CommonHelper::apiResponse(200, __('address created'));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $customerId = auth()->user()->id;
        //check validation
        $validation = CustomerAddress::addressValidation($request->all(), 'update');
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        $address = CustomerAddress::where('id', $request->address_id)
            ->where('customer_id', $customerId)
            ->first();
        if (!$address) {
            return CommonHelper::apiResponse(400, __('address not found'));
        }

        try {

            //if get is_default false rest in db
            if ($request->has('is_default') && $request->is_default == 1 && $address->is_default == 0) {
                CustomerAddress::where('customer_id', $customerId)->update(['is_default' => 0]);
            }

            //update
            $address->update([
                'customer_id' => $customerId,
                'address' => $request->address_line1,
                'country' => $request->country_id,
                'state' => $request->state_id,
                'city' => $request->city_id,
                'pincode' => $request->pincode,
                'is_default' => $request->has('is_default') ? $request->is_default : $address->is_default,
                'logistics_name' => $request->logistics_name,
                'gate_name' => $request->gate_name,
                'full_name' => $request->full_name,
                'dial_code' => $request->dial_code,
                'mobile_number' => $request->mobile_number,
            ]);
            return CommonHelper::apiResponse(200, __('address updated'));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $customerId = auth()->user()->id;
        //1.check validation
        $validation = Validator::make($request->all(), [
            'address_id' => 'required|integer|exists:customers_addresses,id'
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        // 2. Ensure address belongs to this customer
        $address = CustomerAddress::where('id', $request->address_id)
            ->where('customer_id', $customerId)
            ->first();
        if (!$address) {
            return CommonHelper::apiResponse(400, __('address not found'));
        }
        // 3. Prevent deletion of default
        if ($address->is_default == 1) {
            return CommonHelper::apiResponse(400, __('cant delete default one'));
        }
        // 4. Perform delete
        try {

            $address->delete();
            return CommonHelper::apiResponse(200, __('address deleted'));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //set defualt address api
    public function setDefaultAddress(Request $request)
    {
        $customerId = auth()->user()->id;
        //check validation
        $validation = Validator::make($request->all(), [
            'address_id' => 'integer|required|exists:customers_addresses,id'
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        $address = CustomerAddress::where('id', $request->address_id)
            ->where('customer_id', $customerId)
            ->first();
        if (!$address) {
            return CommonHelper::apiResponse(400, __('address not found'));
        }
        try {
            //update to false all address's is_default
            CustomerAddress::where('customer_id', $customerId)
                ->update(['is_default' => 0]);
            //set requested to default address
            $address->is_default = 1;
            $address->save();
            return CommonHelper::apiResponse(200, __('address set default'));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get all countries
    public function getAllCountries()
    {
        try {
            $countries = CommonHelper::getCountries();
            if ($countries->isEmpty()) {
                return CommonHelper::apiResponse(200, __('countries not found'), $countries, '', true);
            }
            return CommonHelper::apiResponse(200, __('countries retrieved succeesfully'), $countries);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get all states
    public function getStatesByCountry($country_id)
    {
        try {
            $states = CommonHelper::getStates($country_id);
            if ($states->isEmpty()) {
                return CommonHelper::apiResponse(200, __('states not found'), $states, '', true);
            }
            return CommonHelper::apiResponse(200, __('states retrieved succeesfully'), $states);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get all cities
    public function getCitiesByState($state_id)
    {

        try {
            $cities = CommonHelper::getCities($state_id);
            if ($cities->isEmpty()) {
                return CommonHelper::apiResponse(200, __('cities not found'), $cities, '', true);
            }
            return CommonHelper::apiResponse(200, __('cities retrieved succeesfully'), $cities);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get default address
    public function getDefaultAddress()
    {
        $customer = auth()->user();
        $customerName = $customer->full_name;
        $customerDialCode = $customer->dial_code;
        $customerMobileNumber = $customer->mobile_number;
        try {
            $defaultAddress = CustomerAddress::where('customer_id', auth()->id())
                ->where('is_default', 1)
                ->first();
            if (!$defaultAddress) {
                return CommonHelper::apiResponse(200, __('no default address found'));
            }
            $country = CommonHelper::getCountries($defaultAddress->country);
            $state = CommonHelper::getStates($defaultAddress->country, $defaultAddress->state);
            $city = CommonHelper::getCities($defaultAddress->state, $defaultAddress->city);
            $data = [
                'full_name' => $customerName,
                'dial_code' => $customerDialCode,
                'mobile_number' => $customerMobileNumber,
                'address' =>  $defaultAddress->address,
                'country_id' => $defaultAddress->country,
                'country_name' => $country->name,
                'state_id' => $defaultAddress->state,
                'state_name' => $state->name,
                'city_id' => $defaultAddress->city,
                'city_name' => $city->name,
                'pincode' => $defaultAddress->state,
            ];


            return CommonHelper::apiResponse(200, __('default address retrieved'), $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
