<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\BillingAddress;
use App\Models\Cart;
use App\Models\CheckoutAddress;
use App\Models\CustomerAddress;
use App\Models\order;
use App\Models\orderDetail;
use App\Models\OrderStatusLog;
use App\Models\ShippingAddress;
use App\Services\CartService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Services\ProductService;
use Throwable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CheckoutController extends Controller
{
    protected $productService;
    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }
    public function checkoutReview(CartService $cartService)
    {

        $customer = auth()->user();
        $cartOverview =  $cartService->getCartOverview($customer);
        if ($cartOverview['empty_cart']) {
            return CommonHelper::apiResponse(200, __('cart empty'));
        }
        $addresses = CommonHelper::getCheckoutAddresses($customer->id);
        //dd($addresses);
        $shippingAddress = $addresses['shipping'];
        // Step A: Get default shipping address
        $defaultShipping = CustomerAddress::where('customer_id', $customer->id)
            ->where('type', 'shipping')
            ->where('is_default', 1)
            ->first();

        // Step B: If no default, use the latest shipping address
        if (!$defaultShipping) {
            $defaultShipping = CustomerAddress::where('customer_id', $customer->id)
                ->where('type', 'shipping')
                ->latest()
                ->first();
        }
        if ($defaultShipping) {
            $shippingAddress = CommonHelper::convertAddressToNames($defaultShipping, $customer);
        }
        // add into response
        $data = [
            'products_data' => $cartOverview['products_data'],
            'removed_items' => $cartOverview['removed_items'],
            'payment_summary' => $cartOverview['payment_summary'],
            'cash_back_detail' => $cartOverview['cash_back_detail'],
            'shipping_address' => $shippingAddress,
            'billing_address'  => $addresses['billing'],
        ];

        return CommonHelper::apiResponse(200, __('checkout review'),  $data);


        //fetch cart detail - original cart count
        /*$originalCartCount = Cart::where('customer_id', $customerId)->count();*/

        /*$deletedInactiveProductMessage = null;*
            if ($cartItems->count() != $originalCartCount) {
                $deletedInactiveProductMessage = "some products are inactive or deleted order placed";
            }*/
    }
    //save address
    public function saveCheckoutAddress(Request $request)
    {
        $customer = auth()->user();

        $rules = [
            'shipping.full_name'     => 'required|string|max:255',
            'shipping.mobile_number' => 'required|string|max:20|digits_between:6,14',
            'shipping.address' => 'required|string|max:255',
            'shipping.city_id'          => 'required|integer|exists:cities,id',
            'shipping.state_id'         => 'required|integer|exists:states,id',
            'shipping.country_id'       => 'required|integer|exists:countries,id',
            'shipping.city'          => 'required|string|max:100',
            'shipping.state'         => 'required|string|max:100',
            'shipping.country'       => 'required|string|max:100',
            'shipping.dial_code'       => 'required|regex:/^\+\d{1,4}$/',
            'shipping.logistics_name' => 'nullable|string',
            'shipping.gate_name' => 'nullable|string',
            'shipping.notes' => 'nullable|string',

            'billing_same_as_shipping' => 'nullable|boolean',

            'billing.full_name'     => 'required_if:billing_same_as_shipping,false|string|max:255',
            'billing.mobile_number' => 'required_if:billing_same_as_shipping,false|string|max:20|digits_between:6,14',
            'billing.address' => 'required_if:billing_same_as_shipping,false|string|max:255',
            'billing.city_id'          => 'required_if:billing_same_as_shipping,false|integer|exists:cities,id',
            'billing.state_id'         => 'required_if:billing_same_as_shipping,false|integer|exists:states,id',
            'billing.country_id'       => 'required_if:billing_same_as_shipping,false|integer|exists:countries,id',
            'billing.city'          => 'required_if:billing_same_as_shipping,false|string|max:100',
            'billing.state'         => 'required_if:billing_same_as_shipping,false|string|max:100',
            'billing.country'       => 'required_if:billing_same_as_shipping,false|string|max:100',
            'billing.dial_code'       => 'required|regex:/^\+\d{1,4}$/',
        ];
        $isShippingPincodeOptional = CommonHelper::isPincodeOptional($request->shipping['country_id']);
        $rules['shipping.pincode'] = ($isShippingPincodeOptional ? 'nullable' : 'required') . '|string|max:20';

        //for billing pincode
        $isBillingPincodeOptional = CommonHelper::isPincodeOptional($request->billing['country_id'] ?? null);
        $rules['billing.pincode'] = ($isBillingPincodeOptional ? 'nullable' : 'required_if:billing_same_as_shipping,false') . '|string|max:20';

        $attributes = [
            'shipping.full_name'     => __('Shipping Full Name'),
            'shipping.mobile_number' => __('Shipping Mobile Number'),
            'shipping.address'       => __('Shipping Address'),
            'shipping.country'       => __('Shipping Country'),
            'shipping.state'         => __('Shipping State'),
            'shipping.city'          => __('Shipping City'),
            'shipping.pincode'       => __('Shipping Pincode'),

            'billing.full_name'     => __('Billing Full Name'),
            'billing.mobile_number' => __('Billing Mobile Number'),
            'billing.address'       => __('Billing Address'),
            'billing.country'       => __('Billing Country'),
            'billing.state'         => __('Billing State'),
            'billing.city'          => __('Billing City'),
            'billing.pincode'       => __('Billing Pincode'),
        ];

        $validation = Validator::make($request->all(), $rules, [], $attributes);

        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        // save shipping
        $shippingAddress =    CheckoutAddress::updateOrCreate(
            ['customer_id' => $customer->id, 'type' => 'shipping'],
            [
                'full_name'     => $request->shipping['full_name'],
                'mobile_number' => $request->shipping['mobile_number'],
                'dial_code'     => $request->shipping['dial_code'],
                'address'       => $request->shipping['address'],
                'city'       => $request->shipping['city_id'],
                'state'      => $request->shipping['state_id'],
                'country'    => $request->shipping['country_id'],
                'pincode'       => $request->shipping['pincode'] ?? null,
                'is_default' =>  1,
                'logistics_name' => $request->shipping['logistics_name'] ?? null,
                'gate_name' => $request->shipping['gate_name'] ?? null,

            ]
        );

        if (!$request->billing_same_as_shipping) {
            // save billing only if different
            CheckoutAddress::updateOrCreate(
                ['customer_id' => $customer->id, 'type' => 'billing'],
                [
                    'full_name'     => $request->billing['full_name'],
                    'mobile_number' => $request->billing['mobile_number'],
                    'dial_code'     => $request->billing['dial_code'],
                    'address'       => $request->billing['address'],
                    'city'       => $request->billing['city_id'],
                    'state'      => $request->billing['state_id'],
                    'country'    => $request->billing['country_id'],
                    'pincode'       => $request->billing['pincode'] ?? null,
                ]
            );
        } else {
            // optional: remove any old billing record
            CheckoutAddress::where('customer_id', $customer->id)
                ->where('type', 'billing')
                ->delete();
        }
        //save shipping address as defualt address in customer address
        if (!CustomerAddress::where('customer_id', auth()->id())->exists()) {
            CustomerAddress::create([
                'customer_id' => $customer->id,
                'type' => 'shipping',
                'address' => $shippingAddress->address,
                'city' => $shippingAddress->city,
                'state' => $shippingAddress->state,
                'country' => $shippingAddress->country,
                'pincode' => $shippingAddress->pincode,
                'is_default' => 1,
                'full_name'     => $request->shipping['full_name'],
                'mobile_number' => $request->shipping['mobile_number'],
                'dial_code'     => $request->shipping['dial_code'],
                'logistics_name' => $shippingAddress->logistics_name,
                'gate_name' => $shippingAddress->gate_name,
            ]);
        }

        return CommonHelper::apiResponse(200, __('Checkout addresses saved'));
    }
    public function getCheckoutAddress(Request $request)
    {
        $customer = auth()->user();

        // Fetch addresses from temp table
        $shipping = CheckoutAddress::where('customer_id', $customer->id)
            ->where('type', 'shipping')
            ->first();

        $billing = CheckoutAddress::where('customer_id', $customer->id)
            ->where('type', 'billing')
            ->first();

        // Determine if billing is same as shipping
        $billingSameAsShipping = false;
        if (!$billing && $shipping) {
            $billingSameAsShipping = true;
        }

        $data = [
            'shipping' => $shipping ? [
                'full_name'     => $shipping->full_name,
                'mobile_number' => $shipping->mobile_number,
                'address'       => $shipping->address,
                'city'          => $shipping->city,
                'state'         => $shipping->state,
                'country'       => $shipping->country,
                'pincode'       => $shipping->pincode,
            ] : null,
            'billing_same_as_shipping' => $billingSameAsShipping,
            'billing' => $billing ? [
                'full_name'     => $billing->full_name,
                'mobile_number' => $billing->mobile_number,
                'address'       => $billing->address,
                'city'          => $billing->city,
                'state'         => $billing->state,
                'country'       => $billing->country,
                'pincode'       => $billing->pincode,
            ] : null,
        ];

        return CommonHelper::apiResponse(200, __('Checkout addresses fetched'), $data);
    }
    //customer exist or not in orders table : to check new customer or not
    public function isReturningCustomer()
    {
        $hasAddress = CustomerAddress::where('customer_id', auth()->id())->exists();
        $data = [
            'old_customer' => (int) $hasAddress,
        ];
        return CommonHelper::apiResponse(200, __('customer_existence_info_fetched'), $data);
    }
    //get country id and name as per dial code
    public function countryInfoDialCode(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'dial_code_iso' => 'required|uppercase|max:2'
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(200, $validation->errors());
        }
        try {
            $country =  DB::table('countries')
                ->where('iso2', strtoupper($request->dial_code_iso))->select('id', 'name')
                ->first();
            if (!$country) {
                return CommonHelper::apiResponse(400, __('country_data_not_found'));
            }
            $data = [
                'id' => $country->id,
                'name' => $country->name
            ];
            return CommonHelper::apiResponse(200, __('fecthed_country_data'), $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
