<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Cart;
use App\Models\Product;
use App\Models\WholesaleIdPlan;
use App\Services\CashBackService;
use App\Services\ProductService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use League\CommonMark\CommonMarkConverter;
use Throwable;

class CartController extends Controller
{
    protected $productService;
    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }
    //update cart at product detail screen (add or update the quantity)
    public function updateCart(Request $request)
    {
        $customer = auth()->user();
        $customerId = $customer->id;
        $currencyCode = $customer->currency;

        $validation = Validator::make($request->all(), [
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:0'
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }

        try {
            //if quantity == 0 then remove product from cart
            if ($request->quantity == 0) {
                Cart::where('customer_id', $customerId)
                    ->where('product_id', $request->product_id)
                    ->delete();
                return CommonHelper::apiResponse(200, __('product removed cart'));
            }
            $product = Product::find($request->product_id);
            if (!$product) {
                return CommonHelper::apiResponse(404, __('product not found or discontinued'));
            }
            if ($product->status == 0) {
                return CommonHelper::apiResponse(400, __('product inactive, cannot be purchased'));
            }
            //same check for out of stock like in view cart api
            if ($product->stock_in_hand == 0) {
                $data = [
                    'out_of_stock' => (int) 1,

                ];
                return CommonHelper::apiResponse(
                    400,
                    __('out of stock', [
                        'product' => $product->name
                    ]),
                    $data
                );
            }
            //check product stock availability
            if ($request->quantity > $product->stock_in_hand) {
                $data = [
                    'less_stock' => (int) 1,
                    'available_stock' => $product->stock_in_hand
                ];
                return CommonHelper::apiResponse(
                    400,
                    __('quantity_exceed'),
                    $data
                );
            }

            //get price
            $productData = $this->productService->getCurrency($product, $currencyCode);
            //update or insert
            $cart = Cart::updateOrCreate(
                [
                    'customer_id' => $customerId,
                    'product_id' => $request->product_id,
                    'price' => $productData['price'],

                ],
                [
                    'quantity' => $request->quantity
                ]
            );

            $data = [
                'product_id' => $cart->product->id,
                'quantity' => $cart->quantity,
                'available_stock' => $productData['stock_in_hand'],
            ];
            return CommonHelper::apiResponse(200, __('cart updated'), $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //delete cart item
    public function deleteCart(Request $request)
    {
        $customerId = auth()->id();
        $validation = Validator::make($request->all(), [
            'product_id' => 'required|exists:products,id'
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        try {
            $cartItem = Cart::where('product_id', $request->product_id)
                ->where('customer_id', $customerId)
                ->first();
            if (!$cartItem) {
                return CommonHelper::apiResponse(404, __('cart_not_found'));
            }
            $cartItem->delete();
            return CommonHelper::apiResponse(200, __('cart_deleted'));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //view cart 
    public function viewCart(CashBackService $cashBackService, Request $request)
    {
        $customer = auth()->user();
        $customerId = $customer->id;
        $currencySymbol = $customer->currency_symbol;
        $currencyCode = $customer->currency;
        $validation = validator::make($request->all(), [
            'current_page' => 'required|integer',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        //pagination
        $perPage = 10;
        $currentPage = $request->current_page;
        $skip = ($currentPage - 1) * $perPage;


        //assign basic plan in the start if customer has no plan
        if (!$customer->activePlanWholesale) {
            $cashback =   $cashBackService->assignBasicPlan($customer);
        } else {
            $cashback = $customer->activePlanWholesale;
        }
        $cashback->load('originPaln');

        //fetch all porducts according to cart 
        $cartItems = Cart::with('product')
            ->where('customer_id', $customerId)
            ->orderBy('created_at', 'desc')
            ->skip($skip)
            ->take($perPage)
            ->get();
        if ($cartItems->isEmpty()) {
            return CommonHelper::apiResponse(200, __('cart empty'));
        }
        //fecth product price column data according to currecny
        $cartData = $cartItems->map(function ($item) use ($currencyCode) {
            $productData = $this->productService->getCurrency($item->product, $currencyCode);
            $subtoatl = $productData['price'] * $item->quantity;
            //derive flags
            $isDeleted = $productData['status'] == 2 ? 1 : 0;
            $isInactive = $productData['status'] == 0 ? 1 : 0;
            $outOfStock = 0;
            $lessStock = 0;
            $priceChange = 0;
            $warning = [];

            if ($item->quantity > $productData['stock_in_hand'] && $productData['stock_in_hand'] != 0) {
                $lessStock = 1;
                $warning[] =  __('only items available', [
                    'stock' => $productData['stock_in_hand'],
                    'product' => $productData['name']
                ]);
            }
            if ($productData['stock_in_hand'] == 0) {
                $outOfStock = 1;
                $warning[] = __('out of stock', ['product' => $productData['name']]);
            }
            if ($item->price != $productData['price']) {
                $priceChange = 1;
                $warning[] = __('price changed', ['product' => $productData['name']]);
            }


            if ($isDeleted) {
                $warning[] = __('product_deleted');
            } elseif ($isInactive) {
                $warning[] = __('product_inactive');
            }
            return [
                'product_id' => $productData['id'],
                'product_name' => $productData['name'],
                'product_image' => $productData['featured_image'],
                'unit_price' => $productData['price'],
                'unit_price_formatted' => number_format($productData['price'], 2),
                'currency_symbol' => $productData['currency_symbol'],
                'quantity' => $item->quantity,
                'available_stock' => $productData['stock_in_hand'],
                'status' => $productData['status'],
                'deleted_at' => $productData['deleted_at'],
                'is_deleted' => $isDeleted,
                'is_inactive' => $isInactive,
                'price_chnaged' => $priceChange,
                'out_of_stock' => $outOfStock,
                'less_stock' => $lessStock,
                'warning' => $warning,
                'sub_total' => $subtoatl,
                'sub_total_formatted' =>  number_format($subtoatl, 2),

            ];
        });

        //prepare summary for payment
        $totalProductAmount = (float)$cartData->sum('sub_total');
        $summary = CommonHelper::taxCalculate($totalProductAmount);

        //fetch wholesale paln if not active any then apply basic one
        //$appliedPlan = $customer->activePlanWholesale ??  WholesaleIdPlan::where('is_default', 1)->first();
        $paymentSummeryData = [
            'product_cost' => $totalProductAmount,
            'product_cost_formatted' => number_format($totalProductAmount, 2),
            'tax_rate' => $summary['tax_rate'],
            'tax_amount' => $summary['tax_amount'],
            'tax_amount_formatted' => number_format($summary['tax_amount'], 2),
            'order_total' => $summary['total_with_tax'],
            'order_total_formatted' => number_format($summary['total_with_tax'], 2),
            'currency_symbol' => $currencySymbol,

        ];
        $cashBackData = [
            'cash_back_paln_id' => $customer->wholesale_plan_id,
            'cash_back_title' => $cashback->title,
            'is_default' => (int) optional($cashback->originPaln)->is_default,

        ];
        //if cart items have some issues
        $checkoutDisable = (int)0;
        if ($cartData->contains('out_of_stock', 1) || $cartData->contains('is_deleted', 1) || $cartData->contains('is_inactive', 1) || $cartData->contains('price_chnaged', 1) || $cartData->contains('less_stock', 1)) {
            $checkoutDisable = (int)1;
        }
        $data = [
            'products_data' => $cartData,
            'payment_summary' => $paymentSummeryData,
            'cash_back_detail' => $cashBackData,
            'checkout_disable' => $checkoutDisable
        ];
        return CommonHelper::apiResponse(200, __('cart data retrieved'),  $data);
    }
}
