<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VehicleType;
use Illuminate\Http\Request;
use App\Helpers\CommonHelper;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\QueryException;
use Throwable;


class VehicleTypeController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-vehicle_type')->only('index');
        $this->middleware('permission:add-vehicle_type')->only(['create', 'store']);
        $this->middleware('permission:edit-vehicle_type')->only(['edit', 'update']);
        $this->middleware('permission:edit-vehicle_type')->only(['edit', 'show']);
        $this->middleware('permission:delete-vehicle_type')->only(['destroy']);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $vehicleTypes = (new VehicleType())->getPaginatedData($request);

        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.vehicleType.pagination', compact('vehicleTypes'))->render()
            ]);
        }

        return view('admin.vehicleType.index', compact('vehicleTypes'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.vehicleType.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|min:3|max:200',
            'status' => 'in:0,1'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {

            VehicleType::create([
                'name' => $request->name,
                'status' => $request->status
            ]);
            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-type.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle type added successfully.', [], $extra);
        } catch (\Throwable $e) {
            DB::rollback();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $vehicleType = VehicleType::where('id', $id)->first();
        if (!$vehicleType) {
            abort(403);
        }
        return view('admin.vehicleType.edit', compact('vehicleType'));
        return CommonHelper::jsonErrorResponseWeb($e);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validation = Validator::make($request->all(), [
            'name' => 'required|max:200,min:3',
            'status' => 'required|in:0,1'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }

        try {
            $vehicleType = VehicleType::findOrFail($id);

            DB::beginTransaction();
            $data = [
                'name' => $request->name,
                'status' => $request->status,
            ];
            $vehicleType->update($data);
            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-type.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle Type updated successfully.', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {

        $item = VehicleType::findOrFail($id);
        try {
            $item->delete();
            $item->update(['status' => 2]);
            $extra = [
                'redirect' => route('admin.vehicle-type.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle Type deleted successfully.', [], $extra);
        } catch (QueryException $e) {
            $msg = $e->getMessage();
            if (strpos($msg, '1451') !== false) { // fk check
                $message = 'Please delete entries first from its relevant sections';
                return response()->json(['status' => false, 'status_code' => 500, 'message' => $message], 500);
            }
        } catch (Throwable $e) {
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
