<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use App\Helpers\CommonHelper;
use App\Models\Customer;
use App\Models\Role;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Database\QueryException;
use Spatie\Permission\Models\Permission;
use Throwable;

class EmployeeController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-employee')->only('index');
        $this->middleware('permission:add-employee')->only(['create', 'store']);
        $this->middleware('permission:edit-employee')->only(['edit', 'update']);
        $this->middleware('permission:edit-employee')->only(['edit', 'show']);
        $this->middleware('permission:delete-employee')->only(['destroy']);
    }
    public function index(Request $request)
    {
        $employees = (new User())->getPaginatedData($request);

        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.employee.pagination', compact('employees'))->render()
            ]);
        }

        return view('admin.employee.index', compact('employees'));
    }
    public function create()
    {
        $roles = Role::orderBy('name', 'asc')->where('name', '!=', SUPER_ADMIN)->get();
        return view('admin.employee.create', compact('roles'));
    }
    public function store(Request $request)
    {
        DB::beginTransaction();
        $timezone = null;
        $validation = Validator::make($request->all(), [
            'name'              => 'required|string|min:3|max:200',
            'email'             => 'required|email|unique:users,email',
            'mobile_number'     => 'required|string|min:7|max:15|unique:users,mobile_number',
            'dob'               => 'nullable|date|before:today',
            'gender'            => 'nullable|in:male,female,other',
            'address'           => 'nullable|string|max:500',
            'image'             => 'nullable|image|mimes:jpeg,jpg,png,gif,webp|max:2048',
            'status'            => 'required|in:0,1',
            'role' => 'required|exists:roles,name',
            'password'          => [
                'required',
                'string',
                'min:8',
                'confirmed',
                'regex:/[A-Z]/',
                'regex:/[a-z]/',
                'regex:/[0-9]/',
                'regex:/[@$!%*#?&]/',
            ],
        ], [
            'password.regex' => 'Password must contain at least one uppercase letter, one lowercase letter, one number, and one special character.',
            'email.unique' => 'This email is already registered.',
            'mobile_number.unique' => 'This Mobile number is already registered.',
        ]);

        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        //check validation of mobile number with dial code
        if (!Customer::validateMobileNumber($request->dial_code, $request->dial_code_iso,  $request->mobile_number)) {
            return CommonHelper::jsonResponseWeb(400, __('Invalid mobile number'));
        }
        $imagePath = DEFAULT_PATH;

        if ($request->hasFile('image')) {
            $imagePath = CommonHelper::fileUpload($request->file('image'), EMPLOYEE_IMAGE_PATH);
        }
        $dob = Carbon::createFromFormat('d-m-Y', $request->dob)->format('Y-m-d');
        //get timezone
        $countryInfo = CommonHelper::getCountryInfo($request->dial_code_iso);
        if ($countryInfo) {
            $timezone = $countryInfo['timezone'];
            $updatedData['timezone'] = $timezone;
        }

        try {

            $employee =  User::create([
                'name'          => $request->name,
                'email'         => $request->email,
                'mobile_number' => $request->mobile_number,
                'dial_code' => $request->dial_code,
                'dial_code_iso' => strtoupper($request->dial_code_iso),
                'dob'           => $dob,
                'gender'        => $request->gender,
                'address'       => $request->address,
                'status'        => $request->status,
                'image'         => $imagePath,
                'password'      => Hash::make($request->password),
                'timezone' => $timezone
            ]);
            $employee->syncRoles($request->role);
            DB::commit();

            $extra = [
                'redirect' => route('admin.employee.index')
            ];

            return CommonHelper::jsonResponseWeb(200, 'Employee created successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function edit($id)
    {
        $employee = User::findOrFail($id);
        $roles = Role::orderBy('name', 'asc')->where('name', '!=', SUPER_ADMIN)->get();
        $employeeRole = $employee->roles->pluck('id');
        //$employee->dob = \Carbon\Carbon::parse($employee->dob)->format('d-m-Y');
        return view('admin.employee.edit', compact('employee', 'roles', 'employeeRole'));
    }
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $employee = User::findOrFail($id);

            // Base validation rules
            $rules = [
                'name' => 'required|string|max:255',
                'gender' => 'nullable|in:male,female,other',
                'dob' => 'nullable|date',
                'address' => 'nullable|string',
                'status' => 'required|in:0,1',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'password' => 'nullable|min:6|confirmed',
                'role' => 'required|exists:roles,name',
            ];

            if ($request->email !== $employee->email) {
                $rules['email'] = 'required|email|unique:users,email';
            }

            if ($request->mobile_number !== $employee->mobile_number) {
                $rules['mobile_number'] = 'required|digits_between:8,15|unique:users,mobile_number';
            }
            $validation = Validator::make($request->all(), $rules);
            if ($validation->fails()) {
                return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
            }
            //check validation of mobile number with dial code
            if (!Customer::validateMobileNumber($request->dial_code, $request->dial_code_iso,  $request->mobile_number)) {
                return CommonHelper::jsonResponseWeb(400, __('Invalid mobile number'));
            }
            $validatedData = $validation->validated($rules);
            if ($request->filled('dob') && $request->dob !== $employee->dob) {
                $dob = Carbon::createFromFormat('d-m-Y', $validatedData['dob'])->format('Y-m-d');
            } else {
                $dob = $employee->dob;
            }
            //check timezone update
            if ($request->dial_code_iso && $request->dial_code_iso !== $employee->dial_code_iso) {
                $countryInfo = CommonHelper::getCountryInfo($request->dial_code_iso);
                if ($countryInfo) {
                    $timezone = $countryInfo['timezone'];
                    $employee->timezone = $timezone;
                }
            }
            $employee->name = $validatedData['name'];
            $employee->gender = $validatedData['gender'];
            $employee->dob = $dob;
            $employee->dial_code = $request->dial_code;
            $employee->dial_code_iso = strtoupper($request->dial_code_iso);
            $employee->address = $validatedData['address'];
            $employee->status = $validatedData['status'];

            if (isset($validatedData['email'])) {
                $employee->email = $validatedData['email'];
            }

            if (isset($validatedData['mobile_number'])) {
                $employee->mobile_number = $validatedData['mobile_number'];
            }

            if (!empty($validatedData['password'])) {
                $employee->password = Hash::make($validatedData['password']);
            }

            if ($request->hasFile('image')) {
                $image = CommonHelper::fileUpload($request->image, EMPLOYEE_IMAGE_PATH,  $employee->image);
                // delete old image if exists
                $oldImagePath = public_path(EMPLOYEE_IMAGE_PATH . '/' . $employee->image);
                if ($employee->image && file_exists($oldImagePath)) {
                    unlink($oldImagePath);
                }


                $employee->image = $image;
            }

            $employee->save();
            $employee->syncRoles($validatedData['role']);
            DB::commit();
            $extra = [
                'redirect' => route('admin.employee.index')
            ];

            return CommonHelper::jsonResponseWeb(200, 'Employee updated successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        $employee = User::findOrFail($id);

        try {
            if ($employee->image_path && file_exists(public_path(EMPLOYEE_IMAGE_PATH . '/' . $employee->image_path))) {
                unlink(public_path(EMPLOYEE_IMAGE_PATH . '/' . $employee->image_path));
            }
            $employee->delete();
            $employee->update(['status' => 2]);
            DB::commit();
            $extra = [
                'redirect' => route('admin.employee.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Employee data deleted successfully.', [], $extra);
        } catch (Throwable $e) {
            db::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    public function createPermission()
    {
        $permissions = Permission::all()->groupBy('module');
        return view('admin.permission.create', compact('permissions'));
    }
    public function storePermission(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'module'              => 'required',
            'permission'             => 'required',

        ]);

        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            Permission::create([
                'module' => $request->module,
                'name' => $request->permission . '-' . $request->module,
            ]);
            DB::commit();
            $extra = [
                'redirect' => route('admin.employee.createPermission'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Permission added successfully.', [], $extra);
        } catch (Throwable $e) {
            db::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
