<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\CustomerWholesalePlan;
use App\Models\WholesaleIdPlan;
use App\Services\CashBackService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Throwable;
use Illuminate\Database\QueryException;
use Carbon\Carbon;
use Monolog\Handler\FingersCrossed\ActivationStrategyInterface;

use Illuminate\Support\Facades\Log;

class CustomerController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-customer')->only('index');
        $this->middleware('permission:edit-customer')->only(['edit', 'update', 'assignWholesalePlanToCustomer']);
        $this->middleware('permission:delete-customer')->only(['destroy']);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $plans = WholesaleIdPlan::orderBy('title', 'asc')->get();
        $customers = (new Customer())->getPaginatedData($request);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.customer.pagination', compact('customers', 'plans'))->render()
            ]);
        }
        Customer::checkAndExpireMyWholesalePlan();
        return view('admin.customer.index', compact('customers', 'plans'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        DB::beginTransaction();
        try {
            $wholesalePlans = WholesaleIdPlan::where('status', 1)->orderBy('title', 'asc')->get();
            $customer = Customer::with(['addresses'])->findOrFail($id);

            $customer->checkAndExpireMyWholesalePlan();

            //from relationship, i get all active expired discount list of customer : active one on the top in list 
            $assignedWholesalePlans = $customer->wholesalePlans;

            $assignplanOriginalID = optional($assignedWholesalePlans->first())->original_plan_id;

            if ($assignplanOriginalID) {
                // Step 4: Check if the assigned plan is inactive
                $assignedPlan = WholesaleIdPlan::where('id', $assignplanOriginalID)->first();

                if ($assignedPlan && $assignedPlan->status == 0) {
                    // Step 5: Append the inactive plan to the list if it's not already included
                    if (!$wholesalePlans->pluck('id')->contains($assignedPlan->id)) {
                        $wholesalePlans->push($assignedPlan);
                    }

                    // Re-sort the collection
                    $wholesalePlans = $wholesalePlans->sortBy('title')->values();
                }
            }

            $activePlan = $customer->fresh()->activePlanWholesale;
            DB::commit();

            return view('admin.customer.edit', compact('customer', 'assignedWholesalePlans', 'wholesalePlans', 'activePlan'));
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'full_name' => 'min:3|max:200|required|string',
            'email' => 'email|required|unique:customers,email,' . $id,
            'mobile_number' => 'required|regex:/^[0-9]{7,15}$/',
            'status' => 'required|in:0,1',
            'description' => 'nullable|string',
            'image' => 'nullable|mimes:jpg,jpeg,png,webp,gif|max:2048',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        if (!Customer::validateMobileNumber($request->dial_code, $request->dial_code_iso,  $request->mobile_number)) {
            return CommonHelper::jsonResponseWeb(400, __('Invalid mobile number'));
        }

        try {

            $customer = Customer::findOrFail($id);
            if (Customer::existMobileNumber($request->dial_code, $request->dial_code_iso,  $request->mobile_number, $customer->id)) {
                return CommonHelper::jsonResponseWeb(400, __('mobile exist'));
            }

            $updatedData = $request->only(['full_name', 'email', 'mobile_number', 'status', 'description', 'dial_code', 'dial_code_iso']);
            //update picture
            if ($request->hasFile('image')) {
                $updatedData['image'] = CommonHelper::fileUpload($request->file('image'), CUSTOMER_PATH);
            }
            //timezone edit check
            if ($request->dial_code_iso && $request->dial_code_iso !== $customer->dial_code_iso) {
                $updatedData['dial_code_iso'] = strtoupper($request->dial_code_iso);
                $countryInfo = CommonHelper::getCountryInfo($request->dial_code_iso);
                if ($countryInfo) {
                    $timezone = $countryInfo['timezone'];
                    $updatedData['timezone'] = $timezone;
                }
            }
            $customer->update($updatedData);
            //if request for inactive account
            if ($request->status == 0) {
                $customer->tokens()->delete();
            }
            // if ($request->wholesale_id) {
            //     $this->assignWholesalePlanToCustomer($customer, $request->wholesale_id);
            // }
            DB::commit();
            $extra = [
                'redirect' => route('admin.customer.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Customer updated succeessfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();

            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $customer = Customer::findOrFail($id);
            $customer->update(['status' => 2]);
            $customer->delete();
            DB::commit();
            $extra = [
                'redirect' => route('admin.customer.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Customer deleted succeessfully', [], $extra);
        } catch (QueryException $e) {
            $msg = $e->getMessage();
            if (strpos($msg, '1451') !== false) { // fk check
                $message = 'Please delete entries first from its relevant sections';
                return response()->json(['status' => false, 'status_code' => 500, 'message' => $message], 500);
            }
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function applyDiscount(Request $request, $id, CashBackService $cashBackService)
    {
        DB::beginTransaction();
        $customer = auth()->user();
        $customerId = $customer->id;
        $validation = Validator::make($request->all(), [
            'wholesale_id' => 'required|exists:wholesale_id_plans,id',
            'start_discount_date' => 'required|date',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        $planId = $request->wholesale_id;

        $plan = WholesaleIdPlan::find($planId);
        if (!$plan) {
            return CommonHelper::apiResponse(400, __('Cashback plan not found'));
        }
        try {

            //check existing active plan with cashback add in wallet
            $existingPlan = $customer->activePlanWholesale;
            if ($existingPlan) {
                $fromDate = Carbon::parse($existingPlan->cashback_from_date);
                $toDate = Carbon::now();

                CashBackService::processCashBack($existingPlan, $customer, $fromDate, $toDate);

                // Close out old plan
                $cashBackService->deactivateExistingPlans($id);
            }

            //check if any existing plan id created for customer
            $customPlanId = $cashBackService->getOrCreateCustomPlanIdForCustomer($id);
            //every plan have one year duration

            $planStartDate = Carbon::parse($request->start_discount_date);
            $endDate = $planStartDate->copy()->addYear();
            //set plan as scheduled if any
            $status = $planStartDate->isSameDay(now()) ? ACTIVE : SCHEDULED;
            CustomerWholesalePlan::create([
                'customer_id' => $id,
                'original_plan_id' => $plan->id,
                'custom_plan_id' => $customPlanId,
                'title' => $plan->title,
                'price_in_usd' => $plan->price_in_usd,
                'price_in_mmk' => $plan->price_in_mmk,
                'price_in_rmb' => $plan->price_in_rmb,
                'discount' => $plan->discount,
                'number_of_days' => 0,
                'min_order_amt_usd' => $plan->min_order_amt_usd,
                'min_order_amt_mmk' => $plan->min_order_amt_mmk,
                'min_order_amt_rmb' => $plan->min_order_amt_rmb,
                'start_date' => $planStartDate->toDateTimeString(),
                'end_date' => $endDate->toDateTimeString(),
                'status' => $status,
                'cashback_from_date' => $planStartDate->toDateTimeString(),
                'cashback_to_date'   => $planStartDate->copy()->endOfMonth()->endOfDay()->toDateTimeString(),
            ]);
            Customer::where('id', $id)->update(['wholesale_plan_id' => $customPlanId]);
            DB::commit();
            $extra = [
                'redirect' => route('admin.customer.edit', $id)
            ];
            return CommonHelper::jsonResponseWeb(200, 'Discount applied succeessfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function deactivateDiscount(Request $request, $id)
    {
        try {
            $discountId = $request->input('discount_id');
            $actionType = $request->input('action_type');
            if (!in_array($actionType, [DEACTIVE, CANCELLED])) {
                return CommonHelper::jsonResponseWeb(400, __("Invalid action type"));
            }
            $query = CustomerWholesalePlan::where('customer_id', $id)->where('id', $discountId);
            if ($actionType === DEACTIVE) {
                $plan = $query->where('status', ACTIVE)->first();
                $newStatus = DEACTIVE;
                $message = __("Plan deactivated successfully");
            }
            if ($actionType === CANCELLED) {
                $plan = $query->where('status', SCHEDULED)->first();
                $newStatus = CANCELLED;
                $message = __("Plan cancelled successfully");
            }
            if (!$plan) {
                return CommonHelper::jsonResponseWeb(404, __("Plan not found"));
            }

            $plan->status = $newStatus;
            $plan->save();
            $extra = [
                'redirect' => route('admin.customer.edit', $id),
            ];
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
