<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Helpers\NotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class ChatContoller extends Controller
{
    //submit conversation
    public function sendMessage(Request $request, $conversationId)
    {


        $allowedImageExtensions = CommonHelper::allowedImageExtension();

        $allowedMimes = implode(',', $allowedImageExtensions);
        $conversation = Conversation::findOrFail($conversationId);
        $validation = Validator::make(
            $request->all(),
            [
                'message' => 'nullable|string',
                'media' => 'nullable|file|mimes:' . $allowedMimes . 'max:2048',
            ],
            [
                'media.max' => __('each_file_size_vald'),
                'media.mimes' => __('file_type_vald'),
            ]
        );
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }

        $message = new Message();
        $message->conversation_id = $conversation->id;
        $message->sender_id = auth()->id();
        $message->sender_type = \App\Models\User::class;
        $message->message = $request->message;

        if ($request->hasFile('media')) {
            $imagePath =  CommonHelper::fileUpload($request->media, MESSAGE_MEDIA);
            $message->attachements = $imagePath;
        }
        $message->save();
        //apply that readable time format
        $message->formatted_date_time = CommonHelper::getReadableTime($message->created_at);
        //save notification
        NotificationHelper::saveNotification(
            auth()->id(),
            ADMIN,
            $conversation->customer_id,
            CUSTOMER,
            CHAT,
            __('new_message'),
            __('new_message_from_admin'),
            []
        );
        return response()->json([
            'success' => true,
            'message' => $message->load('sender')
        ]);
    }
    //fetch conversation
    public function fetchConversation()
    {
        //set that particular notification of message to 1
        Notification::where('type', CHAT)
            ->where('receiver_id', auth()->id())
            ->where('receiver_type', ADMIN)
            ->where('is_read', 0)
            ->update(['is_read' => 1]);
        $conversations = Conversation::with(['customer', 'lastMessage'])
            ->where('status', 'open')
            ->orderBy('created_at', 'desc')
            ->get();
        //to get readable date and time
        $conversations = $conversations->map(function ($conversation) {
            if ($conversation->lastMessage) {
                $conversation->last_message_time = CommonHelper::getReadableTime($conversation->lastMessage->created_at);
            } else {
                $conversation->last_message_time = null;
            }
            return $conversation;
        });

        return view('admin.chat.index', compact('conversations'));
    }
    //show messages
    public function showMessages($conversationId)
    {
        //set message is_read to 1
        Message::where('conversation_id', $conversationId)
            ->where('is_read', 0)
            ->update(['is_read' => 1]);

        $messages = Message::with(['sender'])
            ->where('conversation_id', $conversationId)
            ->orderBy('created_at', 'asc')
            ->get()
            ->map(function ($msg) {
                //set timezone date time
                $timezone = $msg->sender->timezone;
                $msg->attachments = $msg->attachements
                    ? asset($msg->attachements)
                    : null;
                //apply that readable time format
                $msg->formatted_date_time = CommonHelper::getReadableTime($msg->created_at->setTimezone($timezone));
                return $msg;
            });
        $loggedinUser = auth()->user()->id;
        Log::info($loggedinUser);
        return response()->json($messages);
    }
    //close the converstion
    public function closeConversation(Request $request)
    {
        $conversationId = $request->conversation_id_value;
        $conversation = Conversation::findOrFail($conversationId);
        $conversation->status = 'closed';
        $conversation->save();
        return CommonHelper::jsonResponseWeb(200, __('conversation_closed'));
    }
}
