<?php

namespace App\Helpers;

use App\Models\User;

class OrderNotificationHelper
{
    public static function sendOrderStatusNotification($order, $status, $triggeredBy = 'system')
    {
        $title = '';
        $message = '';
        $type = '';
        $receiverId = null;
        $receiverType = null;
        $senderId = 0;
        $senderType = 'system';
        switch ($status) {
            // order placed
            case 0:
                $title = __('new_order');
                $message = __('new_order_message', ['order_number' => $order->order_number]);
                $type = ORDER_PLACED;
                $senderId = $order->customer_id;
                $senderType = CUSTOMER;
                $receiverId = 1;
                $receiverType = ADMIN;
                break;
            //confirm
            case 1:
                $title = __('order_confirmed');
                $message = __('order_confirmed_message', ['order_number' => $order->order_number]);
                $type = ORDER_CONFIRMED;
                $senderId = 1;
                $senderType = ADMIN;
                $receiverId = $order->customer_id;
                $receiverType = CUSTOMER;
                break;
            //shipped
            case 2:
                $title = __('order_shipped');
                $message = __('order_shipped_message', ['order_number' => $order->order_number]);
                $type = ORDER_SHIPPED;
                $senderId = 1;
                $senderType = ADMIN;
                $receiverId = $order->customer_id;
                $receiverType = CUSTOMER;
                break;
            //delivered
            case 3:
                $title = __('order_delivered');
                $message = __('order_delivered_message', ['order_number' => $order->order_number]);
                $type = ORDER_DELIVERED;
                $senderId = 1;
                $senderType = ADMIN;
                $receiverId = $order->customer_id;
                $receiverType = CUSTOMER;
                break;
            //cancel
            case 4:
                $title = $triggeredBy === ADMIN ? __('order_cancel_by_admin') : __('order_cancel_by_customer');
                $message = $triggeredBy === ADMIN ? __('order_cancel_by_admin_message', ['order_number' => $order->order_number]) : __('order_cancel_by_customer_message', ['order_number' => $order->order_number]);
                $type = ORDER_CANCELLED;
                if ($triggeredBy == ADMIN) {
                    $senderId = 1;
                    $senderType = ADMIN;
                    $receiverId = $order->customer_id;
                    $receiverType = CUSTOMER;
                } else {
                    $senderId = $order->customer_id;
                    $senderType = CUSTOMER;
                    $receiverId = 1;
                    $receiverType = ADMIN;
                }
                break;
            //return
            case 5:
                $title = __('order_returned');
                $message = __('order_returned_message', ['order_number' => $order->order_number]);
                $type = ORDER_RETURNED;
                $senderId = $order->customer_id;
                $senderType = CUSTOMER;
                $receiverId = 1;
                $receiverType = ADMIN;
                break;
            //reject
            case 7:
                $title = __('order_rejected');
                $message = __('order_rejected_message', ['order_number' => $order->order_number]);
                $type = ORDER_REJECTED;
                $senderId = 1;
                $senderType = ADMIN;
                $receiverId = $order->customer_id;
                $receiverType = CUSTOMER;
                break;
        }
        if ($title && $message && $receiverId && $receiverType) {
            if ($triggeredBy == ADMIN) {

                NotificationHelper::saveNotification(
                    $senderId,
                    $senderType,
                    $receiverId,
                    $receiverType,
                    $type,
                    $title,
                    $message,
                    ['order_id' => $order->id],
                );
            } else {
                $notifyRoles = CommonHelper::$generalNotifyRoles;
                $admins = User::role($notifyRoles)->get();
                foreach ($admins as $admin) {
                    NotificationHelper::saveNotification(
                        $senderId,
                        $senderType,
                        $admin->id,
                        $receiverType,
                        $type,
                        $title,
                        $message,
                        ['order_id' => $order->id, 'url' => route('admin.orders.index')],
                    );
                }
                //fire event
                event(new \App\Events\AdminNotificationEvent([
                    'message' => $message,
                    'title'   => $title,
                    'order_id' => $order->id,
                    'url' => route('admin.orders.index')
                ]));
            }
        }
    }
    //notification for refund for : reject / return / cancel
    public static function sendRefundNotification($order, $refundSource)
    {
        $title = __('order_refund');
        $message = __('order_refund_message', ['order_number' => $order->order_number]);
        $type = ORDER_REFUND;
        NotificationHelper::saveNotification(
            senderId: 1,
            senderType: ADMIN,
            receiverId: $order->customer_id,
            receiverType: CUSTOMER,
            type: $type,
            title: $title,
            message: $message,
            data: ['order_id' => $order->id]
        );
    }
    //notification for return and cancel accept reject
    public static function sendCancelReturnStatusesNotification($order, $status, $sourceType)
    {
        $title = '';
        $message = '';
        $type = null;
        switch ($sourceType) {
            case 'cancel':
                if ($status == 1) {
                    $title = __('cancel_approved');
                    $message = __('cancel_approved_message', ['order_number' => $order->order_number]);
                    $type = ORDER_CANCEL_APPROVED;
                } elseif ($status == 2) {
                    $title =  __('cancel_rejected');
                    $message = __('cancel_rejected_message', ['order_number' => $order->order_number]);
                    $type = ORDER_CANCEL_REJECTED;
                }
                break;
            case 'return':
                if ($status == 1) {
                    $title = __('return_approved');
                    $message = __('return_approved_message', ['order_number' => $order->order_number]);
                    $type = ORDER_RETURN_APPROVED;
                } elseif ($status == 2) {
                    $title =  __('return_rejected');
                    $message = __('return_rejected_message', ['order_number' => $order->order_number]);
                    $type = ORDER_RETURN_REJECTED;
                }
                break;
        }
        NotificationHelper::saveNotification(
            1,
            ADMIN,
            $order->customer_id,
            CUSTOMER,
            $type,
            $title,
            $message,
            ['order_id' => $order->id]
        );
    }
}
